/********************************************************************
 * (C) Copyright 2016 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/* This plugin is an example for using pack reference functionality. */

#include <AlLiveData.h>
#include <AlUniverse.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlProduct.h>
#include <string>
#include <vector>

#ifdef _WIN32
#include <windows.h>
#endif

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

namespace
{
    std::string
    getTempDirectory()
    {
        std::string dirName;
#ifdef _WIN32
        char tmp[ MAX_PATH ];
        memset( tmp, 0, MAX_PATH );
        UINT retVal = GetTempPathA( MAX_PATH, tmp );

        if( retVal == 0 )
            dirName.assign( "C:\\" );  // out of desperation, use this.
        else
            dirName.assign( tmp ); 

#else
        const char* tmpdir = getenv("TMPDIR"); 

        if( !tmpdir ) // out of desperation, use this.
            tmpdir = "/tmp";
        dirName.assign( tmpdir );
#endif /* AW_OS_WIN */
        return dirName;
    }
    void pack()
    {
        std::string tmpDir = getTempDirectory();
        tmpDir += "refpack";
        statusCode status = AlUniverse::packReference( tmpDir.c_str() );
        if( status != sSuccess )
            AlPrintf( kPrompt, "Failed to pack reference data" );

        std::vector<std::string> cmdLine;
        cmdLine.push_back( "AlToRef" );
        cmdLine.push_back( "-i" );
        cmdLine.push_back( "wire" );
        cmdLine.push_back( "-o" );
        cmdLine.push_back( "ref" );
        cmdLine.push_back( "-productKey" );
        cmdLine.push_back( AlProduct::productKey() );
        cmdLine.push_back( "-productVersion" );
        cmdLine.push_back( AlProduct::productVersion() );
        cmdLine.push_back( "-productLicenseType" );
        cmdLine.push_back( AlProduct::productLicenseType() );
        cmdLine.push_back( "-productLicensePath" );
        cmdLine.push_back( AlProduct::productLicensePath() );
    }
}

// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be nothing_func.S, nothing_func.M
	hFunc.create( "pl_RefPack", pack );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	h.create( "Pack Reference", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "pack_reference" );

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "al_file" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Pack Reference plug-in installed under menu 'File'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}



