/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//  intersection.plugin
//
//  Example use of AlIntersect.h
//

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlObject.h>
#include <AlList.h>
#include <AlLinkItem.h>
#include <AlPickList.h>
#include <AlIntersect.h>
#include <AlSurfaceNode.h>
#include <AlCurveNode.h>
#include <AlSurface.h>

extern char * makeAltPath(const char *dirName, const char *suffix);

void do_intersect_srf_srf(AlSurface* srf1, AlSurface* srf2)
  {
	AlIntersectSurfSurfInfo data;

	if (sSuccess==AlIntersect::intersect(srf1,srf2,data))
	  {
		AlPrintf(kPrompt,"Intersection completed.");
		AlPrintf(kPrompt,"# isolated points: %d",data.numberIsolatedPoints);
		AlPrintf(kPrompt,"# intersection curves: %d",data.numberIntersectionCurves);
		AlPrintf(kPrompt,"# boundary curves: %d",data.numberBoundaryCurves);

		// Let's actually make the curves

		int count;
		for (count=0; count<data.numberIntersectionCurves; count++)
		  {
			AlCurveNode* node=new AlCurveNode;
			node->create(*(data.intersectionCurves+count));
		  }
		for (count=0; count<data.numberBoundaryCurves; count++)
		  {
			AlCurveNode* node=new AlCurveNode;
			node->create(*(data.boundaryCurves+count));
		  }
			
	  }
	else
	  {
		AlPrintf(kPrompt,"An error occurred in calculation.....");
	  }
  }


void do_intersect_crv_srf(AlCurve* crv1, AlSurface* srf2)
  {
    AlIntersectCurveSurfInfo* data=NULL;
	AlList* list=new AlList(data);

    if (sSuccess==AlIntersect::intersect(crv1,srf2,list))
      {
        AlPrintf(kPrompt,"Intersection completed.");
		data=(AlIntersectCurveSurfInfo*)(list->first());
		while (data)
		  {
			AlPrintf(kPrompt,"t=%2.2f P1=%2.2f,%2.2f,%2.2f "
							 "u=%2.2f v=%2.2f P2=%2.2f,%2.2f,%2.2f",
							 data->t, data->pointOnCurve[0],
							 data->pointOnCurve[1],data->pointOnCurve[2],
							 data->u, data->v, data->pointOnSurface[0],
                             data->pointOnSurface[1],data->pointOnSurface[2]);
			data=data->nextItem();
		  }
      }
    else
      {
        AlPrintf(kPrompt,"An error occurred in calculation.....");
      }
	delete list;
  }

void do_intersect_crv_crv(AlCurve* crv1, AlCurve* crv2)
  {
    AlIntersectCurveCurveInfo* data=NULL;
	AlList* list=new AlList(data);

    if (sSuccess==AlIntersect::intersect(crv1,crv2,list))
      {
        AlPrintf(kPrompt,"Intersection completed.");
        data=(AlIntersectCurveCurveInfo*)(list->first());
        while (data)
          {
            AlPrintf(kPrompt,"t1=%2.2f t2=%2.2f P=%2.2f,%2.2f,%2.2f ",
                             data->tCurve1, data->tCurve2,data->point[0],
                             data->point[1],data->point[2]);
            data=data->nextItem();
          }
      }
    else
      {
        AlPrintf(kPrompt,"An error occurred in calculation.....");
      }
	delete list;
  }




void do_intersect( void )
{
	if (sSuccess!=AlPickList::firstPickItem())
	  {
		AlPrintf(kPrompt,"No Objects selected to intersect");
		return;
	  }
	AlObject* obj1=AlPickList::getObject();
	
	if (sSuccess!=AlPickList::nextPickItem())
	  {
		AlPrintf(kPrompt,"Can't intersect object with itself");
		return;
	  }
	AlObject* obj2=AlPickList::getObject();

	switch (obj1->type())
	  {
		case kSurfaceNodeType:
			switch (obj2->type())
			  {
				case kSurfaceNodeType:
					do_intersect_srf_srf(obj1->asSurfaceNodePtr()->surface(),
										 obj2->asSurfaceNodePtr()->surface());
					break;
				case kCurveNodeType:
					do_intersect_crv_srf(obj2->asCurveNodePtr()->curve(),
										 obj1->asSurfaceNodePtr()->surface());
					break;
				default:
					AlPrintf(kPrompt,"Invalid selection");
			  }
			break;
		case kCurveNodeType:
			switch (obj2->type())
			  {
				case kSurfaceNodeType:
					do_intersect_crv_srf(obj1->asCurveNodePtr()->curve(),
										 obj2->asSurfaceNodePtr()->surface());
					break;
				case kCurveNodeType:
					do_intersect_crv_crv(obj1->asCurveNodePtr()->curve(),
										 obj2->asCurveNodePtr()->curve());
					break;
				default:
					AlPrintf(kPrompt,"Invalid selection");
			  }
			break;
		default:
			AlPrintf(kPrompt,"Invalid selection");
	  }

	AlUniverse::redrawScreen();
}

static AlMomentaryFunction hFunc;
static AlFunctionHandle h;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize( kZUp );

	hFunc.create( do_intersect );
	h.create("Intersect Objects",&hFunc);
	h.setAttributeString("intersection");
	h.addToMenu("mp_srftools");
	h.setIconPath( makeAltPath( dirName, NULL ) );
	AlPrintf(kPrompt,"Intersection Plugin installed on Palette 'Surface Edit'");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	h.deleteObject();
    hFunc.deleteObject();
	return 0;
}
