/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*

hiliteNSidePolygon.plugin ( formerly jptNside.plugin )

	This plugin displays polygons with more than n sides.

	Polygons with more than n sides are copied into a
	new polyset and one edge is bent into the center
	so it is easily identified.

	The new polyset is set to be a template so it is
	displayed differently from the original polyset.

Options:

	Sides: any polygon with greater than Sides vertices
		are displayed.

Version History:

    Icon now stays on the shelf after quit.

	5/20:
		Updated pick so new polyset is now picked.

*/

#include <AlUniverse.h>

#include <AlPolyset.h>
#include <AlPolysetNode.h>
#include <AlPolysetVertex.h>
#include <AlPolygon.h>
#include <AlGroupNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlPickList.h>

#include <stdio.h>
#include <string.h>

/*
*
*/
statusCode jPrintError(statusCode code)
{
	switch (code)
	{
		case sSuccess:
	//		fprintf(stderr, "Successful.\n");
			break;
		case sFailure:
			fprintf(stderr, "*** Error: Failure.\n");
			break;
		case sInsufficientMemory:
			fprintf(stderr, "*** Error: Insufficient memory.\n");
			break;
		case sAlreadyCreated:
			fprintf(stderr, "*** Error: Already created.\n");
			break;
		default:
			fprintf(stderr, "%s", "*** Error: Unknown error code %d\n", code);
			break;
	}
	return code;
}

/*
*	Finds the center point of a polygon and return it as (px, py, pz)
*/
void jGetPivot(AlPolygon *polygon, double &px, double &py, double &pz)
{
    int vcount = polygon->numberOfVertices();

    double tx, ty, tz;
    tx = ty = tz = 0;

    int i;
    for (i = 0; i < vcount; i++)
    {
        double x, y, z;
        AlPolysetVertex *vertex;

        vertex = polygon->vertex(i);
        vertex->worldPosition(x, y, z);
        tx += x; ty += y; tz += z;
		delete vertex;
    }

    px = tx / vcount;
    py = ty / vcount;
    pz = tz / vcount;

//  fprintf(stderr, "Polygon pivot = %2.2lf, %2.2lf, %2.2lf\n", px, py, pz);
}


/*
*	Copy polygons with more than n sides into a new polyset 
*	Set the new polyset display to be template
*	Create the n sided polygon as bent
*/
void jCopyPent(AlPolyset *polyset, int side )
{
	int pcount, i;

	if (polyset == NULL || 
		(pcount = polyset->numberOfPolygons()) == 0)	// bogus polyset
	{
	//	fprintf(stderr, "*** Warning: Empty polyset!!!\n");
		AlPrintf( kPrompt, "*** Warning: Empty polyset!!!\n");
		return;
	}

//	fprintf(stderr, "Polygons: %d, Vertices: %d\n", pcount, vcount);

	AlPolyset *newPolyset = new AlPolyset;
	if ( sSuccess != jPrintError(newPolyset->create()) )
	{
		fprintf(stderr, "*** Warning: Can't create polyset\n");
		return;
	}

    AlPolysetNode *newNode = newPolyset->polysetNode();
    AlPolysetNode *oldNode = polyset->polysetNode();
    newNode->setName( oldNode->name() );

    double x, y, z;
    oldNode->scalePivot(x, y, z);
    newNode->setScalePivot(x, y, z);
    oldNode->rotatePivot(x, y, z);
    newNode->setRotatePivot(x, y, z);
    newNode->setDisplayMode( kDisplayModeTemplate, TRUE);

//	oldNode->unpick();
//	newNode->pick();

    delete oldNode;
    delete newNode;


	AlPolygon *polygon = NULL;
	AlPolysetVertex *vertex = NULL;

	// go through all polygons and copy into new polyset
	for (i = 0; i < pcount; i++)
	{
		polygon = polyset->polygon(i);
 		int polyVcount = polygon->numberOfVertices();

		int j;

		if ( polyVcount > side )		// has > side vertices in this polygon
		{
			int newP, newV;

 			if ( (newP = newPolyset->newPolygon()) == -1)
			{
				fprintf( stderr, "*** Error: can't create polygon!!!\n");
				return;
			}

			AlPolygon *np = newPolyset->polygon( newP );

			// go through all vertices in this poly and insert into new polyset

			for (j = 0; j < polyVcount; j++)
			{
				vertex = polygon->vertex(j);
				vertex->worldPosition(x, y, z);
				newV = newPolyset->newVertex(x, y, z);
				np->addVertex(newV);

				delete vertex;

			}	// for j; vcount

			double x, y, z;
			jGetPivot(polygon, x, y, z);
			newV = newPolyset->newVertex(x, y, z);

			np->addVertex(newV);

			delete np;
		}	// if polyVcount

		delete polygon;
	}	// for i; pcount

	if ( !newPolyset->numberOfPolygons() )
	{
		AlPrintf( kPrompt, "no n-sided polygons found\n");
		newPolyset->deleteObject();
		delete newPolyset;
	}
	else
	{
		AlPrintf( kPrompt, "%d n-sided polygons found\n", newPolyset->numberOfPolygons() );
		AlUniverse::redrawScreen( kRedrawTemplate );
	}
	// delete newPolyset;
}


/*
*	The function that gets called by Alias
*	Go through the picklist and operate on polysets
*/
void do_pent()
{
//	AlPrintf( kStdout, " do_pent()\n");

    char done = 0;
    int sides;
    if (sSuccess != AlGetInteger("pent.sides", sides))
        sides = 4;

    AlObject *picked = NULL;
    AlPolysetNode *polynode = NULL;
	statusCode stat;
    for(    stat = AlPickList::firstPickItem();
            stat == sSuccess;
            stat = AlPickList::nextPickItem() )
    {
        picked = AlPickList::getObject();
        polynode = picked->asPolysetNodePtr();
        if ( polynode )
        {
            const char *name = polynode->name();
            AlPrintf(kPrompt, "Checking %s...\n", name );
			AlPolyset *pset = polynode->polyset();
            jCopyPent( pset, sides );
			delete pset;
            done = 1;
			AlUniverse::redrawScreen( kRedrawTemplate );
        }
    }
	delete picked;
    if (!done)
	{
        AlPrintf( kPrompt, "Pent: Please select a polyset object.\n");
		fprintf( stderr, "\n" );
	}
	else
        AlUniverse::redrawScreen( kRedrawTemplate );
}


static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
    AlUniverse::initialize( );

	char *dirScm = makeAltPath( dirName, "scm" );
    hFunc.create( "pl_jptNSide", do_pent );
    h.create( "HiLite NSided Polygon", &hFunc );
    h.setAttributeString( "nsided_polys" );
    h.setOptionBox( "hiliteNSidedPolygon.scm", "pent.options", dirScm);
	h.setIconPath( makeAltPath( dirName, NULL ) );
    h.addToMenu( "al_polyedit" );
    AlPrintf( kPrompt, "HiLite N Sided Polygon plug-in installed under 'Polygon Edit' palette." );
    return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    h.deleteObject();
    hFunc.deleteObject();

    // do nothing
    return 0;
}
