/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
jptExtrudePolys.plugin

    This plugin allows you to extrude multiple polygons in a polyset.
	It works similar to 'Extrude Together' in 3Design.

	Installed under Polygon Edit palette.

Usage:

	1: Select the polygons you want to extrude.

	2: Select 'Extrude Polygons' plugin from Polygon Edit palette.

	3: Drag with the left button to extrude the selected polys.

	4: Drag with the middle button to rotate the selected polys.

	5: Drag with the right button to scale the selected polys.

	6: Control-right button resets the current transform values 
	to [0, 1, 0].  Ctrl-RB again to undo the reset.

	7: Shift-left button repeats the extrution on the selected polys.

Limitations:

	Extruding edges creates double sided polygons.  Clean it using 
	PolyEdit->Clean Polyset Single Side.

	Rotation not implimented yet.

	Transformation get screwed up if polysets are rotated.

Version History:

	7/8/96:
		Extended to interactive extrude.

	5/15/96:
		Finished the first version.

*/

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlDagNode.h>
#include <AlPolyset.h>
#include <AlPolysetNode.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlTM.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#ifdef _WIN32
#define M_PI 3.1415926535
#endif

//----------------------------------------------------------

#define MAX_OBJS 32
#define MAX_CLUSTERS 64
#define MAX_VERTS 64

enum { kLB = 8, kMB = 16, kRB = 32 };

//-----------------------------------

//	Classes

//-----------------------------------

typedef struct {
	double  x;
	double  y;
	double  z;
} VertexCoord;

//-----------------------------------

class PolyData {
public:
	int oldV1[MAX_VERTS], oldV2[MAX_VERTS];
	int newV1[MAX_VERTS], newV2[MAX_VERTS];
	int count;
	PolyData()
	{
		count = 0;
		int i;
		for ( i = 0; i < MAX_VERTS; i++ )
		{
			oldV1[i] = oldV2[i] = -1;
		}
	}

	void InsertData( int o1, int o2, int n1, int n2 )
	{
		if ( !(count < MAX_VERTS) )
			return;
		oldV1[count] = o1;
		oldV2[count] = o2;
		newV1[count] = n1;
		newV2[count] = n2;
		count++;
	}
};

//-----------------------------------

class Cluster {
public:
	int cCount;	// # of clusters
	int cSize[ MAX_CLUSTERS ];		// list of size
	int *cPolys[ MAX_CLUSTERS ];	// list of list of polys
//	VertexCoord cPnorms[ MAX_CLUSTERS ];	// list of poly normals
	int maxPolys;

	int cVsize[ MAX_CLUSTERS ];		// list of vtx size
	int *cVerts[ MAX_CLUSTERS ];		// list of list of verts
	VertexCoord* cVxyz[ MAX_CLUSTERS ];	// list of vertex xyz

	VertexCoord *cCenter;			// list of cluster center point
	VertexCoord *cNorms;			// list of cluster normals

	double cNx[ MAX_CLUSTERS ];
	double cNy[ MAX_CLUSTERS ];
	double cNz[ MAX_CLUSTERS ];

	AlPolyset *pset;

	/////////////////////////

	Cluster( int size, AlPolyset *polyset );
	~Cluster();

	void InsertPolygon( AlPolygon *pgon );	// make polygon clusters
	void Consolidate( );	// consolidate polygon clusters
	void CalcNormals( );	// calculate cluster normals
	void InsertVertex( );	// make vertex clusters
	void InitVertCoords();	// init vertex coordinates
	void XformVerts( double t, double r, double s );	// xform vertex clusters
};

//----------------------------------------------------------

//	Globals

//----------------------------------------------------------

static int gPicked = 0;

static AlPolyset *gPsets[MAX_OBJS];
static Cluster *gClusters[MAX_OBJS];

static AlFunctionHandle h;
static AlContinuousFunction hFunc;

static Cluster* jCreateClusters( AlPolyset *pset );
static void jProcessPicked( );
static void jDuplicatePolys( AlPolyset *pset );

static double gX, gY;			// original cursor pos
static double gTrans, gRot, gScale;	// xforms			
static double gTransRedo, gRotRedo, gScaleRedo;	// undo xforms			

//----------------------------------------------------------

Cluster::Cluster( int size, AlPolyset *polyset )
{
	pset = polyset;
	cCount = 0;
	int i;
	for ( i = 0; i < MAX_CLUSTERS; i++ )
	{
		cSize[i] = 0;
		cVsize[i] = 0;
		cPolys[i] = NULL;
		cVerts[i] = NULL;
		cVxyz[i] = NULL;
	}
	cCenter = NULL;
	cNorms = NULL;
	maxPolys = size;
}

Cluster::~Cluster()
{
	int i;
	for ( i = 0; i < cCount; i++ )
	{
		if ( cPolys[i] )
		{
		//	fprintf( stderr, "deleting %x\n", cPolys[i] );
			delete cPolys[i];
		}
		if ( cVerts[i] )
		{
			delete cVerts[i];
			delete cVxyz[i];
		}
	}
	if ( cCenter )
		delete cCenter;
	if ( cNorms )
		delete cNorms;
}

void Cluster::InsertPolygon( AlPolygon *pgon )
{
	if ( !pgon )
		return;
//	AlPolyset *pset = pgon->polyset();
	if ( !cCount )	// first poly.  Create new cluster
	{
		if (! (cCount < MAX_CLUSTERS) )
			return;
		cPolys[ cCount ] = new int[ maxPolys ];
		cPolys[ cCount ][ 0 ] = pgon->index();
		cSize[ cCount ]++;
		cCount++;
		return;
	}
	// check to see if new polygon touches any clusters
	AlPolygon poly;
	AlPolysetVertex newV, oldV;

	int c, p, v, i;
	for ( c = 0; c < cCount; c++ )	// go through all clusters
	{
		for ( p = 0; p < cSize[c]; p++ )	// go through all polys
		{
			pset->polygonD( cPolys[c][p], poly );
		//	fprintf( stderr, "Cluster %d, Polygon %d: %d\n", c, p, cPolys[c][p] );
			for ( v = 0; v < pgon->numberOfVertices(); v++ )	// go through all verts in picked poly
			{
				pgon->vertexD( v, newV );
				for ( i = 0; i < poly.numberOfVertices(); i++ )	// check all verts in this polygon
				{
					poly.vertexD( i, oldV );
					if ( AlAreEqual( &newV, &oldV ) )	// shared vertex.  is adjacent.  add to this cluster
					{
						cPolys[c][ cSize[c] ] = pgon->index();
					//	fprintf( stderr, "Polygon %d adjacent to polygon %d.  Inserted into cluster %d at %d\n", 
						//	pgon->index(), poly.index(), c, cSize[c] );
						cSize[c]++;
						return;
					}
				}	// for i
			}	// for v
		}	// for p
	}	// for c

	// pgon not connected with any clusters.  add new cluster

	if (! (cCount < MAX_CLUSTERS) )
		return;
	cPolys[ cCount ] = new int[ maxPolys ];
	cPolys[ cCount ][ 0 ] = pgon->index();
	cSize[ cCount ]++;
	cCount++;
}

void Cluster::CalcNormals( )
{
	int c, p;
	AlPolygon pgon;
	for ( c = 0; c < cCount; c++ )
	{
		int totalVerts = 0;
		cNx[c] = 0.0;
		cNy[c] = 0.0;
		cNz[c] = 0.0;
		for ( p = 0; p < cSize[c]; p++ )
		{
		//	fprintf( stderr, "Cluster %d, Polygon %d, Index %d\n",
		//		c, p, cPolys[c][p] );
			pset->polygonD( cPolys[c][p], pgon );
			double x, y, z;
			pgon.normal( x, y, z );
		//	jPolyNormal( &pgon, x, y, z );

			cNx[c] += x;
			cNy[c] += y;
			cNz[c] += z;
			totalVerts += pgon.numberOfVertices();
		}
		cNx[c] /= cCount;
		cNy[c] /= cCount;
		cNz[c] /= cCount;

		// normalize the vector

	//	fprintf( stderr, "Normal for Cluster %d: (%2.2lf, %2.2lf, %2.2lf)\n", c, cNx[c], cNy[c], cNz[c] ); 
		if ( (cNx[c] == cNy[c]) && (cNx[c] == cNz[c]) && (cNx[c] == 0.0 ) )
			cNy[c] = 1.0;
		if ( fabs(cNx[c]) >= fabs(cNy[c]) && fabs(cNx[c]) >= fabs(cNz[c]) )
		{
			cNy[c] /= fabs(cNx[c]);
			cNz[c] /= fabs(cNx[c]);
			if ( cNx[c] > 0 )
				cNx[c] = 1.0;
			else
				cNx[c] = -1.0;
		}
		else if ( fabs(cNy[c]) >= fabs(cNx[c]) && fabs(cNy[c]) >= fabs(cNz[c]) )
		{
			cNx[c] /= fabs(cNy[c]);
			cNz[c] /= fabs(cNy[c]);
			if ( cNy[c] > 0 )
				cNy[c] = 1.0;
			else
				cNy[c] = -1.0;
		}
		else
		{
			cNx[c] /= fabs(cNz[c]);
			cNy[c] /= fabs(cNz[c]);
			if ( cNz[c] > 0 )
				cNz[c] = 1.0;
			else
				cNz[c] = -1.0;
		}
	//	fprintf( stderr, "Normal for Cluster %d: (%2.2lf, %2.2lf, %2.2lf)\n", c, cNx[c], cNy[c], cNz[c] ); 

		cVerts[c] = new int[totalVerts];
	}
}

void Cluster::InsertVertex( )
{
	int c, p, v;
	AlPolygon pgon;
	AlPolysetVertex vtx;

	int marker = 0;
	char st[255];

	// insert picked verts into clusters
	for ( c = 0; c < cCount; c++ )
	{
		marker++;
		sprintf( st, "Processing picked... [" );
		int i;
		for ( i = 0; i < marker; i++ )
		{
			sprintf( st, "%s%c", st, '#' );
		}
		for ( i = marker; i < cCount; i++)
			sprintf( st, "%s-", st );
		sprintf( st, "%s]", st );
		AlPrintf( kPrompt, "%s", st );

		for ( p = 0; p < cSize[c]; p++ )
		{
			pset->polygonD( cPolys[c][p], pgon );

			for ( v = 0; v < pgon.numberOfVertices(); v++ )
			{
				pgon.vertexD( v, vtx );
				if ( vtx.isPicked() )	// not inserted into cluster yet.
				{
					cVerts[ c ][ cVsize[c]++ ] = vtx.index();
					vtx.unpick();
				//	fprintf( stderr, "Vertex %d inserted into cluster %d\n", vtx.index(), c );
				}
			}
		}
		cVxyz[c] = new VertexCoord[ cVsize[c] ];
	}
	cCenter = new VertexCoord[ cCount ];
	InitVertCoords();
}

void Cluster::InitVertCoords()
{
	// insert vertex world positions into clusters
	// also get the center point of the cluster for scaling

	int c, v;
	AlPolysetVertex vtx;
	for ( c = 0; c < cCount; c++ )
	{
		double ox, oy, oz;
		ox = oy = oz = 0;

		for ( v = 0; v < cVsize[c]; v++ )
		{
			pset->vertexD( cVerts[c][v], vtx );
			vtx.pick();
			vtx.worldPosition( cVxyz[c][v].x, cVxyz[c][v].y, cVxyz[c][v].z );
			ox += cVxyz[c][v].x;
			oy += cVxyz[c][v].y;
			oz += cVxyz[c][v].z;
		}
		if ( cVsize[c] )
		{
			ox /= cVsize[c];
			oy /= cVsize[c];
			oz /= cVsize[c];

			cCenter[c].x = ox;
			cCenter[c].y = oy;
			cCenter[c].z = oz;
		}

	}
}

/*
*
*/
static void jRotateAboutAxis( AlTM &tm,
    double ox, double oy, double oz,
    double x, double y, double z, double angle )
{
//  fprintf( stderr, "Pivot = (%lf, %lf, %lf)\n", ox, oy, oz);
//  fprintf( stderr, "Axis = (%lf, %lf, %lf)\n", x, y, z);

    AlTM Rx(1,1,1,1), Ry(1,1,1,1), Rz(1,1,1,1), T(1, 1, 1, 1), iT(1, 1, 1, 1);

    T[3][0] = -ox;
    T[3][1] = -oy;
    T[3][2] = -oz;
    iT[3][0] = ox;
    iT[3][1] = oy;
    iT[3][2] = oz;

    double a, b, c, d, V;

    V = sqrt ( x*x + y*y + z*z );

    a = x / V;
    b = y / V;
    c = z / V;
    d = sqrt( b*b + c*c );

// fprintf(stderr, "a = %lf, b = %lf, c = %lf, d = %lf, V = %lf\n", a, b, c, d, V);

    Rx[1][1] = (d) ? c/d : 1.0;
    Rx[2][1] = (d) ? -b/d : 0.0;
    Rx[1][2] = (d) ? b/d : 0.0;
    Rx[2][2] = (d) ? c/d : 1.0;

    Ry[0][0] = d;
    Ry[0][2] = a;
    Ry[2][0] = -a;
    Ry[2][2] = d;

    Rz[0][0] = cos(angle);
    Rz[1][1] = cos(angle);
    Rz[0][1] = sin(angle);
    Rz[1][0] = -sin(angle);

    tm = T *  Rx * Ry * Rz * Ry.transpose() * Rx.transpose() * iT;
}


void Cluster::XformVerts( double t, double r, double s )
{
	if ( !pset )
		return;
	AlPolysetNode *pnode = pset->polysetNode();
	double gtm[4][4];
	pnode->globalTransformationMatrix( gtm );

	AlTM gTM( gtm );

	int c, v;
	double x, y, z;
	AlPolysetVertex vtx;
	for ( c = 0; c < cCount; c++ )	// go through all clusters
	{
	//	Find post-xformed normals

		double nx, ny, nz, ox, oy, oz;
	//	fprintf( stderr, "before: (%2.2lf, %2.2lf, %2.2lf)\n", cNx[c], cNy[c], cNz[c] );
		AlTM gtm;
		AlPolysetNode *pnode = pset->polysetNode();

		ox = cCenter[c].x;
		oy = cCenter[c].y;
		oz = cCenter[c].z;
		nx = cNx[c] + cCenter[c].x;
		ny = cNy[c] + cCenter[c].y;
		nz = cNz[c] + cCenter[c].z;

		gtm = gtm.inverse();
		pnode->globalTransformationMatrix( gtm );
		gtm.transPoint( nx, ny, nz );
		gtm.transPoint( ox, oy, oz );

		nx -= ox;
		ny -= oy;
		nz -= oz;

	//	fprintf( stderr, "after: (%2.2lf, %2.2lf, %2.2lf)\n", nx, ny, nz );
		delete pnode;

		for ( v = 0; v < cVsize[c]; v++ )	// go through all verts in cluster
		{
			x = cVxyz[c][v].x;
			y = cVxyz[c][v].y;
			z = cVxyz[c][v].z;

		// scale it

			x -= cCenter[c].x;
			y -= cCenter[c].y;
			z -= cCenter[c].z;

			x *= s;
			y *= s;
			z *= s;

			if ( r != 0.0 )
			{
			// rotate it

			//	fprintf( stderr, "rotate pointes...\n" );
				AlTM rotTM;

				jRotateAboutAxis( rotTM,
					0, 0, 0,
					nx, ny, nz, r );

			//	rotTM = AlTM::rotate( nx, ny, nz, r );

				rotTM.transPoint( x, y, z );
			}

			x += cCenter[c].x;
			y += cCenter[c].y;
			z += cCenter[c].z;

		// translate it

			x += nx * t;
			y += ny * t;
			z += nz * t;

			pset->vertexD( cVerts[c][v], vtx );
			vtx.setWorldPosition( x, y, z, gTM );
		}
	}
	pnode->updateDrawInfo();
	delete pnode;
}

void Cluster::Consolidate( )
{
	if ( !cCount )
		return;

	AlPolygon newP, oldP;
	AlPolysetVertex newV, oldV;

	int marker = 0;
	int total = cCount;
	char st[255];

	int c, p, v, done = 0;
	while ( !done )
	{
		for ( c = cCount-1; c > 0; c-- )	// check from last cluster
		{
			marker++;
			sprintf( st, "Processing picked... [" );
			int i;
			for ( i = 0; i < marker; i++ )
			{
				sprintf( st, "%s%c", st, 'X' );
			}
			for ( i = marker; i < total; i++)
				sprintf( st, "%s-", st );
			sprintf( st, "%s]", st );
			AlPrintf( kPrompt, "%s", st );

			for ( p = 0; p < cSize[c]; p++ )	// check polys in cluster
			{
				pset->polygonD( cPolys[c][p], newP );
				for ( v = 0; v < newP.numberOfVertices(); v++ )	// check verts in this poly
				{
					newP.vertexD( v, newV );
					int C, P, V;

					for ( C = 0; C < c; C++ )	// check from first cluster
						for ( P = 0; P < cSize[C]; P++ )	// check against all polys
						{
							pset->polygonD( cPolys[C][P], oldP );
							for ( V = 0; V < oldP.numberOfVertices(); V++ )
							{
								oldP.vertexD( V, oldV);
							//	fprintf( stderr, "Checking Cluster %d->%d, Polygon %d->%d...\n",
								//	c, C, p, P );
								if ( AlAreEqual( &oldV, &newV ) )	// vertex shared.  merge clusters
								{
								//	fprintf( stderr, "Merging Cluster %d into %d...\n", c, C );
									int i;
									for ( i = 0; i < cSize[c]; i++ )	// copy cluster c into cluster C 
									{
										cPolys[C][cSize[C]++] = cPolys[c][i];
									//	fprintf( stderr, "Moving polygon %d into cluster %d...\n", cPolys[c][i], C );
									}
								//	fprintf( stderr, "deleting Cluster %d: %x\n", c, cPolys[c] );
									delete cPolys[c];
									cPolys[c] = NULL;
									if ( c < (cCount - 1 ) )
									{
										int move;
										for ( move = c; move < (cCount-1); move++ )
										{
										//	fprintf( stderr, "   Moving cluster %d to %d\n", move+1, move );
											cPolys[move] = cPolys[move+1];
										//	fprintf( stderr, "NULLING Cluster %d: %x\n", move+1, cPolys[move+1] );
											cPolys[move+1] = NULL;
											cSize[move] = cSize[move+1];
											cSize[move+1] = 0;
										//	fprintf( stderr, "Cluster %d = %d, Cluster %d = %d\n", move, cSize[move], move+1, cSize[move+1] );
										}
									}

									V = oldP.numberOfVertices();
									P = cSize[C];
									C = c;
									v = newP.numberOfVertices();
									p = cSize[c];
									cCount--;
								}
							}
						}
				}
			}
		}
		done = 1;
	}	// while
}


//----------------------------------------------------------

static void jXform()
{
	int i;
	for ( i = 0; i < gPicked; i++ )
	{
		gClusters[i]->XformVerts( gTrans, gRot, gScale );
	}
//	fprintf( stderr, "(%2.2lf, %2.2lf, %2.2lf)\n",
//		gTrans, gRot, gScale );
	AlUniverse::redrawScreen();
}

static void init_func()
{
//	fprintf( stderr, "init...\n" );

	gPicked = 0;
	gTrans = gRot = gTransRedo = gRotRedo = 0.0;
	gScale = gScaleRedo = 1.0;

	AlPrintf( kPrompt, "Extrude selected polygons: LB to extrude, MB to rotate, RB to Scale." );
}

static void cleanup_func()
{
//	fprintf( stderr, "clean up...  deleting %d psets...\n", gPicked );
	int i;
	for ( i = 0; i < gPicked; i++ )
	{
		delete gClusters[i];
		delete gPsets[i];
	}

	gPicked = 0;
}
/*
static void printit( AlPolyset *pset )
{
	int np, nv;
	np = pset->numberOfPolygons();
	nv = pset->numberOfVertices();
	fprintf( stderr, "np = %d, nv = %d\n", np, nv );

	
	AlPolygon pgon;
	AlPolysetVertex vtx;
	int i, j;
	for ( i = 0; i < np; i++ )
	{
		pset->polygonD( i, pgon );
		fprintf( stderr, "  Polygon %d:\n", pgon.index() );
		for ( j = 0; j < pgon.numberOfVertices(); j++ )
		{
			pgon.vertexD( j, vtx );
			fprintf( stderr, "    vtx %d\n", vtx.index() );
		}
	}
}
*/

static void down_func( int input, Screencoord x, Screencoord y )
{
	input = x;
//	fprintf( stderr, "down...\n" );

	if ( !gPicked )	// nothing picked yet.
	{
		jProcessPicked( );

		if ( gPicked )	// make picked polygons into clusters
		{
			int i;
			for ( i = 0; i < gPicked; i++ )
			{
				gClusters[i] = jCreateClusters( gPsets[i] );
			//	printit( gPsets[i] );
			}

		}
	}
	else	// process mouse down
	{
		int button;
    	AlContinuousFunction::translateInput( input, button );
    	int modifier = hFunc.inputModifierMask() & ~button;
	//	fprintf( stderr, "process up...%d, %d, %d\n", input, button, modifier );

		if ( modifier & kModifierControl )	// Control pressed, reset xform
		{
			if ( gTrans == 0 && gRot == 0 && gScale == 1.0 )
			{
				gTrans = gTransRedo;
				gRot = gRotRedo;
				gScale = gScaleRedo;
			}
			else
			{
				gTransRedo = gTrans; 
				gRotRedo = gRot;
				gScaleRedo = gScale;
				gTrans = gRot = 0;
				gScale = 1.0;
			}
		//	fprintf( stderr, "undo... " );
		//	fprintf( stderr, "(%2.2lf, %2.2lf, %2.2lf)\n", gTrans, gRot, gScale );
			jXform();
		}
		else if ( (modifier & kModifierShift) ) //	(button == kRB) )	// Shift + RB, repeat
		{
		//	fprintf( stderr, "repeat..." );
		//	cleanup_func();
		//	jProcessPicked( );

			// make picked polygons into clusters
			int i;
			for ( i = 0; i < gPicked; i++ )
			{
			//	gClusters[i] = jCreateClusters( gPsets[i] );

				jDuplicatePolys( gPsets[i] );
				gClusters[i]->InitVertCoords();
			}
			jXform();
		}
	}
	gX = x;
	gY = y;
}

static void move_func( int input, Screencoord x, Screencoord y )
{
	if ( !gPicked )
		return;

	int button;
    AlContinuousFunction::translateInput( input, button );
    int modifier = hFunc.inputModifierMask() & ~button;

	if ( modifier & kModifierControl )	// Control pressed, ignore
	{
	//	fprintf( stderr, "process move...%d, %d, %d\n", input, button, modifier );
		return;
	}

    //
    // Move those vertices!
    //
//	fprintf( stderr, "move...\n" );
//	fprintf( stderr, "process move...%d, %d, %d\n", input, button, modifier );

	double dx = x - gX;
	double dy = y - gY;
	double dist = sqrt( (dx * dx) + (dy * dy ) );
	dist /= 1024.0;

	if ( x < gX )
		dist = -dist;

	switch( button )
	{
		case kMB:
			gRot += dist;
			break;
		case kRB:
			gScale += dist;
			break;
		default:
			gTrans += dist;
			break;
	}

	gX = x;
	gY = y;

	jXform();
}

static void up_func( int /*input*/, Screencoord /*x*/, Screencoord /*y*/ )
{
/*
	int button;
   	AlContinuousFunction::translateInput( input, button );
   	int modifier = hFunc.inputModifierMask() & ~button;

//	fprintf( stderr, "process up...%d, %d, %d\n", input, button, modifier );

	if ( modifier & kModifierControl )	// Control pressed
	{
		if ( gTrans == 0 && gRot == 0 && gScale == 1.0 )
		{
			gTrans = gTransRedo;
			gRot = gRotRedo;
			gScale = gScaleRedo;
		}
		else
		{
			gTransRedo = gTrans; 
			gRotRedo = gRot;
			gScaleRedo = gScale;
			gTrans = gRot = 0;
			gScale = 1.0;
		}
	//	fprintf( stderr, "undo... " );
	//	fprintf( stderr, "(%2.2lf, %2.2lf, %2.2lf)\n", gTrans, gRot, gScale );
		jXform();
	}
*/
//	fprintf( stderr, "up...\n" );

	if ( gPicked )
		AlPrintf( kPrompt, "LB: extrude, MB: rotate, RB: scale.  Shift-LB to repeat last extrude, Ctrl-RB to reset/redo current transform ( %2.2lf, %2.2lf, %2.2lf )",
			gTrans, gRot / M_PI * 180.0, gScale );
}


/*
*
*/
static void jSwapVertices( AlPolygon *pgon, 
int old1, int old2, int new1, int new2  )
{
//	fprintf( stderr, "Swapping (%d, %d) to (%d, %d)\n", old1, old2, new1, new2 );

	int i, j, nv = pgon->numberOfVertices();
	int *vlist = new int[nv + 1];

	j = 0;
	for ( i = nv - 1; i >= 0; i-- )
	{
		AlPolysetVertex vtx;
		pgon->vertexD( i, vtx );
	//	fprintf( stderr, "vtx %d: %d\n", i, vtx.index() );

		if ( vtx.index() == old1 )
			vlist[j] = new1;
		else if ( vtx.index() == old2 )
			vlist[j] = new2;
		else
			vlist[j] = vtx.index();
		j++;
		pgon->removeVertex( i );
	}
//	fprintf( stderr, "Poly %d now has: \n", pgon->index() );
	for ( i = j-1; i >= 0; i-- )
	{
		pgon->addVertex( vlist[i] );
	//	fprintf( stderr, "	%d: \n", vlist[i] );
	}

    delete[] vlist;
}

/*
*
*/
static int jPolyPicked( AlPolygon *pgon )
{
	int i;

	for ( i = 0; i < pgon->numberOfVertices(); i++ )
	{
		AlPolysetVertex vtx;
		pgon->vertexD( i, vtx );
		if ( !vtx.isPicked() )
			return 0;
	}
	return 1;
}

/*
*
*/
static Cluster* jCreateClusters( AlPolyset *pset )
{
	int np = pset->numberOfPolygons();
	AlPolygon pgon;
	AlPolysetVertex vtx;

	Cluster *cluster = new Cluster( np, pset );

	char st[255];
	int i, j, marker = 0;
	for ( i = 0; i < np; i++ )	// insert picked polys into cluster
	{
		marker = int(10.0 / np * i);
		if ( marker != int(10.0 / np * (i-1) ) )
		{
			sprintf( st, "Processing picked... [" );
			for ( j = 0; j < marker; j++ )
				sprintf( st, "%s%c", st, '>' );
			for ( j = marker; j < 10; j++ )
				sprintf( st, "%s-", st );
			AlPrintf( kPrompt, "%s]", st );
		}
		pset->polygonD( i, pgon );
		if ( jPolyPicked( &pgon ) )
		{
			cluster->InsertPolygon( &pgon );
		}
	}

	cluster->Consolidate( );
	cluster->CalcNormals( );
	cluster->InsertVertex( );

	return cluster;
}

/*
*
*/
static void jDuplicatePolys( AlPolyset *pset )
{
	int nv = pset->numberOfVertices();
	int np = pset->numberOfPolygons();

	int *vData = new int[nv];
	PolyData *pData = new PolyData[np];

	int i, j, lastPicked;
	double x, y, z;

	for ( i = 0; i < nv; i++ )
		vData[i] = -1;

	AlPolygon pgon;
	AlPolysetVertex vtx;

	for ( i = 0; i < np; i++ )	// check all polys
	{
		pset->polygonD( i, pgon );

		lastPicked = -999;
		int nVts = pgon.numberOfVertices();
		if ( jPolyPicked( &pgon ) )
			continue;
		for ( j = 0; j < nVts; j++ )	// create wall polys
		{
			pgon.vertexD( j, vtx );
			if ( vtx.isPicked() )
			{
				int doLast = 0;
				if ( j == 0 )
				{
					AlPolysetVertex lastV;
					pgon.vertexD( nVts - 1, lastV ); 
					if ( lastV.isPicked() )
					{
						lastPicked = nVts-1;
						doLast = 1;
					}
				}

				if ( (lastPicked == j - 1)  || doLast )	// got an edge
				{
					AlPolysetVertex *oldV1 = pgon.vertex( lastPicked );
					AlPolysetVertex *oldV2 = pgon.vertex( j );
					int oV1 = oldV1->index();
					int oV2 = oldV2->index();

					int nV1;
					if ( vData[ oV1 ] == -1 )	// no new vtx created
					{
						oldV1->unaffectedPosition( x, y, z );
						nV1 = pset->newVertex( x, y, z );
						vData [ oV1 ] = nV1;
					}
					else
						nV1 = vData[ oV1 ];

					int nV2;
					if ( vData[ oV2 ] == -1 )
					{
						oldV2->unaffectedPosition( x, y, z );
						nV2 = pset->newVertex( x, y, z );
						vData[ oV2 ] = nV2;
					}
					else
						nV2 = vData[ oV2 ];

					int newP = pset->newPolygon();
					AlPolygon * newPgn = pset->polygon( newP );

					newPgn->addVertex( nV2 );
					newPgn->addVertex( nV1 );
					newPgn->addVertex( oV1 );
					newPgn->addVertex( oV2 );
					delete newPgn;

					pData[ pgon.index() ].InsertData( oV1, oV2, nV1, nV2 );

				//	AlPolysetVertex newV1, newV2;

				//	pset->vertexD( nV1, newV1 );
				//	pset->vertexD( nV2, newV2 );

				//	newV1.pick();
				//	newV2.pick();
				//	oldV1->unpick();
				//	oldV2->unpick();

					delete oldV1;
					delete oldV2;
				}	// if lastPicked = j - 1

				else	// no edge with last vtx.  check next vtx
				{
					int nextV;
					if ( j == nv - 1 )
						nextV = 0;
					else
						nextV = j+1;

					AlPolysetVertex nVtx;
					pgon.vertexD( nextV, nVtx );
					if ( !nVtx.isPicked() )	// this picked vertex is not connected to any edges.  must create new vtx;
					{
						int oV1 = vtx.index();
						int nV1;
						if ( vData[ oV1 ] == -1 )
						{
							vtx.unaffectedPosition( x, y, z );
							nV1 = pset->newVertex( x, y, z );
							vData[ oV1 ] = nV1;
						}
						else
							nV1 = vData[ oV1 ];
						pData[ pgon.index() ].InsertData( oV1, -1, nV1, -1 );
					}
				}
				lastPicked = j;

			}	// if j = 0
		}	// else for
	}

	for ( i = 0; i < np; i++ )
	{
		if ( pData[i].count != 0 )
		{
			AlPolygon pgon;
			pset->polygonD( i, pgon );
			for ( j = 0; j < pData[i].count; j++ )
			{
			//	fprintf( stderr, "Processing polygon %d:\n", pgon.index() );
				jSwapVertices( &pgon, pData[i].oldV1[j], pData[i].oldV2[j], pData[i].newV1[j], pData[i].newV2[j] );
			}
		}
	}

	delete[] vData;
	delete[] pData;
	pset->calcNormals( TRUE );
}

/*
*
*/
static void jProcessPicked( )
{
	AlPickList::firstPickItem();
	if ( AlPickList::getObject() )	// go through all picked objects
	{
/*
    	AlAnswerType answer;
    	AlPromptBox( kOK_Cancel, "Delete polygons?", &answer, 200, 200 );

    	if ( answer != kOK && answer != kYes )
        	return;
*/
		AlObject * obj;
		AlPolyset *newpset, *oldpset = NULL;
		AlPolysetVertex *v, vtx;
		int i;

		do	// go through all picked objects
		{
			if ( obj = AlPickList::getObject() )
			{
				v = obj->asPolysetVertexPtr();

				if ( v )	// is a polyset vertex
				{
					newpset = v->polyset();

					if ( !newpset )
						continue;

					if ( AlAreEqual( newpset, oldpset ) )
						delete newpset;
					else
					{
						int j, nv = newpset->numberOfVertices();
						for ( j = 0; j < nv; j++ )
						{
							newpset->vertexD( j, vtx );
							if ( !vtx.isPicked() )	// unpicked vertex exists
							{
								if ( !(gPicked < MAX_OBJS) )
									return;
								gPsets[ gPicked++ ] = newpset;
								j = nv + 999;
							}
						}
						if ( j == nv )	// all vtx are picked, not valid object
						{
							fprintf( stderr, "\n" );
							AlPolysetNode *pnode = newpset->polysetNode();
							AlPrintf( kPrompt, "WARNING: %s has all of its vertices picked.  Ignored.", pnode->name() );
							delete pnode;
						}
						oldpset = newpset;
					}
				}
				delete obj;
			}

		} while ( AlPickList::nextPickItem() == sSuccess );

		for ( i = 0; i < gPicked; i++ )
		{
			jDuplicatePolys( gPsets[i] );
		}
	}

	if ( gPicked )
	{
	//	AlPickList::clearPickList();
	//	AlUniverse::redrawScreen();
	}
	else
	{
		fprintf( stderr, "\n" );
		AlPrintf( kPrompt, "Please select the vertices in the polygons you want to extrude." );
	}
}



// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	// Invoke the scheme file which sets defaults for the scheme
	// variables.
	// AlInvokeSchemeFile( "wedge_init.scm" );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	hFunc.create( "pl_jptExtrudePolys", init_func, down_func, move_func,
		up_func, cleanup_func, TRUE );

	h.create( "Extrude Polygons", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "extrude_polys" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV or an absolute path, and
	// the second argument is the name of the option box given
	// in the scheme file. If a call to this method is omitted
	// there will be no option box.
	// h.setOptionBox( "wedge.scm", "wedge.options" );

	h.setIconPath( makeAltPath( dirName, NULL ) );

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "al_polyedit" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.

    AlPrintf( kPrompt, "Extrude Polygons plug-in installed under 'Polygon Edit' palette");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()

//	AlUniverse::redrawScreen();

	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}


