/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
exportMTL.plugin ( formerly jptExportMtl.plugin )

	This plugin exports all shaders to a Wavefront .MTL file
	for OBJ materials.  The actual OBJ file can be saved
	using 'File->Save All As' and selecting the OBJ file format. 

	Installed under Render menu.

Options:

	Illumination Model: sets the material illumination model flag 
		for all Lambert shaders.

	Ambient Intensity: since Alias can't handle object ambient
		shading, this value is used for all shaders and entered
		in the .MTL file as 'Ka   ambient ambient ambient'.

Limitations:

	Only the following entries are supported at this time:
		Ka
		Kd
		illum
		Ns
		Ks
		map_Ka
		map_Kd
		map_Ks

	Only color and specular textures are supported, and only if
		they are mapped to a File texture.

	When a color map is detected, Kd is set to (1 1 1).  If you
		don't want this, you'll have to remove the color map.

	All texture names are exported as full path name.

	You'll probably need to set the WF_TEXTURE_DIR and 
		WF_MTL_LIB environment variable later.

	Color per vertex is not supported due to Alias exported
		OBJ file limitation.

	When overwriting a previous generated .MTL file, Alias
		will incorrectly warn about writing over a different
		file type.  Just ignore it.


*/

#include <AlUniverse.h>

#include <AlList.h>
#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlShader.h>
#include <AlMappedFieldItem.h>
#include <AlShadingFieldItem.h>
#include <AlTexture.h>
#include <AlTextureNode.h>
#include <AlIterator.h>
#include <stdio.h>
#include <string.h>

/*
*
*/
class jDoTexture : public AlIterator
{
	FILE *outfile;
	int gotIt;
public:
	jDoTexture( FILE *file ) { outfile = file; gotIt = 0; }
	~jDoTexture() {;}
	boolean gotTexture() { return gotIt; }
	int func( AlObject* object )
	{
		AlTexture *texture = object->asTexturePtr();
		if ( !texture )
			return 0;
		if ( texture->filename() )
		{
			if ( !strcmp( texture->fieldType(), "color" ) )
			{
				gotIt = 1;
				if ( outfile )
				{
					fprintf( outfile, "map_Ka  %s\n", texture->filename() );
					fprintf( outfile, "map_Kd  %s\n", texture->filename() );
				}
			}
			else if ( outfile && !strcmp( texture->fieldType(), "specular" ) )
			{
				fprintf( outfile, "map_Ks  %s\n", texture->filename() );
			}
/*
			else if ( !strcmp( texture->fieldType(), "bump" ) )
			{
				fprintf( outfile, "map_d   %s\n", texture->filename() );
			}
*/
		}

/*
		int retval;
		texture->applyIteratorToTextures(this, retval);
*/

		return 0;
	}
};

/*
*
*/
static void jPrintValues( FILE *outfile, AlShader *shader,
	int map, int illum, double ambient )
{
	double value;
//	fprintf( stderr, "Model: %s\n", shader->shadingModel() );

	fprintf( outfile, "Ka      %2.2lf %2.2lf %2.2lf\n", ambient, ambient, ambient );
/*
	fprintf( outfile, "Ka      " );
	shader->parameter( kFLD_SHADING_COMMON_COLOR_R, value );
	fprintf( outfile, "%2.2lf ", value / 2000.0 );
	shader->parameter( kFLD_SHADING_COMMON_COLOR_G, value );
	fprintf( outfile, "%2.2lf ", value / 2000.0 );
	shader->parameter( kFLD_SHADING_COMMON_COLOR_B, value );
	fprintf( outfile, "%2.2lf\n", value / 2000.0 );
*/
	if ( map )
	{
		fprintf( outfile, "Kd      1.0 1.0 1.0\n" );
	}
	else
	{
		fprintf( outfile, "Kd      " );
		shader->parameter( kFLD_SHADING_COMMON_COLOR_R, value );
		fprintf( outfile, "%2.2lf ", value / 255.0 );
		shader->parameter( kFLD_SHADING_COMMON_COLOR_G, value );
		fprintf( outfile, "%2.2lf ", value / 255.0 );
		shader->parameter( kFLD_SHADING_COMMON_COLOR_B, value );
		fprintf( outfile, "%2.2lf\n", value / 255.0 );
	}
/*
	fprintf( outfile, "Tf      " );
	shader->parameter( kFLD_SHADING_COMMON_TRANSPARENCY_R, value );
	fprintf( outfile, "TRANS R: %2.2lf\n", value );
	shader->parameter( kFLD_SHADING_COMMON_TRANSPARENCY_G, value );
	fprintf( outfile, "TRANS G: %2.2lf\n", value );
	shader->parameter( kFLD_SHADING_COMMON_TRANSPARENCY_B, value );
	fprintf( outfile, "TRANS B: %2.2lf\n", value );
*/

	shader->parameter( kFLD_SHADING_COMMON_REFRACTIVE_INDEX, value );
//	fprintf( outfile, "Ni      %2.2lf\n", value );

	if ( !strcmp( shader->shadingModel(), "PHONG" ) )
	{
		fprintf( outfile, "illum   2\n" );
		shader->parameter( kFLD_SHADING_PHONG_SHINYNESS, value );
		fprintf( outfile, "Ns      %2.2lf\n", value * 10 );

		fprintf( outfile, "Ks      " );
		shader->parameter( kFLD_SHADING_PHONG_SPECULAR_R, value );
		fprintf( outfile, "%2.2lf ", value );
		shader->parameter( kFLD_SHADING_PHONG_SPECULAR_G, value );
		fprintf( outfile, "%2.2lf ", value );
		shader->parameter( kFLD_SHADING_PHONG_SPECULAR_B, value );
		fprintf( outfile, "%2.2lf\n", value );
	}
	else if ( !strcmp( shader->shadingModel(), "BLINN" ) )
	{
		fprintf( outfile, "illum   2\n" );
		shader->parameter( kFLD_SHADING_BLINN_ECCENTRICITY, value );
		fprintf( outfile, "Ns      %2.2lf\n", (1-value) * 1000 );

		fprintf( outfile, "Ks      " );
		shader->parameter( kFLD_SHADING_BLINN_SPECULAR_R, value );
		fprintf( outfile, "%2.2lf ", value );
		shader->parameter( kFLD_SHADING_BLINN_SPECULAR_G, value );
		fprintf( outfile, "%2.2lf ", value );
		shader->parameter( kFLD_SHADING_BLINN_SPECULAR_B, value );
		fprintf( outfile, "%2.2lf\n", value );
	}
	else
	{
		fprintf( outfile, "illum   %d\n", illum );
	//	fprintf( outfile, "Ns      800\n" );
	}


/*
    kFLD_SHADING_PHONG_DIFFUSE                       = 20,
    kFLD_SHADING_PHONG_REFLECTIVITY                  = 25,
    kFLD_SHADING_BLINN_DIFFUSE                       = 27,
    kFLD_SHADING_BLINN_SPECULAR_ROLLOFF              = 31,
    kFLD_SHADING_BLINN_REFLECTIVITY                  = 33,
*/

}

/*
*
*/
static char* jFixName( const char *name )
{
	char *newName = strdup(name);

	int i;
	for ( i = 0; i < strlen(newName); i++ )
		if ( newName[i] == '#' )
			newName[i] = '_';
	return newName;
}

/*
*
*/
static void jExportShader( FILE *outfile, AlShader *shader, int illum, double ambient )
{
	if ( shader == NULL )
		return;

	char *newName = jFixName( shader->name() );
	fprintf( outfile, "\n" );
	fprintf( outfile, "newmtl  %s\n", newName );
	free( newName );

    // Check the shader's textures

	int ret;
	jDoTexture checkTex( NULL );
	shader->applyIteratorToTextures( &checkTex, ret );

    // Print the necessary values on this shader

	jPrintValues( outfile, shader, checkTex.gotTexture(), illum, ambient );

    // Print the shader's textures

	jDoTexture printTex( outfile );
	shader->applyIteratorToTextures( &printTex, ret );

	AlShader *next = AlUniverse::nextShader( shader );
	jExportShader( outfile, next, illum, ambient );
	delete next;
}

/*
*
*/
static void do_it( void )
{
	FILE *outfile;
	char *filename = NULL;

	if ( sSuccess != AlFileBrowser( kFileBrowseWrite, &filename, "Export", FALSE, NULL ) )
	{
		free( filename );
		return;
	}

	outfile = fopen( filename, "w" );

	if ( outfile == NULL )
	{
		fprintf(stderr, "^G\n");
		AlPrintf(kPrompt, "*** ERROR: Can't open file '%s'.\n", filename );
		free( filename );
		return;
	}

	int illum;
	double ambient;

	if ( sSuccess != AlGetInteger("exportMTL.illum", illum) )
		illum = 1;
	if ( sSuccess != AlGetDouble("saveSquare.ambient", ambient) )
		ambient = 0.0;

	AlShader *shader = AlUniverse::firstShader();

	jExportShader( outfile, shader, illum, ambient );

	free( filename );
	delete shader;
	fclose( outfile );
};

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	char *dirScm = makeAltPath( dirName, "scm" );

	// Invoke the scheme file which sets defaults for the scheme
	// variables.
	// AlInvokeSchemeFile( "nothing_init.scm" );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be nothing_func.S, nothing_func.M
	hFunc.create( "pl_jptExportMTL", do_it );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.

	h.create( "Export MTL", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "export_mtl" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV or an absolute path, and
	// the second argument is the name of the option box given
	// in the scheme file. If a call to this method is omitted
	// there will be no option box.
	h.setOptionBox( "exportMTL.scm", "exportMTL.options", dirScm );

	//	Alternatively, you could have used:
	//	h.setOptionBox( "nothing.scm", "nothing.options", dirName )

	h.setIconPath( makeAltPath( dirName, NULL ) );

	//
	//	Then the scheme file and the icon images would be loaded from
	//	the same path as the plugin rather than from the standard locations

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "rp_render" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Export MTL plug-in installed under Menu 'Render'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}
