/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
exportCameraData.plugin ( formerly jptSaveCamera.plugin )

    This plugin exports animation values of the perspective
    camera.  Exported data include:

        World position of Eye ( x, y, z )
        World position of View ( x, y, z )
        World position of Up ( x, y, z )
        Angle of view ( angle )
        Twist angle ( angle )
        Film back ( w, h, z )
        Film offset ( x, y )
        Focal length ( length )
        Depth of field ( isOn, fStop, focalDistance )

	Installed under Animation menu.

Options:

    Output File:
        Name of the camera data file to be written.

    Start, End, By:
        Frames to be exported.

    Use This Camera Only:
        If you have multiple cameras in your scene, you can
        specify the name of the camera to be exported.
        Otherwise if this flag is off, all cameras will be exported.

Format:

    camera1
    frame 1
    eye
    view
    up
    angle
    twist
    film back
    film offset
    depth
    frame 2
    eye
    ...

    camera2
    frame 1
    eye
    view
    up
    ...

Version History:

    Icon now stays on the shelf after quit.

*/


#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlObject.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlIterator.h>
#include <AlDagNode.h>
#include <AlGroupNode.h>
#include <AlCameraNode.h>
#include <AlPerspectiveCamera.h>
#include <AlPlayFrame.h>

#include <stdio.h>
#include <string.h>

#define MAX_CAMS 64

AlPerspectiveCamera *gCameras[MAX_CAMS];
int gNumCam = 0;
int gEye =1;
int gView =1;
int gUp =1;
int gAngle =0;
int gTwist = 0;
int gOffset = 0;
int gBack = 0;
int gLength = 0;
int gDepth = 0;
int gUseName = 0;

const char *gCameraName;

FILE *gOutfile = NULL;

void jDeleteCameras()
{
	int i;

	for (i = 0; i < gNumCam; i++)
		delete gCameras[i];
}

void jProcessObject( AlDagNode *object )
{
	if ( object == NULL )
		return;

	if ( object->asCameraNodePtr() )
	{
	//	fprintf(stderr, "Camera found!!!\n");
		if ( ((AlCameraNode*)object)->isUpNode() )
		{
			AlPerspectiveCamera *camera = (AlPerspectiveCamera*)((AlCameraNode*)object)->camera();
			// fprintf(stderr, "Camera = %s\n", camera->name() );
			if (gUseName)
			{
				if ( !strcmp(camera->name(), gCameraName) )
				{
					if ( gNumCam < MAX_CAMS )
						gCameras[gNumCam++] = camera;
					// fprintf(stderr, "camera found!!!\n");
				}
			}
			else if ( gNumCam < MAX_CAMS )
				gCameras[gNumCam++] = camera;
		}
	}

	if ( object->asGroupNodePtr() )
	{
		AlDagNode *child = ((AlGroupNode*)object)->childNode();

//		fprintf(stderr, "-----  child node found...\n");
		jProcessObject( child );

		delete child; 
	}

	AlDagNode *next = object->nextNode();
	jProcessObject( next );

	delete next;
}

void jOutputCamera( AlPerspectiveCamera *camera )
{
	double x1, x2, x3, y1, y2, y3, z1, z2, z3;
	camera->worldEyeViewUp(
			x1, y1, z1,
			x2, y2, z2,
			x3, y3, z3 );
	if ( gEye )
		fprintf(gOutfile, "%2.16lf, %2.16lf, %2.16lf\n", x1, y1, z1);

	if ( gView )
		fprintf(gOutfile, "%2.16lf, %2.16lf, %2.16lf\n", x2, y2, z2);

	if ( gUp )
		fprintf(gOutfile, "%2.16lf, %2.16lf, %2.16lf\n", x3, y3, z3);

	if ( gAngle )
		fprintf(gOutfile, "%lf\n", camera->angleOfView() );

	if ( gTwist )
		fprintf(gOutfile, "%lf\n", camera->twistAngle() );

	if ( gBack )
	{
		camera->filmBack( x1, x2 );
		fprintf(gOutfile, "%2.16lf, %2.16lf\n", x1, x2 );
	}

	if ( gOffset )
	{
		camera->filmOffset( x1, x2 );
		fprintf(gOutfile, "%2.16lf, %2.16lf\n", x1, x2 );
	}

	if ( gLength )
		fprintf(gOutfile, "%lf\n", camera->focalLength() );

	if ( gDepth )
	{
		boolean isOn;
		camera->depthOfField( isOn, x1, x2 );
		fprintf(gOutfile, "%d, %2.16lf, %2.16lf\n", isOn, x1, x2 );
	}
}

void do_it( void )
{
	gNumCam = 0;


	if ( sSuccess != AlGetInteger("saveCamera.use", gUseName) )
		gUseName = 0;
	if ( gUseName )
		if ( sSuccess != AlGetString("saveCamera.name", gCameraName) )
		{
			gCameraName = NULL;
			gUseName = 0;
		}
	// fprintf(stderr, "gUseName = %d, name = %s\n", gUseName, gCameraName);

	AlDagNode *dag = AlUniverse::firstDagNode();

	jProcessObject( dag );

	if (gUseName && !gNumCam )
	{
		AlPrintf(kPrompt, "*** Warning: Camera '%s' not found.\n", gCameraName);
		fprintf(stderr, "\n");
	}

	delete dag;

	double start, end, by;
	const char *filename;

	if ( sSuccess != AlGetDouble("saveCamera.start", start) )
		start = 1.0;
	if ( sSuccess != AlGetDouble("saveCamera.end", end) )
		end = 30.0;
	if ( sSuccess != AlGetDouble("saveCamera.by", by) )
		by = 1.0;

	if ( by == 0.0 )
		by = 1.0;

	if ( start > end )
	{
		double tmp = start;
		start = end;
		end = tmp;
	}
	if ( by < 0 ) 
		by = -by;

	if ( sSuccess != AlGetInteger("saveCamera.eye", gEye) )
		gEye = 1;
	if ( sSuccess != AlGetInteger("saveCamera.view", gView) )
		gView = 1;
	if ( sSuccess != AlGetInteger("saveCamera.up", gUp) )
		gUp = 1;
	if ( sSuccess != AlGetInteger("saveCamera.angle", gAngle) )
		gAngle = 0;
	if ( sSuccess != AlGetInteger("saveCamera.length", gLength) )
		gLength = 0;
	if ( sSuccess != AlGetInteger("saveCamera.offset", gOffset) )
		gOffset = 0;
	if ( sSuccess != AlGetInteger("saveCamera.back", gBack ) )
		gBack = 0;
	if ( sSuccess != AlGetInteger("saveCamera.twist", gTwist) )
		gTwist = 0;
	if ( sSuccess != AlGetInteger("saveCamera.depth", gDepth) )
		gDepth = 0;

	if ( sSuccess != AlGetString("saveCamera.file", filename) )
		filename = strdup( "camera.anim" );


//	fprintf(stderr, "Writting output to %s...\n", filename );
	if ( (gOutfile = fopen(filename, "w")) == NULL )
	{
		AlPrintf(kPrompt, "*** Warning: Can't open %s for output.  Using stderr.\n\n", filename);
		gOutfile = stderr;
	}

	int j;
	for ( j = 0; j < gNumCam; j++ )
	{
		fprintf(gOutfile, "\n\n%s\n", gCameras[j]->name() );

		// fprintf(stderr, "%lf, %lf, %lf\n", start, end, by); 
		AlPlayFrame playFrame;
		double i;
		for ( i = start; i <= end; i += by )
		{
			playFrame.viewFrame( i, FALSE );
			fprintf(gOutfile, "\n%2.4lf\n", i);
			jOutputCamera( gCameras[j] );
		}
	}

	jDeleteCameras();
	if ( gOutfile != stderr && !(gUseName && !gNumCam ))
	{
		fclose( gOutfile );
		AlPrintf(kPrompt, "Camera animation saved to file %s\n", filename );
	}
//	delete filename;
}


// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
//	fprintf(stderr, "DirName = %s\n", dirName);

	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	char *dirScm = makeAltPath( dirName, "scm" );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be nothing_func.S, nothing_func.M
	hFunc.create( "pl_jptSaveCamera", do_it );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.

	h.create( "Export Camera Data", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "save_camera" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV or an absolute path, and
	// the second argument is the name of the option box given
	// in the scheme file. If a call to this method is omitted
	// there will be no option box.
    h.setOptionBox( "exportCameraData.scm", "exportCamera.options", dirScm );


	h.setIconPath( makeAltPath( dirName, NULL ) );

	//
	//	Then the scheme file and the icon images would be loaded from
	//	the same path as the plugin rather than from the standard locations

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "ap_animwinds" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Export Camera Data plug-in installed under Menu 'Animation'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}
