/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

// dynamicEditorExample.plugin
// 
// This is a skeleton momentary function plugin.  It implements a simple
// momentary function that calls the do_stuff_here() function.  
// The option boxes that are used in this plug-in are dynamic.
//

#include <AlUniverse.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlEditor.h>

#ifdef _WIN32
#include <string.h>
#else
#include <strings.h>
#endif

//
// Editor types
//

// Structure used for creating a descriptor
// list from an array.
typedef struct
{
	const char *label;
} StringItem;

//
// Forward prototypes
//

static ComponentDescriptor *idToDescriptor( ComponentId itemId );
static ComponentDescriptor *labelToDescriptor( const char *label );
static void printDescriptorList( void );
static void printComponentInformation( ComponentDescriptor *descriptor );
static bool matchDescriptorToLabel( ComponentDescriptor *descriptor, const char *label, bool allowPartialMatch = FALSE );

//
// Permanent storage for the plug-in's lifetime.  This
// storage must be permanent( wrt the plug-ins' life if
// not programming errors will occur as the variable is
// removed from the stack.
//

static int checkValue = 1;
static float floatValue = 10;
static int intValue = 33;
static float tripleValue[3] = { 1, 2, 3 };

static AlEditor *apieditor = NULL;

//
// Editor callback functions
//

// Callback that is used for most components.
static void idCallback( ComponentId id )
{
	ComponentDescriptor *descriptor = idToDescriptor( id );
	if ( descriptor != NULL )
		printComponentInformation( descriptor );	
	else
		printf("idCallback: %d\n",id);


	if ( matchDescriptorToLabel( descriptor, "Check Box" ) )
	{
		printf("Found Check Box...\n");
		ComponentDescriptor *d = labelToDescriptor( "Button A" );			
		if ( d != NULL )
		{
			printf("disabling Button A\n");
			apieditor->enableItem( d->id(), (boolean) checkValue );
			apieditor->update();
		}
		d = labelToDescriptor( "Float" );			
		if ( d != NULL )
		{
			printf("disabling Float\n");
			apieditor->enableItem( d->id(), (boolean) checkValue );
			apieditor->update();
		}
		d = labelToDescriptor( "menu-item-a-3" );			
		if ( d != NULL )
		{
			printf("disabling menu-item-a-3\n");
			if ( apieditor->enableItem( d->id(), (boolean) checkValue )!=sSuccess)
				printf("\tfailed to disable menu item\n");
			apieditor->update();
		}
	}
}

// Callback used for string components.
static void idStringCallback( ComponentId id, const char *str )
{
	ComponentDescriptor *descriptor = idToDescriptor( id );
	if ( descriptor != NULL )
		printf("Component: label <%s> Id <%d> Data <%s>\n",descriptor->label(),id,str);	
	else
		printf("idStringCallback: %d %s\n",id,str);
}

//
// Editor utility functions
//

// Component ids are allocated dynamically by the AlEditor class. As a result, the 
// plug-in code must build a list of ComponentDescriptor so that ids and names
// can be matched in the callback handlers.  

static ComponentDescriptor *descriptorList = NULL;

// Returns a component list that is built from the StringItem array.
ComponentDescriptor *buildDescriptorListFromStringList( StringItem *array, int arraySize )
{
	if ( array == NULL || arraySize < 1 )
		return NULL;
		
	int i;
	ComponentDescriptor *head = NULL;
	for ( i=0; i < arraySize; i++ )
	{
		ComponentDescriptor *descriptor = new ComponentDescriptor( array[i].label );
		if ( descriptor == NULL )
			return NULL;
			
		if ( head == NULL )
		{
			head = descriptor;
		}
		else
		{
			ComponentDescriptor *n = head;
			head = descriptor;
			head->setNext( n );
		}
	}
	return head;
}

// De-allocate a descriptor list.
void deleteADescriptorList( ComponentDescriptor *head )
{
	ComponentDescriptor *descriptor = head;
	while ( descriptor )
	{
		ComponentDescriptor *next = descriptor->next();
		delete descriptor;
		descriptor = next;
	}	
}

// Starting from 'head', print a descriptor list.
void printADescriptorList( ComponentDescriptor *head, const char *info )
{
	if ( head == NULL )
		return;
	if ( info == NULL )
		info = "NO INFO:";
		
	printf("%s\n",info);
	ComponentDescriptor *descriptor = head;
	while ( descriptor )
	{
		printf("\tid: %d label <%s>\n",descriptor->id(),descriptor->label());
		descriptor = descriptor->next();
	}	
}

// Print the global descriptor list for the plug-in.
void printDescriptorList( void )
{
	printADescriptorList( descriptorList, "PRINT DESCRIPTOR LIST:" );
}

// Add a descriptor to the global descriptor list.
void addToDesciptorList( ComponentDescriptor *descriptor )
{
	if ( descriptor == NULL )
		return;

	if ( descriptorList == NULL )
	{
		descriptorList = descriptor;
	}
	else
	{
		ComponentDescriptor *d = descriptorList;	// Current head
		descriptorList = descriptor;				// New head
		
		// Find the end of the new head
		ComponentDescriptor *d2 = descriptorList;
		while ( d2 != NULL )
		{
			// Are we at the end
			if ( d2->next() == NULL )
			{
				d2->setNext( d );
				break;
			}
			d2 = d2->next();
		}
	}
}

// De-allocate the global descriptor list
void deleteDescriptorList( void )
{
	deleteADescriptorList( descriptorList );
	descriptorList = NULL;
}

// Search the global descriptor list for itemId and 
// return the associated descriptor.
ComponentDescriptor *idToDescriptor( ComponentId itemId )
{
	ComponentDescriptor *d = descriptorList;
	while ( d != NULL )
	{
		if ( d->id() == itemId )
			return d;
		d = d->next();
	}
	return NULL;
}

// Search the global descriptor list for label and return the
// associated descriptor.
ComponentDescriptor *labelToDescriptor( const char *label )
{
	if ( label == NULL )
		return NULL;
		
	ComponentDescriptor *d = descriptorList;
	while ( d != NULL )
	{
		const char *dlabel = d->label();
		if ( strcmp( label, dlabel ) == 0  )
			return d;
		d = d->next();
	}
	return NULL;
}

// Testing function to disable components
void disableSomeComponents( void )
{
	ComponentDescriptor *d = descriptorList;
	while ( d != NULL )
	{
		boolean skip = matchDescriptorToLabel( d, "MenuA" ) | matchDescriptorToLabel( d, "Button A" ) |
			matchDescriptorToLabel( d, "menu-item-a-3" ) | matchDescriptorToLabel( d, "radio-item-2" );
		if ( ! skip )
		{
			if ( apieditor->isEnabled( d->id() ) )
			{
				if ( apieditor->enableItem( d->id(), FALSE ) == sSuccess )
					printf("Disabled: label <%s> Id <%d>\n",d->label(),d->id());
				else
					printf("Failed to disabled: label <%s> Id <%d>\n",d->label(),d->id());			
			}
			else
			{
					printf("Already disabled: label <%s> Id <%d>\n",d->label(),d->id());
			}
		}
		else
		{
			printf("Skipping disable option: label <%s> Id <%d>\n",d->label(),d->id());
		}
		d = d->next();
	}
}

// Simple matching of a string pattern to a descriptor label.
bool matchDescriptorToLabel( ComponentDescriptor *descriptor, const char *label, bool allowPartialMatch )
{
	if ( descriptor == NULL || label == NULL )
		return false;
		
	const char *dlabel = descriptor->label();
	if ( dlabel == NULL )
		return false;
		
	if ( allowPartialMatch )
	{
		if ( strstr( dlabel, label ) != NULL )
			return true;
	}
	else
	{
		if ( strcmp( dlabel, label ) == 0 )
			return true;
	}
	return false;
}

// Prints the component information associated with a descriptor.
// Some components on creation have an associated reference
// variable.  This function outputs the reference information.
void printComponentInformation( ComponentDescriptor *descriptor )
{
	if ( descriptor == NULL || descriptor->label() == NULL )
		return;
		
	char data[100];
	data[0] = 0;

	// If the component has a reference associated with it,
	// output some additional information that the 
	// reference describes.	
	if ( matchDescriptorToLabel( descriptor, "Check Box" ) )
		sprintf(data,"%s", ( checkValue ) ? "On" : "Off" );
	else if ( matchDescriptorToLabel( descriptor, "Float" ) )
		sprintf(data,"%g", floatValue );
	else if ( matchDescriptorToLabel( descriptor, "Integer" ) )
		sprintf(data,"%d", intValue );
	else if ( matchDescriptorToLabel( descriptor, "Triple" ) )
		sprintf(data,"%g %g %g", tripleValue[0], tripleValue[1], tripleValue[2] );
		
	if ( data[0] != 0 )
		printf("Component: label <%s> Id <%d> Data <%s>\n", descriptor->label(), descriptor->id(), data );
	else
		printf("Component: label <%s> Id <%d>\n", descriptor->label(), descriptor->id() );
}

// 
// Editor functions
//

// Build an editor.  Once this call is made, we can add 
// components to the editor.
statusCode makeEditor( void )
{
	apieditor = new AlEditor;
	if ( apieditor == NULL )
	{
		printf("Error: failed to allocate AlEditor class.\n");
		return sFailure;
	}

	if ( apieditor->create( "Dynamic Editor Example" ) != sSuccess )
	{
		printf("Error: failed to create dynamic editor.\n");
		return sFailure;
	}

	return sSuccess;
}

// De-allocate an editor and free up its
// descriptor list.
statusCode removeEditor( void )
{
	if ( apieditor == NULL )
	{
		printf("Error: trying to remove a NULL editor.\n");
		return sFailure;
	}
	
	if ( apieditor->close() != sSuccess )
	{
		printf("Error: failed to close editor\n");
		return sFailure;
	}

	if ( apieditor->deleteEditor() != sSuccess )
	{
		printf("Error: failed to delete editor\n");
		return sFailure;
	}
	
	deleteDescriptorList();

	delete apieditor;
	apieditor = NULL;
		
	return sSuccess;
}

// Add the components to the editor. Note that the plug-in
// can either have a pulldown menu or a button, but not both.
// Adding a component involves the following steps:
//	1. allocate the descriptor.
//	2. allocate a descriptor list if there are sub-components
//	3. allocate the associated data class if required
//	4. add the component to the editor
//	5. add the descriptor to the plug-in's global descriptor list
//	6. add the subcomponent descriptor list(if it exists) to the 
//	plug-in's global descriptor list
//	7. de-allocate the data class if it was created
statusCode buildEditorComponents( boolean addPulldownComponent, boolean addButtonComponent )
{
	ComponentDescriptor *descriptor = NULL;
	ComponentDescriptor *clist = NULL;
	
	descriptor = new ComponentDescriptor("Enter some text");
	if ( apieditor->addString( descriptor, "Hello world!", idStringCallback ) != sSuccess )
	{
		printf("Error: failed to add string component.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );
	
	descriptor = new ComponentDescriptor("Separator");
	if ( apieditor->addSeparator( descriptor ) != sSuccess )
	{
		printf("Error: failed to create separator.\n");			
		return sFailure;
	}
	addToDesciptorList( descriptor );

	descriptor = new ComponentDescriptor("Check Box");
	if ( apieditor->addCheckBox( descriptor, checkValue, idCallback ) != sSuccess )
	{
		printf("Error: failed to create check box.\n");			
		return sFailure;
	}
	addToDesciptorList( descriptor );
	
	descriptor = new ComponentDescriptor("GroupA");
	if ( apieditor->addGrouping( descriptor, FALSE ) != sSuccess )
	{
		printf("Error: failed to add group A.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );

	FloatComponentData *fdata = new FloatComponentData( 10.0, 20.0 );
	descriptor = new ComponentDescriptor("Float");
	if ( apieditor->addFloat(descriptor, floatValue, fdata, idCallback ) != sSuccess )
	{
		printf("\tfailed to create float.\n");			
		return sFailure;
	}
	addToDesciptorList( descriptor );
	delete fdata;
			
	IntComponentData *idata = new IntComponentData( 30, 40 );
	descriptor= new ComponentDescriptor("Integer");
	if ( apieditor->addInt( descriptor, intValue, idata, idCallback ) != sSuccess )
	{
		printf("Error: failed to create int.\n");			
		return sFailure;
	}
	addToDesciptorList( descriptor );
	delete idata;

	descriptor = new ComponentDescriptor("Triple");
	if ( apieditor->add3Floats( descriptor, tripleValue, idCallback ) != sSuccess )
	{
		printf("Error: failed to create triple.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );

	descriptor = new ComponentDescriptor("Group B");	
	if ( apieditor->addGrouping( descriptor ) != sSuccess )
	{
		printf("Error: failed to add group B.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );

	StringItem defaultRadioGroupList[3] = 
	{
		"Choice_1",
		"Choice_2",
		"Choice_3",	
	};
			
	descriptor = new ComponentDescriptor("Radio Group");
	clist = buildDescriptorListFromStringList( defaultRadioGroupList, 3 );
	if ( apieditor->addRadioGroup( descriptor, kDefault, clist, idCallback ) != sSuccess )
	{
		printf("Error: failed to add default radio group.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );
	addToDesciptorList( clist );

	descriptor = new ComponentDescriptor("Group C");	
	if ( apieditor->addGrouping( descriptor ) != sSuccess )
	{
		printf("Error: failed to add group C.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );

	StringItem pushButtonRadioGroupList[5] = 
	{
		"radio-item-0",
		"radio-item-1",
		"radio-item-2",	
		"radio-item-3",
		"radio-item-4",	
	};
	
	descriptor = new ComponentDescriptor("Push Button Radio Group");			
	clist = buildDescriptorListFromStringList( pushButtonRadioGroupList, 5 );
	if ( apieditor->addRadioGroup( descriptor, kPushButton, clist, idCallback ) != sSuccess )
	{
		printf("Error: failed to add push button radio group.\n");
		return sFailure;
	}
	addToDesciptorList( descriptor );
	addToDesciptorList( clist );
		
	if ( addPulldownComponent )
	{
		StringItem aArray[6] =
		{
			"menu-item-a-0",
			"menu-item-a-1",
			"menu-item-a-2",
			"menu-item-a-3",
			"menu-item-a-4",
			"menu-item-a-5",
		};

		StringItem bArray[8] =
		{
			"menu-item-b-0",
			"menu-item-b-1",
			"menu-item-b-2",
			"menu-item-b-3",
			"menu-item-b-4",
			"menu-item-b-5",
			"menu-item-b-6",
			"menu-item-b-7",
		};

		descriptor = new ComponentDescriptor("MenuA");
		clist = buildDescriptorListFromStringList( aArray, 6 );
		if ( apieditor->addPulldownMenu( descriptor, clist, idCallback ) != sSuccess )
		{
			printf("Error: failed to add pulldown A.\n");
			return sFailure;
		}
		addToDesciptorList( descriptor );
		addToDesciptorList( clist );

		descriptor = new ComponentDescriptor("MenuB");
		clist = buildDescriptorListFromStringList(  bArray, 8 );
		if ( apieditor->addPulldownMenu( descriptor, clist, idCallback ) != sSuccess )
		{
			printf("Error: failed to add pulldown B.\n");
			return sFailure;
		}
		addToDesciptorList( descriptor );
		addToDesciptorList( clist );
	}
	
	if ( addButtonComponent )
	{
		descriptor = new ComponentDescriptor("Button A");			
		if ( apieditor->addButton( descriptor, idCallback ) != sSuccess )
		{
			printf("Error: failed to add a button A.\n");
			return sFailure;
		}
		addToDesciptorList( descriptor );

		descriptor = new ComponentDescriptor("Button B");			
		if ( apieditor->addButton( descriptor, idCallback ) != sSuccess )
		{
			printf("Error: failed to add a button B.\n");
			return sFailure;
		}
		addToDesciptorList( descriptor );
	}
				
	return sSuccess;
}

// One the components are added, we can open 
// the editor.
statusCode openEditor( void )
{
	if ( apieditor->open() != sSuccess )
	{
		printf("Error: failed to open editor.\n");
		return sFailure;
	}

	// Leave out if not testing.
	// disableSomeComponents();
		
	return sSuccess;
}

// Entry point that is called when the user
// selects the plug-in menu entry from the
// Utilities menu.
// Note that thisis is a contrived example so
// below we build the editor differently on
// subsequent opens.
void do_dynamic_stuff_here( void ) 
{
	// Has the editor been created?
	if ( apieditor == NULL )
	{
		static int counter = 0;
		makeEditor();
		if ( ( counter % 2 ) == 0 )
			buildEditorComponents( TRUE /* pulldown */, FALSE /* buttons */ );
		else
			buildEditorComponents( FALSE /* pulldown */, TRUE /* buttons */ );	
		openEditor();
		counter++;
	}
	else
	{
		// If we have an editor allocated, but its not open, then
		// the user must have closed it.  We can toggle it open
		// to get the opposite state.
		if ( ! apieditor->isOpen() )
		{
			apieditor->toggle();
		}
		else
		{
			// De-allocate the editor.
			removeEditor();
		}
	}
}

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be dynamicEditor_func.S, dynamicEditor_func.M
	hFunc.create( "dynamicEditor_func", do_dynamic_stuff_here );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.

	h.create( "DynamicEditor", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "DynamicEditor" );

	h.setIconPath( makeAltPath( dirName, NULL ) );

	// Indicate which menu to add the plugin to. 
	h.installOnMenu( "al_goto", FALSE /* top */ );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Dynamic Editor plug-in installed under the 'Utilities' menu.");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the api editor if it is still allocated.
	if ( apieditor != NULL )
		removeEditor();
		
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}

