/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
duplicatePolygons.plugin ( formerly jptDuplicatePolys.plugin )

    This plugin allows you to duplicate multiple edges/polygons in a 
	polyset.  It works similar to Extrude Together in 3Design.

	Make sure you don't pick nothing after you do a duplicate.  The newly
	created polygons/vertices must be manualy xformed because they overlap
	the old vertices.

	Installed under Polygon Edit palette.

Limitations:

	Extruding edges only creates double sided polygons.  Clean it
	using PolyEdit->Clean Polyset Single Side.

Version History:

	5/15/96:
		Finished the first version.


*/

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlDagNode.h>
#include <AlPolyset.h>
#include <AlPolysetNode.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <stdio.h>
#include <string.h>

#define MAX_OBJS 64
#define MAX_VERTS 64

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

class PolyData {
public:
int oldV1[MAX_VERTS], oldV2[MAX_VERTS], newV1[MAX_VERTS], newV2[MAX_VERTS];
int count;
PolyData()
{
	count = 0;
	int i;
	for ( i = 0; i < MAX_VERTS; i++ )
	{
		oldV1[i] = oldV2[i] = -1;
	}
}

void InsertData( int o1, int o2, int n1, int n2 )
{
	if ( !(count < MAX_VERTS) )
		return;
	oldV1[count] = o1;
	oldV2[count] = o2;
	newV1[count] = n1;
	newV2[count] = n2;
	count++;
}

};


//----------------------------------------------------------


/*
*
*/
static void jSwapVertices( AlPolygon *pgon, 
int old1, int old2, int new1, int new2  )
{
//	fprintf( stderr, "Swapping (%d, %d) to (%d, %d)\n", old1, old2, new1, new2 );

	int i, j, nv = pgon->numberOfVertices();
	int *vlist = new int[nv + 1];

	j = 0;
	for ( i = nv - 1; i >= 0; i-- )
	{
		AlPolysetVertex vtx;
		pgon->vertexD( i, vtx );
	//	fprintf( stderr, "vtx %d: %d\n", i, vtx.index() );

		if ( vtx.index() == old1 )
			vlist[j] = new1;
		else if ( vtx.index() == old2 )
			vlist[j] = new2;
		else
			vlist[j] = vtx.index();
		j++;
		pgon->removeVertex( i );
	}
//	fprintf( stderr, "Poly %d now has: \n", pgon->index() );
	for ( i = j-1; i >= 0; i-- )
	{
		pgon->addVertex( vlist[i] );
	//	fprintf( stderr, "	%d: \n", vlist[i] );
	}

    delete[] vlist;
}

/*
*
*/
static int jPolyPicked( AlPolygon *pgon )
{
	int i;

	for ( i = 0; i < pgon->numberOfVertices(); i++ )
	{
		AlPolysetVertex vtx;
		pgon->vertexD( i, vtx );
		if ( !vtx.isPicked() )
			return 0;
	}
	return 1;
}

/*
*
*/
static void jDuplicatePolys( AlPolyset *pset )
{
	int nv = pset->numberOfVertices();
	int np = pset->numberOfPolygons();

	int *vData = new int[nv];
	PolyData *pData = new PolyData[np];

	int i, j, lastPicked;
	double x, y, z;

	for ( i = 0; i < nv; i++ )
		vData[i] = -1;

	AlPolygon pgon;
	AlPolysetVertex vtx;

	for ( i = 0; i < np; i++ )	// check all polys
	{
		pset->polygonD( i, pgon );

		lastPicked = -999;
		int nVts = pgon.numberOfVertices();
		if ( jPolyPicked( &pgon ) )
			continue;
		for ( j = 0; j < nVts; j++ )	// create wall polys
		{
			pgon.vertexD( j, vtx );
			if ( vtx.isPicked() )
			{
				int doLast = 0;
				if ( j == 0 )
				{
					AlPolysetVertex lastV;
					pgon.vertexD( nVts - 1, lastV ); 
					if ( lastV.isPicked() )
					{
						lastPicked = nVts-1;
						doLast = 1;
					}
				}

				if ( (lastPicked == j - 1)  || doLast )	// got an edge
				{
					AlPolysetVertex *oldV1 = pgon.vertex( lastPicked );
					AlPolysetVertex *oldV2 = pgon.vertex( j );
					int oV1 = oldV1->index();
					int oV2 = oldV2->index();

					int nV1;
					if ( vData[ oV1 ] == -1 )	// no new vtx created
					{
						oldV1->unaffectedPosition( x, y, z );
						nV1 = pset->newVertex( x, y, z );
						vData [ oV1 ] = nV1;
					}
					else
						nV1 = vData[ oV1 ];

					int nV2;
					if ( vData[ oV2 ] == -1 )
					{
						oldV2->unaffectedPosition( x, y, z );
						nV2 = pset->newVertex( x, y, z );
						vData[ oV2 ] = nV2;
					}
					else
						nV2 = vData[ oV2 ];

					int newP = pset->newPolygon();
					AlPolygon * newPgn = pset->polygon( newP );

					newPgn->addVertex( nV2 );
					newPgn->addVertex( nV1 );
					newPgn->addVertex( oV1 );
					newPgn->addVertex( oV2 );
					delete newPgn;

					pData[ pgon.index() ].InsertData( oV1, oV2, nV1, nV2 );

				//	AlPolysetVertex newV1, newV2;

				//	pset->vertexD( nV1, newV1 );
				//	pset->vertexD( nV2, newV2 );

				//	newV1.pick();
				//	newV2.pick();
				//	oldV1->unpick();
				//	oldV2->unpick();

					delete oldV1;
					delete oldV2;
				}	// if lastPicked = j - 1

				else	// no edge with last vtx.  check next vtx
				{
					int nextV;
					if ( j == nv - 1 )
						nextV = 0;
					else
						nextV = j+1;

					AlPolysetVertex nVtx;
					pgon.vertexD( nextV, nVtx );
					if ( !nVtx.isPicked() )	// this picked vertex is not connected to any edges.  must create new vtx;
					{
						int oV1 = vtx.index();
						int nV1;
						if ( vData[ oV1 ] == -1 )
						{
							vtx.unaffectedPosition( x, y, z );
							nV1 = pset->newVertex( x, y, z );
							vData[ oV1 ] = nV1;
						}
						else
							nV1 = vData[ oV1 ];
						pData[ pgon.index() ].InsertData( oV1, -1, nV1, -1 );
					}
				}
				lastPicked = j;

			}	// if j = 0
		}	// else for
	}

	for ( i = 0; i < np; i++ )
	{
		if ( pData[i].count != 0 )
		{
			AlPolygon pgon;
			pset->polygonD( i, pgon );
			for ( j = 0; j < pData[i].count; j++ )
			{
			//	fprintf( stderr, "Processing polygon %d:\n", pgon.index() );
				jSwapVertices( &pgon, pData[i].oldV1[j], pData[i].oldV2[j], pData[i].newV1[j], pData[i].newV2[j] );
			}
		}
	}

	delete[] vData;
	delete[] pData;
	pset->calcNormals( TRUE );
}

/*
*
*/
static void do_it( )
{
 	int picked = 0;

	AlPickList::firstPickItem();
	if ( AlPickList::getObject() )
	{
/*
    	AlAnswerType answer;
    	AlPromptBox( kOK_Cancel, "Delete polygons?", &answer, 200, 200 );

    	if ( answer != kOK && answer != kYes )
        	return;
*/

		AlObject * obj;
		AlPolyset *newpset, *oldpset = NULL;
		AlPolysetVertex *v;
		int i;
		AlPolyset *psets[MAX_OBJS];

		do	// go through all picked objects
		{
			if ( obj = AlPickList::getObject() )
			{
				v = obj->asPolysetVertexPtr();

				if ( v )	// is a polyset vertex
				{
					newpset = v->polyset();

					if ( AlAreEqual( newpset, oldpset ) )
						delete newpset;
					else
					{
						oldpset = newpset;

						if ( picked < MAX_OBJS )
							psets[ picked++ ] = newpset;
						else
							return;
					}
				}
				delete obj;
			}

		} while ( AlPickList::nextPickItem() == sSuccess );

		for ( i = 0; i < picked; i++ )
		{

			AlPolysetNode* pnode;
			pnode = psets[i]->polysetNode();

		//	fprintf( stderr, "processing %s\n", pnode->name() );
			jDuplicatePolys( psets[i] );
		//	pnode->updateDrawInfo();
		//	pnode->doUpdates( TRUE );
			delete pnode;

			delete psets[i];
		}
	}

	if ( picked )
	{
	//	AlPickList::clearPickList();
		AlUniverse::redrawScreen();
	}
	else
	{
		fprintf( stderr, "\n" );
		AlPrintf( kPrompt, "Please select the vertices in the polygons you want to duplicate." );
	}
}



// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	// Invoke the scheme file which sets defaults for the scheme
	// variables.
	// AlInvokeSchemeFile( "wedge_init.scm" );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	hFunc.create( "pl_jptDuplicatePolys", do_it );

	h.create( "Duplicate Polygons", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "dup_polys" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV or an absolute path, and
	// the second argument is the name of the option box given
	// in the scheme file. If a call to this method is omitted
	// there will be no option box.
	// h.setOptionBox( "wedge.scm", "wedge.options" );

	h.setIconPath( makeAltPath( dirName, NULL ) );

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "al_polyedit" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.

    AlPrintf( kPrompt, "duplicatePolygons plug-in installed under 'Polygon Edit' palette");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()

	AlUniverse::redrawScreen();

	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}


