/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
deletePolygons.plugin ( formerly jptDeletePolys.plugin )

    This plugin allows you to delete selected polygons in a polyset.
	Polygons are selected by selecting the vertices in the polygons.

	Installed under Polygon Edit palette.

Limitations:

	Can only handle 32 different picked polysets at a time.

	Deletes vertices that don't belong to any polygon in a polyset.

Version History:

	4/22/95:
		Finished the first version.

*/

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlDagNode.h>
#include <AlPolyset.h>
#include <AlPolysetNode.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <stdio.h>
#include <string.h>

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

int *gActivePolys, *gActiveVerts;

#define MAX_OBJS 64

//----------------------------------------------------------

/*
*
*/
static int isPolygonActive ( AlPolygon *pgon )
{
    AlPolysetVertex *vert;
    int num_verts, v;
    int poly_picked;

    vert = NULL;
    poly_picked = FALSE;

    if ( AlIsValid ( pgon ) )
    {
        num_verts = pgon->numberOfVertices();
        poly_picked = TRUE;

        for ( v=0; v < num_verts; v++ )
        {
            vert = pgon->vertex( v );

            if ( AlIsValid ( vert ) )
            {
                if ( vert->isPicked() )
                {
				//	data->verts[ vert->index() ] = 1;
                    delete ( vert );
                    vert = NULL;

                    continue;
                }
                else
                {
                    delete ( vert );
                    vert = NULL;

                    poly_picked = FALSE;
                    break;
                }
            }
            else
            {
                poly_picked = FALSE;
                break;
            }
        }
    }

    return ( poly_picked );
}


/*
*
*/
static int deletePolygonsActive ( AlPolyset *pset )
{
    AlPolygon *pgon;
    int num_polys, p, deleted = 0;

    if ( AlIsValid ( pset ) )
    {
        num_polys = pset->numberOfPolygons();

        for ( p=num_polys - 1; p > -1; p-- )
        {
            pgon = pset->polygon( p );

            if ( AlIsValid ( pgon ) )
            {
                if ( isPolygonActive ( pgon ) )
                {
                    // the polygon at index 'p' in this polyset has all of
                    // it's vertices active, so we take it as "active"

					pset->deletePolygon( pgon->index() );

					deleted++;
                }

                delete ( pgon );
                pgon = NULL;
            }
        }

		AlPrintf( kPrompt, "%d polygons deleted from polyset.", deleted );
    }

	return deleted;
}


/*
*
*/
static void jCleanVertices( AlPolyset *pset )
{
	int nv = pset->numberOfVertices();
	int np = pset->numberOfPolygons();

	int i, j, *verts = new int[nv];

	for ( i = 0; i < nv; i++ )	// init to 0
		verts[i] = 0;


	AlPolygon pgon;
	AlPolysetVertex vtx;

	for ( i = 0; i < np; i++ )	// check all polys
	{
		pset->polygonD( i, pgon );

		for ( j = 0; j < pgon.numberOfVertices(); j++ )	// record vtx
		{
			pgon.vertexD( j, vtx );
			verts[ vtx.index() ] = 1;
		}
	}

	for ( i = nv - 1; i > -1; i-- )	// check verts list for bogus verts
		if ( verts[i] == 0 )	// not referenced by any polygon
			pset->deleteVertex( i );

	delete[] verts;
}



/*
*
*/
static void do_it( )
{
 	int picked = 0;

	AlPickList::firstPickItem();
	if ( AlPickList::getObject() )
	{
    	AlAnswerType answer;
    	AlPromptBox( kOK_Cancel, "Delete polygons?", &answer, 200, 200 );

    	if ( answer != kOK && answer != kYes )
        	return;

		AlObject * obj;
		AlPolyset *newpset, *oldpset = NULL;
		AlPolysetVertex *v;
		int i;
		AlPolyset *psets[MAX_OBJS];

		do	// go through all picked objects
		{
			if ( obj = AlPickList::getObject() )
			{
				v = obj->asPolysetVertexPtr();

				if ( v )	// is a polyset vertex
				{
					newpset = v->polyset();

					if ( AlAreEqual( newpset, oldpset ) )
						delete newpset;
					else
					{
						oldpset = newpset;

						if ( deletePolygonsActive( newpset ) )
						{
							if ( picked < MAX_OBJS )
								psets[ picked++ ] = newpset;
							else
								return;
						}
					}
				}
				delete obj;
			}

		} while ( AlPickList::nextPickItem() == sSuccess );

		for ( i = 0; i < picked; i++ )
		{
			jCleanVertices( psets[i] );

			AlPolysetNode* pnode;
			pnode = psets[i]->polysetNode();
			pnode->updateDrawInfo();
			// pnode->doUpdates( TRUE );
			delete pnode;

			delete psets[i];
		}
	}

	if ( picked )
	{
		AlPickList::clearPickList();
		AlUniverse::redrawScreen();
	}
	else
	{
		fprintf( stderr, "\n" );
		AlPrintf( kPrompt, "Please select the vertices in the polygons you want to delete." );
	}
}



// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	hFunc.create( "pl_jptDeletePolys", do_it );

	h.create( "Delete Polygons", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "del_polys" );

	h.setIconPath( makeAltPath( dirName, NULL ) );

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "al_polyedit" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.

    AlPrintf( kPrompt, "Delete Polygons plug-in installed under 'Polygon Edit' palette");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()

	AlUniverse::redrawScreen();

	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}


