/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlUniverse.h>
#include <AlDagNode.h>
#include <AlPickList.h>
#include <AlPickable.h>

char inbuf[ 256 ];
char dag_name[ 100 ];
const char * my_prompt = "Enter the name of the Dagnode to delete: %s";

static void my_delete_dag( void )
{
	AlObject * obj;
	statusCode code;

	for(;;) {
		code = AlPickList::firstPickItem();
		if( code != sSuccess ) {
			break;
		}
		obj = AlPickList::getObject();
		if( !obj ) return;
		AlPickable * pb = obj->asPickablePtr();
		if( !pb ) {
			// Something very weird has just happened.  If something is
			// on the Alias pick list but cannot be converted into an
			// AlPickable, it must be a type that LiveData does not yet 
			// support as AlPickable.
			delete obj;
			return;
		}
		// unpick the object before deleting it.  This is absolutely 
		// crucial at the present time, because the pick list is not
		// automatically updated.  
		pb->unpick();
		// Delete the object.  This may, of course, not actually work
		// for some types, i.e. LiveData does not let you delete things like
		// curveCVs.  But the call is still there.  It will just fail, 
		// that's all.
		obj->deleteObject();
		delete obj;
	}

	AlPickList::clearPickList();
	AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
}

void init_func( void )
{
	dag_name[ 0 ] = 0;
	AlPickList::clearPickList();
	AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
}

void down_func( int, Screencoord, Screencoord )
{}

void move_func( int input, Screencoord, Screencoord )
{
	int button;
	if( AlContinuousFunction::translateInput( input, button ) 
				== kInputKeyboard ) {
		sscanf( inbuf, "%s", dag_name );
		if( dag_name[ 0 ] ) {
			statusCode c = AlPickList::pickByName( dag_name );
			if( c != sSuccess ) {
				AlPrintf( kPrompt, "No such dagnode.  Please try again." );
				dag_name[ 0 ] = 0;
			} else {
				my_delete_dag();
			}
		}
	}
}

void up_func( int input, Screencoord x, Screencoord y )
{
	int button;
	if( AlContinuousFunction::translateInput( input, button )
				== kInputButton ) {
		statusCode c = AlPickList::pickFromScreen( x, y );
		if( c == sSuccess ) {
			my_delete_dag();
		}
	}
	AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
}

void cleanup_func()
{
}

const char * prompt_func( void )
{
	return my_prompt;
}

static AlFunctionHandle h;
static AlContinuousFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize();

	hFunc.create( "pl_DeleteDag", init_func, down_func, move_func, 
								up_func, cleanup_func, TRUE );
	hFunc.setPrompt( prompt_func, inbuf, kFilterLinear );
	h.create( "Delete Dagnode", &hFunc );
	h.setAttributeString( "deldag" );
	h.addToMenu( "mp_objtools" );
	h.setIconPath( makeAltPath( dirName, NULL ) );

	AlPrintf( kPrompt, "Delete dag installed on Palette 'Object Edit'");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	h.removeFromMenu();
	h.deleteObject();
	hFunc.deleteObject();

	return 0;
}
