/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	This is a sample OpenAlias plugin which demonstrates the use of the 
//	'eval' method in the AlSurface and the AlCurve methods.
//
//
//
//	It performs a simple uniform tesselation  by uniformly sampling the
//	surface or curve in world coordianates.  It creates a closed polygon
//	for a curve or uniform patches for the surface.
//
//	It tesselates the picked object and generates a copy in world coordinates
//
//	More error checking should be performed, but this is an illustrative
//	example only.
//

#include <AlUniverse.h>
#include <AlPickList.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlSurface.h>
#include <AlSurfaceNode.h>

#include <AlCurveNode.h>
#include <AlCurve.h>

#include <AlPolyset.h>
#include <AlPolysetNode.h>
#include <AlPolygon.h>

extern char * makeAltPath(const char *dirName, const char *suffix);

const double dScale = 20/100.0;	// 20% derivative length

void tesselateCurve( AlCurve *curve )
{
	AlPolyset *pset = new AlPolyset;
	pset->create();

	int polyIndex = pset->newPolygon();
	AlPolygon* poly = pset->polygon( polyIndex );

	AlCurveNode *dNode = new AlCurveNode;
	AlCurve *d = new AlCurve;

	for( double t = 0; t < 1; t += 0.1 )
	{
		double P[3], dP[3], P2[3];
		curve->eval( t, TRUE, P, dP );
		int index = pset->newVertex( P[0], P[1], P[2] );
		poly->addVertex( index );

		//	make an arrow pointing in the derivative direction
		P2[0] = P[0] + dP[0]*dScale;
		P2[1] = P[1] + dP[1]*dScale;
		P2[2] = P[2] + dP[2]*dScale;
		d->createLine( P, P2 );
		dNode->create( d );
	}
	delete poly;
	delete pset;
	delete d;
	delete dNode;
}

void tesselateSurface( AlSurface* surf )
{
	AlPolyset *pset = new AlPolyset;
	pset->create();

	double xMin, xMax;
	double yMin, yMax;

	//	Determine the lower/upper knot values
	{
		int numU = surf->uNumberOfKnots();
		double *uVector = new double[ numU ];

		surf->uKnotVector( uVector );
		xMin = uVector[0];
		xMax = uVector[numU-1];
		delete[] uVector;

		int numV = surf->vNumberOfKnots();
		double *vVector = new double[ numV ];

		surf->vKnotVector( vVector );
		yMin = vVector[0];
		yMax = vVector[numV-1];
		delete[] vVector;
	}

	double xStep = 0.5, yStep = 0.5;
	int xSize = (int)((xMax-xMin)/xStep)+1;
	int ySize = (int)((yMax-yMin)/yStep)+1;

	int *indecies = new int[ xSize * ySize ];

	AlCurveNode *dNode = new AlCurveNode;
	AlCurve *d = new AlCurve;

	// Generate the vertices
	//
	{
		int *linePtr = indecies;
		for( double y =yMin; y < yMax; y += yStep )
		{
			int *ptr = linePtr;
			for( double x =xMin; x < xMax; x += xStep )
			{
				double P[3], n[3], P2[3];
				surf->eval( x, y, TRUE, P, NULL, NULL, n );
				*ptr = pset->newVertex( P[0], P[1], P[2] );
	
				//	make an arrow pointing in the derivative direction
				P2[0] = P[0] + n[0]*dScale;
				P2[1] = P[1] + n[1]*dScale;
				P2[2] = P[2] + n[2]*dScale;
				
				d->createLine( P, P2 );
				dNode->create( d );
	
				ptr++;
			}
			linePtr += xSize;
		}
	}
	//
	//	Now tile the matrix of points
	//
	{
		int *linePtr = indecies;
		for( int y=0; y < ySize-1; y++ )
		{
			int *ptr = linePtr;
			for( int x =0 ; x < xSize-1; x++ )
			{
				int polyIndex = pset->newPolygon();
				AlPolygon* poly = pset->polygon( polyIndex );
				if( poly )
				{
					poly->addVertex( ptr[0] );
					poly->addVertex( ptr[1] );
					poly->addVertex( ptr[xSize+1] );
					poly->addVertex( ptr[xSize] );
					delete poly;
				}
				ptr++;
			}
			linePtr += xSize;
		}
	}

	delete[] indecies;
	delete pset;
	delete d;
	delete dNode;
}

void doTesselate()
{
	AlDagNode *dag = AlUniverse::firstDagNode();

	AlObject *obj;
	AlPickList::firstPickItem ();;
	obj = AlPickList::getObject();
	if( obj )
	{
		if( obj->asCurveNodePtr() )
		{
			AlCurve *curve = obj->asCurveNodePtr()->curve();
			tesselateCurve( curve );
			delete curve;
		}
		else if( obj->asSurfaceNodePtr() )
		{
			AlSurface *surf = obj->asSurfaceNodePtr()->surface();
			tesselateSurface( surf );
			delete surf;
		}
		else
			AlPrintf( kPrompt, "Tesselate: nothing selected, just deleteAll");
		delete obj;
	}
	AlUniverse::redrawScreen();
}

// Our instance of the function handle
//
static AlFunctionHandle handle;
static AlMomentaryFunction	tesslateFunc;

extern "C" 
PLUGINAPI_DECL int plugin_init (const char *dirName)
{
	// Initialize OpenAlias
	//
	AlUniverse::initialize (kYUp);

	// Establish ourselves as a OpenAlias plug-in
	//
	tesslateFunc.create (doTesselate);
	handle.create ("UniformTesselate", &tesslateFunc);
	handle.setAttributeString ("tesselate");

	// Attach our option box
	//
	handle.setIconPath( makeAltPath( dirName, NULL ) );
	handle.appendToMenu ("mp_objtools");
	return (0);
}

extern "C" 
PLUGINAPI_DECL int plugin_exit (void)
{
	handle.deleteObject ();
	tesslateFunc.deleteObject();
	return (0);
}

