/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

// cylinder based tree
//
#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>
#include <AlIterator.h>
#include <AlCurve.h>
#include <AlCurveNode.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>
#include <AlGroupNode.h>

#include <AlTM.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <math.h>
#include <string.h>

#include <AlVector.h>

//
//	Version 2 of the tree generator.  This version uses cylinders for
//	the branches instead of polygons.
//	It demonstrates the use of the surface creation functions.
//

#define INIT_OPTIONBOX_FILE	"createTree_init.scm"
#define OPTIONBOX_FILE		"createTree.scm"

#ifdef _WIN32
#define M_PI 3.145926535
#define srandom srand
#define random rand
#endif

typedef enum {kLEAF_NONE, kLEAF_OAK, kLEAF_MAPLE, kLEAF_PINE } LeafType;

class TreeOutput;

class Tree
{
public:
	int seed;
	int max_branch;
	double bend;
	double fork_angle;
	double branch_frequency;
	double gravity;
	double branch_length;
	double trunk_taper;
	double min_branch_size;
	double twist;

	AlGroupNode * parent;

public:
	Tree();
	virtual	~Tree();

	statusCode branch(		TreeOutput &out,
						  	const AlVector& center,
                            const AlVector& x,
                            const AlVector& y,
                            const AlVector& z,
                            double thickness, int branch_level ) const;

	statusCode extendtrunk(	TreeOutput &out,
							const AlVector &base,
							const AlVector &x_basis,
							const AlVector &y_basis,
							double thickness, int branch_level ) const;

	statusCode add_leaf(	TreeOutput &out,
							const AlVector& center,
                            const AlVector& x,
							const AlVector& y,
							const AlVector& z,
                            double thickness ) const;

	statusCode plant(		TreeOutput &out,
							const AlVector &base, double thickness ) const;

	// routine to fetch the parameters
	statusCode getParams();
};

class TreeOutput
{
public:
	virtual ~TreeOutput();

	virtual statusCode trunk(	const AlVector &pos1, const AlVector &x1, const AlVector &y1, double rad_1, const AlVector &pos2, const AlVector &x2, const AlVector &y2, double rad_2, AlGroupNode * parent );
protected:
	TreeOutput();
};

class TreeOutputCylinder : public TreeOutput
{
public:
	virtual statusCode trunk( 	const AlVector &pos1,
								const AlVector &x1, const AlVector &y1,
								double rad_1,
								const AlVector &pos2,
								const AlVector &x2, const AlVector &y2,
								double rad_2,
								AlGroupNode * parent );
};

TreeOutput::TreeOutput()	{}
TreeOutput::~TreeOutput()	{}
statusCode TreeOutput::trunk(
			 const AlVector&, const AlVector&, const AlVector&, double,
			 const AlVector&, const AlVector&, const AlVector&, double,
			 AlGroupNode * )
{
	return sFailure;
}


static inline double DegreesToRads( double degrees )
{
	return ((double(degrees)) * M_PI / 180.0);
}

Tree::Tree()
{
	seed			= 0;
	max_branch		= 15;	// this should be stopped by the taper ..
							// but just in case ...
	bend			= 0.2;
	fork_angle	= DegreesToRads( 30 );
	branch_frequency	= 0.5;
	gravity			= -0.1;
	branch_length	= 5;
	trunk_taper		= 0.95;
	min_branch_size	= 0.025;
	twist 			= DegreesToRads( 45 );

	parent = new AlGroupNode;
	parent->create();
	parent->setName( "tree" );
}

Tree::~Tree()
{
	delete parent;
}

statusCode Tree::getParams()
{
	if( sSuccess == AlGetDouble( "gravity",				gravity ) &&
		sSuccess == AlGetDouble( "twist",				twist ) &&
		sSuccess == AlGetDouble( "bend",				bend ) &&
		sSuccess == AlGetDouble( "trunk_taper",			trunk_taper ) &&
		sSuccess == AlGetDouble( "branch_length",		branch_length ) &&
		sSuccess == AlGetDouble( "branch_frequency",	branch_frequency ) &&
		sSuccess == AlGetDouble( "fork_angle",		fork_angle ) &&
		sSuccess == AlGetDouble( "min_branch_size",	min_branch_size ))
	{
		if( sSuccess == AlGetInteger( "seed", seed ) &&
			sSuccess == AlGetInteger( "max_branch", max_branch ))
		{
			// convert to radians
			twist = DegreesToRads( twist );
			fork_angle = DegreesToRads( fork_angle );
			branch_frequency /= 100.0;
			trunk_taper /= 100.0;
			return sSuccess;
		}
	}
	else
		AlPrintf( kPrompt, "Invalid size parameter!");
	return sFailure;
}

// random low..high
static inline double rangeRand(double low, double high)
{
	double r = (double)random() / (double)0x7fffffff;
	return r* ( high-low) + low; 
}

//
//	Rotate on the z axis relative to a pair of orthogonal x/y vectors
//
static void RotateOnZ_xyBasis(	AlVector& x1, AlVector& y1,
								const AlVector& basis_x,
								const AlVector& basis_y,
								double angle )
{
	// there is a lot of hidden matrix math here..
	x1 = basis_x *  cos( angle ) + basis_y * sin( angle );
	y1 = basis_x * -sin( angle ) + basis_y * cos( angle ); 
}

//
// x-y - vectors in trunk plane
// z - directional vector of the trunk
// center - location of new growth ring
//
//
//

statusCode Tree::add_leaf(	TreeOutput&,
							const AlVector&,
							const AlVector&,
							const AlVector&,
							const AlVector&,
							double ) const
{
	// add this branch (clone 'trunk it' )
	// put needles at random positions in x-y plane
	// at a random between center & center_top
	return sSuccess;
}

statusCode Tree::branch(	TreeOutput &out,
							const AlVector& center,
							const AlVector& x,
							const AlVector& y,
							const AlVector& z,
							double thickness, int branch_level ) const
{
	// hit the max branch level ... oops
	if( branch_level == 0 )
		return sSuccess;

	// preserve wood at each branch 
	thickness *= sqrt( 0.5 );

	// fork one way
	AlVector left_x, left_y, left_center;
	AlVector right_x, right_y, right_center;
	AlVector left_vx, right_vx, vy;

	// calculate the centers
	RotateOnZ_xyBasis( left_vx, vy, x, z, -fork_angle );
	RotateOnZ_xyBasis( right_vx, vy, x, z, fork_angle );
	left_center  = center + left_vx * (thickness / 2);
	right_center = center - right_vx * (thickness / 2);

	// recalculate the directional vectors for the branches
	left_x = -y;
	left_y = left_vx;

	right_x = right_vx;
	right_y = y;

	statusCode stat;

	// extend left
	stat = extendtrunk( out, left_center, left_x, left_y,
						thickness, branch_level);
	if( stat != sSuccess )
		return stat;

	// extend right
	stat = extendtrunk( out, right_center, right_x, right_y,
						thickness, branch_level);
	return stat;
}

statusCode Tree::extendtrunk(	TreeOutput &out,
								const AlVector &base,
								const AlVector &x_basis,
								const AlVector &y_basis,
								double thickness, int branch_level ) const
{
	AlVector x, y, z;
	const AlVector down(0.0, 0.0, -1.0 );

	if ( AlEscapeKeyPressed() )
		return sFailure;

	// rotate by 90 degrees ?
	if( rangeRand( 0.0, 1.0 ) < 0.5 )
	{
		// new x/y basis is a rotated 90 version of the old one
		RotateOnZ_xyBasis( x, y, x_basis, y_basis, DegreesToRads( 90.0 ) );
	}
	else	// no 90 deg twist
	{
		x = x_basis;
		y = y_basis;
	} 
	z = x ^ y;

	// old growth axises
	x = x.norm();
	y = y.norm();
	z = z.norm();
	
	z += down * rangeRand( 0.0, gravity );
	z = z.norm();

	// twist it
	RotateOnZ_xyBasis( x, y, x, y, rangeRand( 0.0, twist ));

	// add random bend 
	z.x += bend * rangeRand( -1.0, 1.0 );
	z.y += bend * rangeRand( -1.0, 1.0 );
	z.z += bend * rangeRand( -1.0, 1.0 );

	// calc the basis vectors perpendicular to the growth direction
	y = z ^ x;
	x = y ^ z;

	// normalize the vectors
	x = x.norm();
	y = y.norm();
	z = z.norm();

	// middle of new growth ring
	AlVector center = base + z * (branch_length * thickness);

	// taper the thickness
	double new_thickness = thickness * trunk_taper;
	if( thickness <= min_branch_size )
	{
		return add_leaf( out, center, x, y, z, new_thickness );
	}

	out.trunk(	base, x_basis, y_basis, thickness,
				center, x, y, new_thickness, parent );

	// decide if we fork
	if( (new_thickness > 2 * min_branch_size) &&
		(rangeRand( 0.0, 1.0 ) < branch_frequency ) )
	{
		return branch( out, center, x, y, z, new_thickness, branch_level-1);
	}
	else	// no fork -> extend the current trunk
	{	
		// orthogonal vectors outward from top of trunk
		return extendtrunk( out, center, x, y, new_thickness, branch_level );
	}
}

statusCode Tree::plant(	TreeOutput &out,
						const AlVector &base, double thickness ) const
{
	// direction vectors for our tree plane
	// they also determine the thickness
	//
	AlVector x_basis( -1, 0, 0 );
	AlVector y_basis( 0, -1, 0 );

	// create the initial four corners
	// start off the random number generator
	srandom( seed );
	
	// start off the main area
	return extendtrunk( out, base, x_basis, y_basis, thickness, max_branch );
}

boolean makePerpBasis( const AlVector& v1,
						AlVector &w1, AlVector &w2, AlVector &w3 )
//
//	Given a vector, find an orthonormal basis with the first vector
//	being a unit vector of the given vector
//
{
	AlVector v2, v3;

	// figure out two vectors colinear to v1
	if( v1.x != 0 )
	{
		v2 = AlVector( 0,1,0);
		v3 = AlVector( 0,0,1);
	}
	else if( v1.y != 0 )
	{
		v2 = AlVector( 1,0,0);
		v3 = AlVector( 0,0,1);
	}
	else if( v1.z != 0 )
	{
		v2 = AlVector( 1,0,0);
		v3 = AlVector( 0,1,0);
	}
	else
		return  FALSE;

	// Gram-Schmit 'em
	w1 = (v1							).norm();
	w2 = (v2 - (v2*w1)*w1				).norm();  
	w3 = (v3 - (v3*w1)*w1 - (v3*w2)*w2	).norm();
	return TRUE;
}

boolean findPerpVector( const AlVector& in, AlVector &out )
//
//	Given a vector, find an orthonormal basis with the first vector
//	being a unit vector of the given vector
//
{
	AlVector w1 = in.norm();
	AlVector v2;

	// figure out two vectors colinear to b1
	if( w1.x != 0 )
	{
		// v2 = (0,1,0)
		v2 = AlVector( 0,1,0);
	}
	else if( w1.y != 0 )
	{
		v2 = AlVector( 0,0,1);
	}
	else if( w1.z != 0 )
	{
		v2 = AlVector( 1,0,0);
	}
	else
		return  FALSE;

	// Gram-Schmit 'em
	out = v2 - (v2*w1)*w1;  
	return TRUE;
}

statusCode makeCylinder(	AlSurface *surf,
							const AlVector& pos1, double rad_1,
							const AlVector& pos2, double rad_2 )
{
	AlVector dir = pos2 - pos1;
	AlVector perp;
	if( findPerpVector( dir, perp ) == FALSE )
		return sFailure;
	
	// normalize
	perp = perp.norm();

	// produce a line instead for now ... 
	AlVector a = pos1 + perp * rad_1;
	AlVector b = pos2 + perp * rad_2;

	double a_pt[3], b_pt[3];
	a.put( a_pt );
	b.put( b_pt );

	AlCurve line;
	line.createLine( a_pt, b_pt );

	// now create a surface
	double start[3], end[3];
	pos1.put( start );
	pos2.put( end );

	statusCode stat = surf->createRevolvedSurface(  start, end, 0, 360, &line );
	line.deleteObject();
	return stat;
}

statusCode TreeOutputCylinder::trunk(	
	const AlVector& pos1, const AlVector &, const AlVector &, double rad_1,
	const AlVector& pos2, const AlVector &, const AlVector &, double rad_2,
	AlGroupNode * parent )
{
	AlSurface surf;
    AlSurfaceNode surfNode;
	if( makeCylinder( &surf, pos1, rad_1, pos2, rad_2 ) == sSuccess )
	{
	    statusCode code = surfNode.create( &surf );
		if( code == sSuccess ) { 
			parent->addChildNode( &surfNode );
			return sSuccess;
		} 
		return code;
	}
	else
		return sFailure;
}

static void do_tree()
{
	Tree tree;
	TreeOutputCylinder cylinder;

	AlPrintf( kPrompt, "Fractal Tree begin");
	if( tree.getParams() == sSuccess )
	{
		AlVector position( 0, 0, 0 );
		double start_thickness = 1;

		if ( tree.plant( cylinder, position, start_thickness ) == sSuccess )
		{
			AlPrintf( kPrompt, "Fractal completed");
		}
		else
		{
			AlPrintf( kPrompt, "Fractal not completed");
		}

		// Compress the group node that holds the tree in order to
		// avoid clogging the SBD window with hundreds of dagnodes.
        tree.parent->setDisplayMode( kDisplayModeCompressedSbd, TRUE);

		AlUniverse::redrawScreen();
	}
}

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
	char *dirScm;

	AlUniverse::initialize( kZUp );

	dirScm = makeAltPath(dirName,"scm");

	AlInvokeSchemeFile( INIT_OPTIONBOX_FILE, dirScm );

	hFunc.create( "pl_FractalTree2", do_tree );
	h.create( "Create Tree - Fractal", &hFunc );
	h.setAttributeString( "tree" );

	if ( sSuccess != h.setOptionBox( OPTIONBOX_FILE, "tree.options", dirScm) ) {
		AlPrintf( kPrompt, "createTree2 plug-in unable to find .scm file for option box" );
		return 1;
	}
	h.setIconPath( makeAltPath( dirName, NULL ) );
	h.appendToMenu( "mp_objtools" );
	
    AlPrintf( kPrompt, "createTree2 installed on Palette 'Object Edit'");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	h.removeFromMenu();
	h.deleteObject();
	hFunc.deleteObject();

	// do nothing
	return 0;
}

