/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>
#include <AlIterator.h>

#include <AlTM.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <math.h>

#include <AlVector.h>

#include <string.h>

#ifndef M_PI
#define M_PI 3.141592653589
#define srandom srand
#define random rand
#endif

//
//	This plugin implements a polygonal based binary fractal tree generator.
//	The leaves are left to as an exercise for the reader.
//
//	This demonstrates the creation of polysets and the use of simple shaders.
//
//

#define INIT_OPTIONBOX_FILE	"createTree_init.scm"
#define OPTIONBOX_FILE		"createTree.scm"

typedef enum {kLEAF_NONE, kLEAF_OAK, kLEAF_MAPLE, kLEAF_PINE } LeafType;

class Tree
{
public:
	int seed;
	int max_branch;
	double bend;
	double fork_angle;
	double branch_frequency;
	double gravity;
	double branch_length;
	double trunk_taper;
	double min_branch_size;
	double twist;

public:
	Tree();

	statusCode add_leaf(	AlPolyset *polyset, int v[4],
						AlVector& center,
						AlVector& x, AlVector& y, AlVector& z,
						double thickness ) const;
	statusCode branch(	AlPolyset *polyset, int v[4],
						AlVector& center,
						AlVector& x, AlVector& y, AlVector& z,
						double thickness, int branch_level  ) const;
	statusCode extendtrunk(	AlPolyset *polyset, int v[4], int branch_level ) const;
	statusCode plant( AlPolyset *polyset ) const;

	// routine to fetch the parameters
	statusCode getParams();
};

static inline double DegreesToRads( double degrees )
{
	return degrees * M_PI / 180.0;
}

Tree::Tree()
{
	seed			= 0;
	max_branch		= 15;	// this should be stopped by the taper ..
							// but just in case ...
	bend			= 0.2;
	fork_angle	= DegreesToRads( 30 );
	branch_frequency	= 0.5;
	gravity			= -0.1;
	branch_length	= 5;
	trunk_taper		= 0.95;
	min_branch_size	= 0.025;
	twist 			= DegreesToRads( 45 );
}

statusCode Tree::getParams()
{
	if( sSuccess == AlGetDouble( "gravity",				gravity ) &&
		sSuccess == AlGetDouble( "twist",				twist ) &&
		sSuccess == AlGetDouble( "bend",				bend ) &&
		sSuccess == AlGetDouble( "trunk_taper",			trunk_taper ) &&
		sSuccess == AlGetDouble( "branch_length",		branch_length ) &&
		sSuccess == AlGetDouble( "branch_frequency",	branch_frequency ) &&
		sSuccess == AlGetDouble( "fork_angle",		fork_angle ) &&
		sSuccess == AlGetDouble( "min_branch_size",	min_branch_size ))
	{
		if( sSuccess == AlGetInteger( "seed", seed ) &&
			sSuccess == AlGetInteger( "max_branch", max_branch ))
		{
			// convert to radians
			twist = DegreesToRads( twist );
			fork_angle = DegreesToRads( fork_angle );
			branch_frequency /= 100.0;
			trunk_taper /= 100.0;
			return sSuccess;
		}
	}
	else
		AlPrintf( kPrompt, "Invalid size parameter!");
	return sFailure;
}

// random low..high
static inline double rangeRand(double low, double high)
{
	double r = (double)random() / (double)0x7fffffff;
	return r* ( high-low) + low; 
}

//
// builds a quadrateral
//

static int addQuad( AlPolyset *polyset, int v1, int v2, int v3, int v4 )
{
	int polygon_index = polyset->newPolygon();

	if( polygon_index != -1 )
	{
		AlPolygon *polygon = polyset->polygon( polygon_index );
		if( polygon )
		{
			polygon->addVertex( v1 );
			polygon->addVertex( v2 );
			polygon->addVertex( v3 );
			polygon->addVertex( v4 );

			delete polygon;
		}
	}
	else
		AlPrintf( kPrompt, "Error: can create polygon");
	return polygon_index;
}

static inline int addQuadRev( AlPolyset *polyset, int v1, int v2, int v3, int v4 )
{
	return addQuad( polyset, v4, v3, v2, v1 );
}

static int addTri( AlPolyset *polyset, int v1, int v2, int v3 )
{
	int polygon_index = polyset->newPolygon();

	if( polygon_index != -1 )
	{
		AlPolygon *polygon = polyset->polygon( polygon_index );
		if( polygon )
		{
			polygon->addVertex( v1 );
			polygon->addVertex( v2 );
			polygon->addVertex( v3 );

			delete polygon;
		}
	}
	else
		AlPrintf( kPrompt, "Error: can create polygon");
	return polygon_index;
}

//
//	Rotate on the z axis relative to a pair of orthogonal x/y vectors
//
static void RotateOnZ_xyBasis(	AlVector& x1, AlVector& y1,
								const AlVector& basis_x,
								const AlVector& basis_y,
								double angle )
{
	// there is a lot of hidden matrix math here..
	x1 = basis_x *  cos( angle ) + basis_y * sin( angle );
	y1 = basis_x * -sin( angle ) + basis_y * cos( angle ); 
}

//
// x-y - vectors in trunk plane
// z - directional vector of the trunk
// center - location of new growth ring
//
//
//

statusCode Tree::add_leaf(	AlPolyset *polyset, int trunkBase[4],
							AlVector& center,
							AlVector& x, AlVector& y, AlVector& z,
							double thickness ) const
{
	// add this branch (clone 'trunk it' )
	// put needles at random positions in x-y plane
	// at a random between center & center_top
	//	AlVector center_top = center + z * (branch_length * thickness);


	//	Really boring for now... just cap the end of the branch
	if( addQuad( polyset,	trunkBase[0], trunkBase[1],
							trunkBase[2], trunkBase[3] ) != -1 )
		return sSuccess;
	else
		return sFailure;
}

statusCode Tree::branch(	AlPolyset *polyset, int trunk_base[4],
							AlVector& center,
							AlVector& x, AlVector& y, AlVector& z,
							double thickness, int branch_level ) const
{
	int trunk_left[4], trunk_right[4];

	AlVector pos[4];
	AlVector x1, y1, z1, t;

	if( branch_level == 0 )
	{
		if( -1 != addQuadRev( polyset, trunk_base[0], trunk_base[1], trunk_base[2], trunk_base[3] ) )
			return sSuccess;
		else
			return sFailure;
	}

	// preserve wood at each branch 
	thickness *= sqrt( 0.5 );

	// fork one way
	RotateOnZ_xyBasis( x1, y1, x, z, -fork_angle );

	t = y * (0.5 * thickness); 
	pos[0] = center + t;
	pos[1] = center - t;

	t = x1 * thickness;
	pos[3] = pos[0] + t;
	pos[2] = pos[1] + t;

	int i;
	for( i = 0; i< 4; i++)
	{
		trunk_left[i] = polyset->newVertex( pos[i].x, pos[i].y, pos[i].z );
	}

	// fork the other way
	RotateOnZ_xyBasis( x1, z1, x, z, fork_angle );
	
	trunk_right[0] = trunk_left[1];
	trunk_right[1] = trunk_left[0]; 

	t = x1 * -thickness;
	pos[2] = pos[0] + t;
	pos[3] = pos[1] + t;

	for( i = 2; i < 4; i++ )
	{
		trunk_right[i] = polyset->newVertex( pos[i].x, pos[i].y, pos[i].z );
	}

	// tube for trunk
	int quad1 = addQuadRev( polyset,	trunk_base[2],	trunk_left[3],
										trunk_left[2],	trunk_base[1] );
	int quad2 = addQuadRev( polyset,	trunk_base[1],	trunk_left[2],
										trunk_right[3],	trunk_base[0] );
	int quad3 = addQuadRev( polyset,	trunk_base[0],	trunk_right[3],
										trunk_right[2],	trunk_base[3]);
	int quad4 = addQuadRev( polyset,	trunk_base[3],	trunk_right[2],
										trunk_left[3],	trunk_base[2] );

	// fill in sides of the peak
	int tri7 = addTri( polyset,	trunk_right[2],	trunk_left[3],	trunk_left[0]);
	int tri8 = addTri( polyset,	trunk_left[2],	trunk_right[3],	trunk_right[0]);

	if( quad1 != -1 && quad2 != -1 && quad3 != -1 && quad4 != -1 &&
		tri7  != -1 && tri8  != -1 )
	{
		if( sSuccess == extendtrunk( polyset, trunk_left, branch_level ) &&
			sSuccess == extendtrunk( polyset, trunk_right, branch_level ))
		{
			//AlUniverse::redrawScreen();
			return sSuccess;
		}
	}
	return sFailure;
}

statusCode Tree::extendtrunk( AlPolyset *polyset, int trunk[4], int branch_level ) const
{
	AlVector x, y, z;
	AlVector trunkPos[4];
	
	double thickness;
	AlVector down(0.0, 0.0, -1.0 );

	if ( AlEscapeKeyPressed() )
		return sFailure;	

	// rotate by 90 degrees ?
	if( rangeRand( 0.0, 1.0 ) < 0.5 )
	{
		int temp = trunk[0];
		trunk[0] = trunk[1];
		trunk[1] = trunk[2];
		trunk[2] = trunk[3];
		trunk[3] = temp;
	} 

	// get the positions from the indices 
	AlPolysetVertex *pVertex = new AlPolysetVertex;
	for ( int i = 0; i < 4; i++ )
	{
		polyset->vertexD( trunk[i], *pVertex );
		if ( pVertex )
			pVertex->unaffectedPosition(	trunkPos[i].x,
											trunkPos[i].y,
											trunkPos[i].z );
	}
	delete pVertex;

	x = trunkPos[1] - trunkPos[0];
	thickness = x.size();

	y = trunkPos[3] - trunkPos[0];
	z = x ^ y;

	// old growth axises
	x = x.norm();
	y = y.norm();
	z = z.norm();
	
	z += down * rangeRand( 0.0, gravity );
	z = z.norm();

	RotateOnZ_xyBasis( x, y, x, y, rangeRand( 0.0, twist ));

	// add random bend 
	z.x += bend * rangeRand( -1.0, 1.0 );
	z.y += bend * rangeRand( -1.0, 1.0 );
	z.z += bend * rangeRand( -1.0, 1.0 );

	// calc the basis vectors perpendicular to the growth direction
	y = z ^ x;
	x = y ^ z;

	// normalize the vectors
	x = x.norm();
	y = y.norm();
	z = z.norm();

	// calculate the middle of the old growth ring
	AlVector center = (trunkPos[0] + trunkPos[1] + trunkPos[2] + trunkPos[3] )
						/ 4;

	if( thickness <= min_branch_size )
	{
		return add_leaf( polyset, trunk, center, x, y, z, thickness );
	}

	// middle of new growth ring
	center += z * (branch_length * thickness);

	// taper the thickness
	thickness *= trunk_taper;

	// decide if we fork
	if( (thickness > 2 * min_branch_size) &&
		(rangeRand( 0.0, 1.0 ) < branch_frequency ) )
	{
		return branch( polyset, trunk, center, x, y, z, thickness,
							branch_level-1);
	}
	else	// no fork -> extend the current trunk
	{	
		// orthogonal vectors outward from top of trunk
		AlVector dx = x * ( thickness / 2 );
		AlVector dy = y * ( thickness / 2 );

		// figure out the new four corners about the center
		trunkPos[0] = center - dx - dy;
		trunkPos[1] = center + dx - dy;
		trunkPos[2] = center + dx + dy;
		trunkPos[3] = center - dx + dy;

		int trunk_top[4];

		for( int i = 0; i< 4; i++)
		{
			trunk_top[i] = polyset->newVertex(	trunkPos[i].x, trunkPos[i].y,
												trunkPos[i].z);
		}
	
		// build the box trunk
		addQuad( polyset,	trunk[0],		trunk_top[0],
										trunk_top[1],	trunk[1]);
		addQuad( polyset,	trunk[1],		trunk_top[1],
										trunk_top[2],	trunk[2]);
		addQuad( polyset,	trunk[2],		trunk_top[2],
										trunk_top[3],	trunk[3]);
		addQuad( polyset,	trunk[3],		trunk_top[3],
										trunk_top[0],	trunk[0]);
		return extendtrunk( polyset, trunk_top, branch_level );
	}
}

statusCode Tree::plant( AlPolyset *polyset ) const
{
	int base[4];
	// create the initial four corners
	
	base[0] = polyset->newVertex(  0.5,  0.5,  0.0 );
	base[1] = polyset->newVertex( -0.5,  0.5,  0.0 );
	base[2] = polyset->newVertex( -0.5, -0.5,  0.0 );
	base[3] = polyset->newVertex(  0.5, -0.5,  0.0 );

	if( base[3] != -1 )
	{
		// start off the random number generator
		srandom( seed );
	
		addQuad( polyset, base[0], base[1], base[2], base[3] );
		// start off the main area
		return extendtrunk( polyset, base, max_branch );
	}
	else
	{
		AlPrintf( kPrompt, "Error: can't create initial points");
		return sFailure;
	}
}

static void do_tree()
{
	Tree tree;

	AlPrintf( kPrompt, "Fractal Tree begin");
	if( tree.getParams() == sSuccess )
	{
		AlPolyset polyset;

		if( polyset.create() != sSuccess )
		{
			AlPrintf( kPrompt, "Couldn't create polyset");
		}
		else
		{
			if ( tree.plant( &polyset ) == sSuccess )
			{
				AlPrintf( kPrompt, "Fractal completed");
			}
			else
			{
				AlPrintf( kPrompt, "Fractal not completed");
			}
		}

		AlUniverse::redrawScreen();
	}
}

static AlFunctionHandle h;
static AlMomentaryFunction hFunc; 

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
	char *dirScm;
		
	AlUniverse::initialize( kZUp );

	dirScm = makeAltPath(dirName,"scm");

	AlInvokeSchemeFile( INIT_OPTIONBOX_FILE, dirScm );

	hFunc.create( "pl_FractalTree", do_tree );
	h.create( "Create Tree - Polys", &hFunc ); 
	h.setAttributeString( "tree" );

	if ( sSuccess != h.setOptionBox( OPTIONBOX_FILE, "tree.options", dirScm ) ) {
		AlPrintf( kPrompt, "createTree plug-in unable to find .scm file for option box" );
		return 1;
	}
	h.setIconPath( makeAltPath( dirName, NULL ) );
	h.appendToMenu( "mp_objtools" );

	AlPrintf( kPrompt, "createTree installed on Palette 'Object Edit'");
	
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	h.removeFromMenu();
	h.deleteObject();
	hFunc.deleteObject();

	// do nothing
	return 0;
}


		
