/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>
#include <AlIterator.h>
#include <AlShader.h>
#include <AlTexture.h>

#include <AlTM.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <math.h>
#include <string.h>
#include <AlVector.h>

#define INIT_OPTIONBOX_FILE	"createSurface_init.scm"
#define OPTIONBOX_FILE		"createSurface.scm"

#ifdef _WIN32
#define srandom srand
#define random rand
#endif

class FractParam
{ 
public:
	statusCode get();

public:
	int seed;
	int subdivisions;
	double taper;
	double sizeX, sizeY; 
	AlVector variance;

	double level1, level2, level3;
	double level_jitter;
};

class FractSurf
{
public:
	FractSurf( const FractParam &param );
	~FractSurf();

	int operator !() const;

public:
	int size, size_x, size_y;

	AlVector *array;
};

enum ShaderIndex { kIndexSnow = 1, kIndexRock, kIndexGrass, kIndexSand};

FractSurf::~FractSurf()
{
	if( array ) {
		delete [] array;
		array = NULL;
	}
}

int FractSurf::operator !() const
{
	return array == NULL;
}

// random 0..1
inline double myRandPos()
{
	return( (double)random() / (double)0x7fffffff);
}

// random -1..1
inline double myRandPosNeg()
{
	return( myRandPos()*2 - 1);
}

inline void perturb( AlVector &pt, const AlVector &var )
{
	pt.x += myRandPosNeg()*var.x;
	pt.y += myRandPosNeg()*var.y;
	pt.z += myRandPosNeg()*var.z;
}

inline void avgPerturb( AlVector *out, const AlVector &var,
						const AlVector *v1, const AlVector *v2 )
{
	out->x = (v1->x + v2->x ) / 2 + myRandPosNeg()*var.x;
	out->y = (v1->y + v2->y ) / 2 + myRandPosNeg()*var.y;
	out->z = (v1->z + v2->z ) / 2 + myRandPosNeg()*var.z;
}

inline void avgPerturb( AlVector *out, const AlVector &var,
						const AlVector *v1, const AlVector *v2,
						const AlVector *v3, const AlVector *v4 )
{
	out->x = (v1->x + v2->x + v3->x + v4->x ) / 4 + myRandPosNeg()*var.x;
	out->y = (v1->y + v2->y + v3->y + v4->y ) / 4 + myRandPosNeg()*var.y;
	out->z = (v1->z + v2->z + v3->z + v4->z ) / 4 + myRandPosNeg()*var.z;
}

FractSurf::FractSurf( const FractParam &param )
{
	AlVector variance = param.variance;
	// convert the size to a power of 2
	size = (1 << param.subdivisions );	// size = 2^inSize;
	size_x = size + 1;
	size_y = size + 1;

	// scale & convert 100%..0% to 0..1 
	double taper = (100 - param.taper) / 100.0;

	srandom( param.seed );

	// allocate the row pointers
	array = new AlVector[ size_x * size_y ];
	if( !array )
		return;

	// set up the initial four corners
	AlVector pt1(           0,           0, 0 );
	AlVector pt2( param.sizeX,           0, 0 );
	AlVector pt3( param.sizeX, param.sizeY, 0 );
	AlVector pt4(           0, param.sizeY, 0 );

	perturb( pt1, variance );
	perturb( pt2, variance );
	perturb( pt3, variance );
	perturb( pt4, variance );
	variance *= taper;	// taper the variance

	array[    0 ] = pt1;
	array[ size ] = pt2;
	array[ size * size_x + size ] = pt3;
	array[ size * size_x +    0 ] = pt4;

	int step = size;
	int stepV = step * size_x;
	while( step > 1 )
	{ 
		AlPrintf( kPromptNoHistory, "Iteration size %d", step );

		// offset half a square
		int halfStep = step / 2;	
		int halfStepV = stepV / 2;

		// generate horizontal mid points
		{
			AlVector *row = array;

			// Note:           |----- equality for this
			for( int y = 0; y <= size; y += step )
			{
				AlVector *h = row;
				AlVector *mid = h + halfStep;
	
				for( int x = 0; x < size; x += step )
				{
					avgPerturb( mid, variance, h, h+step );
					mid += step;
					h += step;
				}
				row += stepV;
			}
		}
	
		// generate vertical mid points
		{
			AlVector *col = array;
			// Note:           |---- equality for this
			for( int x = 0; x <= size; x += step )
			{
				AlVector *v = col;
				AlVector *mid = v + halfStepV;

				for( int y = 0; y < size; y += step )
				{
					avgPerturb( mid, variance, v, v+stepV );
					mid += stepV;
					v += stepV;
				}
				col += step;
			}
		}

		// generate the middles of the grid
		{
			AlVector *row = array + halfStepV;
			AlVector *col = array + halfStep;

			for( int y = 0; y < size; y += step )
			{
				AlVector *v = row;
				AlVector *h = col;
				AlVector *mid = v + halfStep; 

				for( int x = 0; x < size; x += step )
				{
					avgPerturb( mid, variance, v, v+step, h, h + stepV );
					v += step;
					h += step;
					mid += step;
				}
				row += stepV;
				col += stepV;
			}
		}

		step = halfStep;
		stepV = halfStepV;
		variance *= taper;	// taper the variance
	}
}

inline void swap( double &a, double &b )
{
	double t;
	t =a;
	a = b;
	b = t;
}

statusCode FractParam::get()
{
	if(	sSuccess != AlGetDouble( "taper", taper ) ) {
		AlPrintf( kPrompt, "Invalid taper" );
		return sFailure;
	}

	if( sSuccess != AlGetInteger( "subdivisions", subdivisions ) ) {
		AlPrintf( kPrompt, "Invalid size or taper parameter!");
		return sFailure;
	}

	if( sSuccess != AlGetDouble( "variance.x", variance.x ) ||
		sSuccess != AlGetDouble( "variance.y", variance.y ) ||
		sSuccess != AlGetDouble( "variance.z", variance.z ))
	{
		AlPrintf( kPrompt, "Invalid variance parameter!");
		return sFailure;
	}

	if( sSuccess != AlGetInteger( "seed", seed ) ) {
		AlPrintf( kPrompt, "Invalid seed parameter!");
		return sFailure;
	}

	if( sSuccess != AlGetDouble( "level1", level1 ) ||
		sSuccess != AlGetDouble( "level2", level2 ) ||
		sSuccess != AlGetDouble( "level3", level3 ) ||
		sSuccess != AlGetDouble( "level_jitter", level_jitter ))
	{
		AlPrintf( kPrompt, "Invalid level parameter!");
		return sFailure;
	}

	// sort them
	if( level2 < level1 )
		swap( level1, level2 );
	if( level3 < level1 )
		swap( level1, level3 );
	if( level3 < level2 )
		swap( level3, level2 );

	if( sSuccess != AlGetDouble( "size.x", sizeX ) ||
		sSuccess != AlGetDouble( "size.y", sizeY ) )
	{
		AlPrintf( kPrompt, "Invalid size parameter!");
		return sFailure;
	}

	return sSuccess;
}

// lazy
static inline void posToVec( AlVector& result, AlPolyset *polyset, int index )
{
	AlPolysetVertex *vertex;

    if( polyset )
	{
		vertex = polyset->vertex( index );
		if( vertex )
		{
    		vertex->unaffectedPosition( result.x, result.y, result.z );
			delete vertex;
		}
	}
}


ShaderIndex calcShaderIndex(	const FractParam &param, AlPolyset *polyset,
								int n1, int n2, int n3, int n4 )
{
	// calculate the middle 
	AlVector v1, v2, v3, v4;
	posToVec( v1, polyset, n1 );
	posToVec( v2, polyset, n2 );
	posToVec( v3, polyset, n3 );
	posToVec( v4, polyset, n4 );
					
	AlVector center = (v1+v2+v3+v4)/4;

	double height = center.z + param.level_jitter * myRandPosNeg() ;

	if( height > param.level3 )	return kIndexSnow;
	if( height > param.level2 ) return kIndexRock;
	if( height > param.level1 )	return kIndexGrass;
	
	return kIndexSand;
}

//
// now make verticies out of the vector positions
//
statusCode generateSurface( const FractParam &param, AlPolyset *polyset, FractSurf &fractSurf )
{
	if( !fractSurf )
		return sInvalidArgument;
	int width = fractSurf.size_x;
	int height = fractSurf.size_y;

	int numIndecies = width * height;
	int *indecies;

	if( indecies = new int [ numIndecies ] )
	{
		AlVector *v = fractSurf.array;

		for( int i = 0; i < numIndecies; i++)
		{
			indecies[ i ] = polyset->newVertex( v->x, v->y, v->z );
			v++;
		}

		// now tile the surface
		int *index_row = indecies;
			
		for( int y = 0; y < height - 1; y++)
		{
			int *index = index_row;

			AlPrintf( kPromptNoHistory, "Tiling row %d of %d", y, height );
			for( int x = 0; x < width - 1; x++ )
			{
				int polygon_index = polyset->newPolygon();

				if( polygon_index != -1 )
				{
					AlPolygon *polygon = polyset->polygon( polygon_index );
					if( polygon )
					{
						int n[4];
						n[0] = index[0];
						n[1] = index[1];
						n[2] = index[width+1];
						n[3] = index[width];

						polygon->addVertex( index[0] );
						polygon->addVertex( index[1] );
						polygon->addVertex( index[width+1] );
						polygon->addVertex( index[width] );

						ShaderIndex index = calcShaderIndex( param, polyset,
						n[0], n[1], n[2], n[3] );
		
						polygon->setShaderIndex( index );
						delete polygon;
					}
				}
				else
				{
					delete [] indecies;
					AlPrintf( kPrompt, "Error: can create polygon");
					return sFailure;
				}
				index++;
			}
			index_row += width;
		}
		delete [] indecies;
		return sSuccess;
	}
	return sFailure;
}

typedef statusCode makeShaderFunc( AlShader *shader );

static statusCode makeSnow( AlShader *shader )
{
	statusCode stat;

	stat = shader->setParameter( kFLD_SHADING_COMMON_COLOR_R, 255 );
	if( stat != sSuccess )	return stat;
	stat = shader->setParameter( kFLD_SHADING_COMMON_COLOR_G, 255 );
	stat = shader->setParameter( kFLD_SHADING_COMMON_COLOR_B, 255 );
	return stat;
}

static statusCode makeRock( AlShader *shader )
{
	return shader->addTexture( "color", "Fractal" );
}

static statusCode makeSand( AlShader *shader )
{
	statusCode stat;

	stat =shader->addTexture( "color", "sRock" );
	if( stat == sSuccess )
	{
		AlTexture *texture = shader->firstTexture();
		if( texture )
		{
			texture->setParameter( kFLD_SHADING_SROCK_COLOR1_R, 255 );
			texture->setParameter( kFLD_SHADING_SROCK_COLOR1_G, 187 );
			texture->setParameter( kFLD_SHADING_SROCK_COLOR1_B, 77 );

			texture->setParameter( kFLD_SHADING_SROCK_COLOR2_R, 73 );
			texture->setParameter( kFLD_SHADING_SROCK_COLOR2_G, 53 );
			stat = texture->setParameter( kFLD_SHADING_SROCK_COLOR2_B, 29 );

			delete texture;
		}
	}
	return stat;
}

static statusCode makeGrass( AlShader *shader )
{
	statusCode stat;

	stat = shader->setParameter( kFLD_SHADING_COMMON_COLOR_R, 0 );
	if( stat != sSuccess )	return stat;
	stat = shader->setParameter( kFLD_SHADING_COMMON_COLOR_G, 255 );
	stat = shader->setParameter( kFLD_SHADING_COMMON_COLOR_B, 0 );
	return stat;
}

static statusCode makeShader( AlPolyset *polyset, makeShaderFunc *func, ShaderIndex index )
{
	statusCode stat;
	AlShader shader;

	stat = shader.create();
	if( stat == sSuccess )
	{
		stat = func( &shader );
		if( stat == sSuccess )
		{
			stat = polyset->assignShader( index, &shader );
			if( stat == sSuccess )
			{
				return sSuccess;
			}
		}
		shader.deleteObject();
	}
	return stat;
}

statusCode createShaders( AlPolyset *polyset )
{
	statusCode stat;

	stat = makeShader( polyset, makeSnow,	kIndexSnow );
	if( stat != sSuccess ) return stat;
	stat = makeShader( polyset, makeRock,	kIndexRock );
	if( stat != sSuccess ) return stat;
	stat = makeShader( polyset, makeGrass,	kIndexGrass );
	if( stat != sSuccess ) return stat;
	stat = makeShader( polyset, makeSand,	kIndexSand );
	return stat;
}

void do_surf()
{
	FractParam param;

	AlPrintf( kPrompt, "Create Surface begins...");
	if( param.get() == sSuccess )
	{
		AlPolyset polyset;

		if( polyset.create() != sSuccess )
		{
			AlPrintf( kPrompt, "Couldn't create polyset");
		}
		else
		{
			if( sSuccess != createShaders( &polyset ) )
			{
				AlPrintf( kPrompt, "Couldn't create shaders");
			}
			else
			{ 
				FractSurf surf( param );
				if( !!surf )
				{
					generateSurface( param, &polyset, surf );
				}
			}
		}
		AlUniverse::redrawScreen();
	}
	AlPrintf( kPrompt, "Create surface ends...");
}

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
	char *dirScm;

	AlUniverse::initialize( kZUp );

	dirScm = makeAltPath(dirName,"scm");

	AlInvokeSchemeFile( INIT_OPTIONBOX_FILE, dirScm );

	AlDebugResetOptionBox( "surf.options ");

	//
	//	note that the icon files must match the function name
	//	ie. FractSurf.S and FractSurf.M
	//
	hFunc.create( "FractSurf", do_surf );

	h.create( "Create Surface", &hFunc ); 
	h.setAttributeString( "fract surf" );

	if ( sSuccess != h.setOptionBox( OPTIONBOX_FILE, "surf.options", dirScm ) ) {
		AlPrintf( kPrompt, "createSurface plug-in unable to find .scm file for option box" );
		return 1;
	}

	h.setIconPath( makeAltPath( dirName, NULL ) );
	h.appendToMenu( "mp_objtools" );
	
    AlPrintf( kPrompt, "Create Surface plugin installed under 'Object Edit' palette");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    h.deleteObject();
	hFunc.deleteObject();
    // do nothing
    return 0;
}

