/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlDagNode.h>
#include <AlCurveNode.h>
#include <AlSurfaceNode.h>
#include <AlCurve.h>
#include <AlSurface.h>
#include <AlTM.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlWindow.h>

#include <math.h>
#include <AlVector.h>

//
//	This plugin provides an example of interactive editing using a
//	continuous function
//

struct ScreenRange
{
	Screencoord min, max;
};

struct WindowRange
{
	double min, max;
};

struct vars
{
	ScreenRange sx, sy;
	WindowRange rx, ry;
	char inputBuffer[256];
	char outputBuffer[256];

	double 			first_x, first_y, first_z;
	AlWindow		*window;
	//Screencoord		first_x, first_y;
	double			ratio_x, ratio_y, ratio_z;
	AlSurface		*sphere;
	AlSurfaceNode	*sphereNode;
} v;

// makes a '(' half circle
statusCode makeHalfCircle( AlCurve &curve )
{
    double center[3]=       { 0, 0, 0};
    double major_axis[3]=   { 1, 0, 0};
    double minor_axis[3]=   { 0, 1, 0};
    double start[3]=        { 0,-1, 0};
    double end[3]=          { 0, 1, 0};

    return curve.createEllipse( center, major_axis, minor_axis, start, end );
}

statusCode makeSphere( AlSurface &surf )
{
    AlCurve curve;
	statusCode stat;

    stat = makeHalfCircle( curve );
	if( stat == sSuccess )
    {
        double startPoint[3]={0,0,0};
        double endPoint[3]={0,1,0};
        statusCode stat = surf.createRevolvedSurface(   startPoint, endPoint,
                                                        0, 360, &curve );
		curve.deleteObject();
    }
    return stat;
}

void preinit_func()
{
	// nothing
}

void init_func()
{
	// create the sphere using a revolved surface
	v.sphere = new AlSurface;
	v.ratio_x = 1;
	v.ratio_y = 1;
	v.ratio_z = 1;
	v.window = NULL;
	makeSphere( *(v.sphere) );
}

void do_input_keyboard()
{
	// get the data out of our buffer
	sscanf( v.inputBuffer,	"%lf %lf %lf",
							&v.ratio_x, &v.ratio_y, &v.ratio_z );
}

void down_func( int input, Screencoord x, Screencoord y )
//
//	Add the sphere
//
{
	int button;

	switch( AlContinuousFunction::translateInput( input, button ) )
	{
		default:
			break;

		case kInputKeyboard:
			do_input_keyboard();
			break;

		case kInputButton:
		{
			//ScreenRange sx, sy;
			AlWindow::AlViewType tp;

			// you may need these to clip the mouse to the window
			//AlContinuousFunction::windowBounds(	sx.min, sy.min,
			//									sx.max, sy.max );

			// figure out which view that we're on
			v.window = AlUniverse::currentWindow();
			if( !AlIsValid( v.window ) ) {
				return;
			}

			v.window->windowType( tp );
			switch( tp ) {
				case AlWindow::kTop:
				case AlWindow::kBottom:
				case AlWindow::kFront:
				case AlWindow::kBack:
				case AlWindow::kRight:
				case AlWindow::kLeft:
				{
					double px, py, pz;
					double qx, qy, qz;
					v.window->mapToWorldSpace( x,y, px,py,pz, qx,qy,qz );
					if( px!=qx ) px=0.0;
					if( py!=qy ) py=0.0;
					if( pz!=qz ) pz=0.0;

					v.sphereNode = new AlSurfaceNode;
					v.sphereNode->create( v.sphere );
					v.sphereNode->setTranslation( px, py, pz );
					// set a scale of 0, since the cursor will be onto top
					// of the drop point, so the distance will be 0
					v.sphereNode->setScale( 0, 0, 0 );
					v.first_x = px; v.first_y = py; v.first_z = pz;
				}
				break;
				default:
					delete v.window;
					v.window = NULL;
			}
			AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
		}
	}
}

void move_func( int input, Screencoord x, Screencoord y )
//
//	Scale the sphere
//
{
	int button;
	switch( AlContinuousFunction::translateInput( input, button ) )
	{
		default:
			break;

		case kInputKeyboard:
			do_input_keyboard();
			break;

		case kInputButton:
		{
			if( !AlIsValid( v.window ) ) {
				return;
			}

			if( v.sphereNode )
			{
				// do a little math to scale the sphere to the magnitude of
				// the difference
				double px, py, pz, qx, qy, qz;

				v.window->mapToWorldSpace( x,y, px,py,pz, qx,qy,qz );
				if( px!=qx ) px=0.0;
				if( py!=qy ) py=0.0;
				if( pz!=qz ) pz=0.0;
	
				double dx = px - v.first_x;
				double dy = py - v.first_y;
				double dz = pz - v.first_z;
				double size = sqrt( dx * dx + dy * dy + dz * dz );

				v.sphereNode->setScale( v.ratio_x * size, v.ratio_y * size,
										v.ratio_z * size );

				AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
			}
		} 
	}
}

void up_func( int input, Screencoord, Screencoord )
//
//	Do nothing
//
{
	int button;
	switch( AlContinuousFunction::translateInput( input, button ) )
	{
		default:
			break;

		case kInputKeyboard:
			do_input_keyboard();
			break;

		case kInputButton:
		{
			if( !AlIsValid( v.window ) ) {
				return;
			}

			delete v.sphereNode;
			v.sphereNode = NULL;

			// make another for the next copy
			delete v.sphere;
			v.sphere = new AlSurface;

			makeSphere( *(v.sphere) );
			delete v.window;
			v.window = NULL;
		}
	}
}

void cleanup_func()
{
	if( v.sphere )
	{
		v.sphere->deleteObject();
		delete v.sphere;
		v.sphere = NULL;
	}
	if( AlIsValid( v.window ) ) {
		delete v.window;
	}
}

const char *computePrompt_func()
//
//	Anytime that the prompt is redisplayed, we want to update it correctly
//	to reflect the user's coordinate preference
//
//	By specifying an update function, we can return any string that we want
//	(note that this string pointer must be valid after this function exits,
//	so it cannot point to a local variable).
//
{
	if( AlContinuousFunction::keyboardCoordinateMode() == kCoordinateRelative )
		return "Enter a sphere ratio (relative): %f %f %f";
	else
		return "Enter a sphere ratio (absolute): %f %f %f";
}

static AlFunctionHandle h;
static AlContinuousFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize( kZUp );

	hFunc.create(	"SpherePlugin",
					init_func, down_func, move_func, up_func, cleanup_func,
					FALSE );

	hFunc.setPrompt(	computePrompt_func, v.inputBuffer, kFilterLinear );

	h.create( "Contin Sphere", &hFunc ); 
	h.setAttributeString( "contin test" );
	h.appendToMenu( "mp_objtools" );
	h.setIconPath( makeAltPath( dirName, NULL ) );

	AlPrintf( kPrompt, "Sphere installed on Palette 'Object Edit'");
	
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    h.removeFromMenu();
    h.deleteObject();
	hFunc.deleteObject();

    // do nothing
    return 0;
}

