/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

// createPerspCameraVector.plugin
//
//	Select the perspective modeling window in order to create a "view" vector.  
//	The "view" vector is created between the the camera eye position and
//	the camera look at point.  The view vector has unit length.
//
//	This vector is useful for referring back to the
//	view that was set at the time the vector was created.
//
//	Installed under Construction palette.
//

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlUniverse.h>
#include <AlDagNode.h>
#include <AlPickList.h>
#include <AlPickable.h>
#include <AlWindow.h>
#include <AlTM.h>
#include <AlCamera.h>
#include <AlPerspectiveCamera.h>
#include <AlConstructionVector.h>
#include <AlSpacePoint.h>
#include <AlDebug.h>

#include <math.h>

char inbuf[ 256 ];
char text_buf[ 100 ];
const char * my_prompt = "Select the Perspective Modeling window: %s";

#ifdef _WIN32
AlOutputType outputTyp = kPrompt;
#else
AlOutputType outputTyp = outputTyp;
#endif

static void setToolTipString()
{
	char buf[1024];
	
	sprintf(buf,"( ui-function-tooltip-item  \"%s\" \"%s\" )",
		"pl_CreatePerspCamVec",
		"Select the perspective modeling window to create a unit length camera vector." );

	AlInvokeSchemeCommand( buf );
}

void go_button_pressed( void )
{
}

static void init_func( void ) 
{
}

static void down_func( int , Screencoord , Screencoord  )
{
}
	
static void move_func( int , Screencoord , Screencoord  )
{
}

static statusCode findPerspectiveCamera( AlWindow *window, AlCamera *camera, AlPerspectiveCamera *&persp )
{
	persp = AlUniverse::firstCamera();
	if ( persp != NULL )
	do 
	{
		if ( AlAreEqual( camera, persp ) )
		{
			return sSuccess;
		}
	} while ( AlUniverse::nextCameraD( persp ) != sFailure );
	if ( persp != NULL )
		delete persp;
	return sFailure;
}

static statusCode createVectorFromPerspCamera( AlPerspectiveCamera * persp )
{
	//
	// First get the camera eye and view positions
	//
	double wvx, wvy, wvz; // world view
	double wex, wey, wez; // world eye
	
	if ( persp->worldView( wvx, wvy, wvz ) != sSuccess )
		return sFailure;

	if ( persp->worldEye( wex, wey, wez ) != sSuccess )
		return sFailure;
		
	//
	// Normalize the vector
	//
	double nx, ny, nz;
	double divisor = sqrt( (wex-wvx)*(wex-wvx) + (wey-wvy)*(wey-wvy) + (wez-wvz)*(wez-wvz) );
	nx = (  wvx - wex ) / divisor;
	ny = (  wvy - wey ) / divisor;
	nz = (  wvz - wez ) / divisor;
	
	//
	// Next create the required space points at the view point
	//
	AlSpacePoint *view = NULL;
	AlSpacePoint *eye = NULL;
	AlConstructionVector *vec = NULL;
	statusCode result = sFailure;
	
	view = new AlSpacePoint;
	eye = new AlSpacePoint;
	
	//
	//	The following approach based on 'if success' condition
	//	was chosen so that wrapper clean up could be easily done
	//	at the end of the function.
	//
	if ( view != NULL )
	{
		if ( eye != NULL )
		{
			if ( view->create( (wex + nx), (wey + ny), (wez + nz) ) == sSuccess )
			{
				if ( eye->create( wex, wey, wez ) == sSuccess )
				{
					vec = new AlConstructionVector;
					if ( vec != NULL )
					{
						if ( vec->create( eye, view ) == sSuccess )
						{
							result = sSuccess;
						}
					}
				}
			}
		}
	}
	
	AlUniverse::redrawScreen( kRedrawAll );
	
	if ( view != NULL )
		delete view;
	if ( eye != NULL )
		delete eye;
	if ( vec != NULL )
		delete vec;

	return result;
}

static statusCode createVector( AlWindow *window )
{
	if ( window == NULL )
		return sFailure;
		
	AlWindow::AlViewType vt;
	if ( window->windowType( vt ) == sSuccess )
	{
		if ( vt == AlWindow::kPerspective )
		{
			AlCamera *camera = window->camera();
			if ( camera != NULL )
			{
				statusCode result = sFailure;
				AlPerspectiveCamera *persp = NULL;
				if ( findPerspectiveCamera( window, camera, persp ) == sSuccess )
				{
					result = createVectorFromPerspCamera( persp );
				}
				
				// Free wrapper
				if ( persp != NULL )
					delete persp;
					
				delete camera;	// Non NULL check already passed
				
				if ( result == sSuccess )
				{
					AlPrintf( kPrompt, "Created perspective camera vector."); 
					return sSuccess;
				}
			}
			
		}
		else
		{
			AlPrintf( kPrompt, "Please select the perspective window.");
			return sSuccess;
		}
	}
	return sFailure;
}

static void up_func( int input, Screencoord x, Screencoord y )
{
	int button;
	double dx=-1, dy=-1, dz=-1, ex=-1, ey=-1, ez=-1;
	AlWindow * w;

	switch( AlContinuousFunction::translateInput( input, button ) ) {
	case kInputButton:
		w = AlUniverse::currentWindow();
		w->mapToWorldSpace( x, y, dx, dy, dz, ex, ey, ez );
		
		if ( createVector( w ) != sSuccess )
		{
			AlPrintf( kPrompt, "Error: failed to create perspective camera vector!");
		}
		
		delete w; 
		break;
	}

#if 0	
	// Turn this code on to trap wrapper leakage.
	printf("\t*****************************\n");
	AlDebug::outputWrapperUsage( kStderr );
	printf("\t*****************************\n");
#endif
}

static void cleanup_func() 
{
}

static AlFunctionHandle h;
static AlContinuousFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize();

	hFunc.create( "pl_CreatePerspCamVec", init_func, down_func, move_func, 
								up_func, cleanup_func, TRUE );
	hFunc.setPrompt( my_prompt, inbuf, kFilterNone );
	h.create( "Create Persp Camera Vector", &hFunc );
	h.setAttributeString( "CreatePerspCameraVector" );
	h.setIconPath( makeAltPath( dirName, NULL ) );
	h.addToMenu( "mp_grid" );

	setToolTipString();
	
	AlPrintf( kPrompt, 
		"Create Perspective Camera Vector example installed on Palette 'Construction'.");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	h.removeFromMenu();
	h.deleteObject();
	hFunc.deleteObject();

	return 0;
}
