/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

// ground.plugin
//
// This is a continuous function which creates a polygonal planar mesh
// by dragging to size in the modeling windows.  It is a useful example
// in that it demonstrates the use of 'helper' geometry to guide the
// construction of an object in cases where the real geometry may be too
// costly to draw in real time.
//

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlPolyset.h>
#include <AlPolysetNode.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlPickList.h>

#include <AlTM.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlWindow.h>

#include <math.h>
#include <string.h>

struct vars
{
	double 			down[3];
	int				thread_count;
	int				do_inline;
	AlPolysetNode   *node;
	int				*arr;
	AlWindow		*window;
} v;

void generate_guide( void ) 
{
	if( v.node ) {
		return;
	}

	AlPolygon *pgon;
	AlPolyset *pset;
	int tl, tr, br, bl, p;

	pset = new AlPolyset;
	if( !pset ) {
		return;
	}

	if( pset->create() != sSuccess ) {
		return;
	}

	bl = pset->newVertex( 0,0,0 );
	tl = pset->newVertex( 0,1,0 );
	tr = pset->newVertex( 1,1,0 );
	br = pset->newVertex( 1,0,0 );
	p = pset->newPolygon();
	pgon = pset->polygon( p );
	pgon->addVertex( bl );
	pgon->addVertex( tl );
	pgon->addVertex( tr );
	pgon->addVertex( br );
	delete pgon;

	/* Do the xforms on the dagnode */
	v.node = pset->polysetNode();
	v.node->setScalePivot( 0,0,0 );
}

AlPolyset* generate_polyset( double down[ 3 ], double up[ 3 ] )
{
	AlPolygon *pgon;
	AlPolyset *pset;
	int idx; int jdx;
	int *arr;

	pset = new AlPolyset;
	if( !pset ) {
		return NULL;
	}

	if( pset->create() != sSuccess ) {
		return NULL;
	}

	int t = v.thread_count + 1;
	v.arr = new int[ (t+1)*(t+1) + 1 ];
	if( !v.arr ) {
		return NULL;
	}
	v.arr[ 0 ] = (t+1);

	arr = v.arr + 1;

	double dx,dy,dz,di,dj,dk;
	int min = 0;
	dx = up[0]-down[0]; dy = up[1]-down[1]; dz = up[2]-down[2];
	if( dy*dy < dx*dx && dy*dy < dz*dz ) 
		min = 1; 
	else if( dz*dz < dx*dx && dz*dz < dy*dy )
		min = 2; // Min is assumed to be the axis that doesn't move.

	/* Generate the vertices */
	for( idx = 0; idx <= t; idx++ ) {
		di = ((double)idx)/((double)t);
		for( jdx = 0; jdx <= t; jdx++ ) {
			dj = ((double)jdx)/((double)t);
			if( min == 0 ) {
				dk = dj;
			} else if( min == 1 ) {
				dk = di;
			} else {
				dk = 0;
			}
			arr[ ( (t+1) * idx ) + jdx ] = pset->newVertex( 
				down[0] + dj*dx, down[1] + di*dy, down[2] + dk*dz );
		}
	}

	/* Generate the polygons */
	for( idx = 1; idx <= t; idx++ ) {
		for( jdx = 1; jdx <= t; jdx++ ) {
			int pgon_index = pset->newPolygon();
			if( pgon_index != -1 ) {
				pgon = pset->polygon( pgon_index );
				if( pgon ) {
					pgon->addVertex( arr[ ( idx - 1 )*(t+1) + ( jdx - 1 ) ] );
					pgon->addVertex( arr[ ( idx - 1 )*(t+1) + ( jdx ) ] );
					pgon->addVertex( arr[ ( idx )*(t+1) + ( jdx ) ] );
					pgon->addVertex( arr[ ( idx )*(t+1) + ( jdx - 1 ) ] );
					delete pgon;
				}
			}
		}
	}

	return pset;
}

void preinit_func()
{
	v.node = NULL;
	v.arr = NULL;
}

void do_input_keyboard()
{
}

void init_func( void )
{
	int res;

	v.thread_count = 1;
	v.do_inline = 0;
	v.window = NULL;

	statusCode code = AlGetInteger( "ground.thread_count", res );
	if( code == sSuccess )
		v.thread_count = res;

	code = AlGetInteger( "ground.inline", res );
	if( code == sSuccess )
		v.do_inline = res;
}

void down_func( int input, Screencoord x, Screencoord y )
{
	int button;
	v.window = AlUniverse::currentWindow();
	if( !AlIsValid( v.window ) ) {
		return;
	}

	AlWindow::AlViewType tp;
	v.window->windowType( tp );

	if( tp != AlWindow::kFront && 
		tp != AlWindow::kBack && 
		tp != AlWindow::kTop && 
		tp != AlWindow::kBottom && 
		tp != AlWindow::kRight && 
		tp != AlWindow::kLeft ) {
		delete v.window;
		v.window = NULL;
		return;
	}

	AlPickList::clearPickList();
	AlUniverse::redrawScreen( kRedrawInactive );

	if( !v.node ) {
		if( v.do_inline ) {
			double d[3] = { 0,1,0 };
			double u[3] = { 1,0,0 };
			AlPolyset* pset = generate_polyset( d, u );
			if( !pset ) {
				return;
			}
			v.node = pset->polysetNode();
			v.node->setScalePivot( 0,0,0 );
			delete pset;
		} else {
			generate_guide();
		}
	}

	if( !v.node ) {
		return;
	}

	v.node->pick();

	switch( AlContinuousFunction::translateInput( input, button ) )
	{
		case kInputButton:
		{
			double px,py,pz,qx,qy,qz;
			v.window->mapToWorldSpace( x,y, px,py,pz, qx,qy,qz );
			if(px!=qx)px=0.0; if(py!=qy)py=0.0; if(pz!=qz)pz=0.0;
			v.down[0]=px;v.down[1]=py;v.down[2]=pz;

			switch( tp )
			{
				case AlWindow::kTop:
				case AlWindow::kBottom:
				case AlWindow::kBack:
				case AlWindow::kFront:
					if( tp == AlWindow::kFront ) {
						v.node->setRotation( 90, 0, 0 );
					}
				case AlWindow::kRight:
				case AlWindow::kLeft:
					if( tp == AlWindow::kRight ) {
						v.node->setRotation( 0, 90, 0 );
					}
					v.node->setTranslation( v.down[0],v.down[1],v.down[2] );
					v.node->setScale( 0, 0, 0 );
				default:
					break;
			}
			AlUniverse::redrawScreen( kRedrawActive );
		}
	}
}

void move_func( int input, Screencoord x, Screencoord y )
{
	int button;
	switch( AlContinuousFunction::translateInput( input, button ) )
	{
		case kInputButton:
		{
			if( !AlIsValid( v.window ) ) {
				return;
			}

			double px,py,pz,qx,qy,qz;
			double xr, yr, zr = 1;

			v.window->mapToWorldSpace( x,y, px,py,pz, qx,qy,qz );
			if(px!=qx)px=0.0;if(py!=qy)py=0.0;if(pz!=qz)pz=0.0;

			AlWindow::AlViewType tp;
			v.window->windowType( tp );
			switch( tp ) {
			case AlWindow::kTop:
			case AlWindow::kBottom:
				xr = px - v.down[0];
				yr = py - v.down[1];
				break;
			case AlWindow::kFront:
			case AlWindow::kBack:
				xr = px - v.down[0];
				yr = pz - v.down[2];
				break;
			case AlWindow::kRight:
			case AlWindow::kLeft:
				yr = py - v.down[1];
				xr = v.down[2] - pz;
				break;
			default:
				return;
			}

			v.node->setScale( xr, yr, zr );

			AlUniverse::redrawScreen( kRedrawActive );
		} 
	}
}

void up_func( int input, Screencoord x, Screencoord y )
{
	int button;
	statusCode code;
	int res;
	int perturb;

	if( !AlIsValid( v.window ) ) {	
		return;
	}

	AlWindow::AlViewType tp;
	v.window->windowType( tp );
	switch( AlContinuousFunction::translateInput( input, button ) )
	{
		case kInputButton:
		{
			double p[3];
			double qx,qy,qz;

			if( v.node ) {
				if( !v.do_inline ) {
					AlPickList::clearPickList();
					v.node->deleteObject();
					delete v.node;
					v.node = NULL;
					v.window->mapToWorldSpace( x,y, p[0],p[1],p[2], qx,qy,qz );
					if(p[0]!=qx)p[0]=0.0;
					if(p[1]!=qy)p[1]=0.0;
					if(p[2]!=qz)p[2]=0.0;

					AlPolyset * pset = generate_polyset( v.down, p );
					if( !pset )
						return;

					v.node = pset->polysetNode();
					delete pset;
					v.node->pick();
				}

				code = AlGetInteger( "ground.perturb", res );
				if( code == sSuccess ) {
					perturb = res;
				}
				if( perturb ) {
					// Must cache vertex data into blind data area.
					v.node->setBlindData( 
						12001, 
						(v.arr[ 0 ] + 1)*sizeof( int ), 
						(char*)(v.arr) );
					v.arr = NULL;
				} else {
					delete [] v.arr;
					v.arr = NULL;
				}

				delete v.node;
				v.node = NULL;
				delete v.window;
				v.window = NULL;

				AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
			}
		}
	}
}

void cleanup_func()
{}

static AlFunctionHandle h;
static AlContinuousFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
	char *dirScm;

	v.node = NULL;
	AlUniverse::initialize( kZUp );

	dirScm = makeAltPath(dirName,"scm");

	AlInvokeSchemeFile( "createMeshExample_init.scm", dirScm );

	hFunc.create(	"pl_PlanarMesh",
					init_func, down_func, move_func, up_func, cleanup_func,
					FALSE );

	h.create( "Planar Mesh", &hFunc ); 
	h.setAttributeString( "mesh" );

    if ( sSuccess != h.setOptionBox( "createMeshExample.scm", "ground.options", 
		dirScm ) ) {
        	AlPrintf( 
			kPrompt, 
			"createMesh plug-in unable to find .scm file for option box"
		);
        return (1);
    }

	h.appendToMenu( "mp_objtools" );
	h.setIconPath( makeAltPath( dirName, NULL ) );
	AlPrintf( kPrompt, "Create Mesh installed on Palette 'Object Edit'");
	
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    h.removeFromMenu();
    h.deleteObject();
	hFunc.deleteObject();

    // do nothing
    return 0;
}

