/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
 * helix.c++
 *
 * This is a sample plugin which creates a double helix.  It uses a
 * standard parametric function of time, ( cos(t), sin(t), t ) to 
 * calculate the curves.
 *
 * In addition, this plugin demonstrates the use of confirm boxes in 
 * a plugin.
 */

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlTM.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <math.h>

#ifdef _WIN32
#define PLUGINAPI_DECL __declspec(dllexport)
#else
#define PLUGINAPI_DECL
#endif /* _WIN32 */

void do_double_helix( void )
{
	int last_a = -1, last_b = -1;
	int now_a = -1, now_b = -1;

	statusCode code;
	AlAnswerType ans;

	code = AlPromptBox( kOK_Cancel, 
						"Are you sure you want\nto do this?",
						&ans, -1, -1 );
	
	if( code != sSuccess ) {
		AlPrintf( kPrompt, "AlPromptBox failed!" );
		return;
	}

	if( ans == kCancel ) {
		AlPrintf( kPrompt, "Double Helix canceled." );
		return;
	}

	AlPolyset * pset = new AlPolyset;

	if( !pset ) {
		AlPrintf( kPrompt, "Coudn't allocate polyset." );
		return;
	}

	code = pset->create();
	if( code != sSuccess ) {
		AlPrintf( kPrompt, "Coudn't create polyset." );
		delete pset;
		return;
	}

	int count = 0;
	for( double t = -4.0; t <= 4.0; t += .2 ) {
		if( count == 0 ) {
			now_a = pset->newVertex( 2.0 * sin( t ), 2.0 * cos( t ), t );
			now_b = pset->newVertex( -2.0 * sin( t ), -2.0 * cos( t ), t );
		} else {
			last_a = now_a; last_b = now_b;
			now_a = pset->newVertex( 2.0 * sin( t ), 2.0 * cos( t ), t );
			now_b = pset->newVertex( -2.0 * sin( t ), -2.0 * cos( t ), t );
			int pgon = pset->newPolygon();
			AlPolygon *polygon = pset->polygon( pgon );
			polygon->addVertex( last_a ); polygon->addVertex( last_b );
			polygon->addVertex( now_b ); polygon->addVertex( now_a );
			delete polygon;
		}
		count++;
	}

	delete pset;

	AlPrintf( kPrompt, "double helix complete." );
	AlUniverse::redrawScreen();

	code = AlPromptBox( kYes_No_Cancel, 
						"So - do you like\nmy double helix?",
						&ans, -1, -1 );
	
	if( code != sSuccess ) {
		AlPrintf( kPrompt, "AlPromptBox failed!" );
		return;
	}

	if( ans == kCancel ) {
		AlPrintf( kPrompt, "Result: Abstain" );
	} else if( ans == kYes ) {
		AlPrintf( kPrompt, "Result: Yes." );
	} else {
		AlPrintf( kPrompt, "Result: No." );
	}
}

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize( kZUp );

	hFunc.create( do_double_helix );
	h.create( "Helix", &hFunc ); 
	h.setAttributeString( "2*helix" );
	h.appendToMenu( "mp_objtools" );
	h.setIconPath( makeAltPath( dirName, NULL ) );
	
	AlPrintf( kPrompt, "Double Helix Installed on Palette 'Object Edit'." );
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    h.deleteObject();
    hFunc.deleteObject();
	return 0;
}
