/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

// control.plugin
//
// This plugin implements a novel approach to using option boxes.
// Because the option box code stores the contents of the widgets it
// uses _directly_ into the ui-symbol that stores that widget's value,
// and because idle events get processed even while an option box is
// open, it is possible to write a plugin that opens an option box,
// and during idle events updates Alias based on the realtime value stored
// in the option box.  In this case, the three sliders will control the
// X Y and Z translations of the first DagNode on the pick list.  So pick
// an object and fire up the option box for this menu item.  Notice how
// the init_option_function is necessary to fire off the first idle event
// and load the initial position of the object into the option box.

#include <AlObject.h>
#include <AlUniverse.h>
#include <AlDagNode.h>
#include <AlPickList.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlMessage.h>
#include <stdio.h>
#include <string.h>

static AlDagNode * node = NULL;
static int my_message_type = -1;
static int option_box_open = 0;

static void on_idle_event( void* )
{
	if( !AlIsValid( node ) ) {
		return;
	}

	double x, y, z;
	AlGetDouble( "control.pos_x", x );
	AlGetDouble( "control.pos_y", y );
	AlGetDouble( "control.pos_z", z );
	
	//printf( "idle event... %lf %lf %lf.\n", x,y,z );
	node->setTranslation( x,y,z );
	AlUniverse::redrawScreen( kRedrawActive );
	
	if( option_box_open ) {
		AlMessage::sendMessage( my_message_type, NULL, AlMessage::kIdleQueue );
	}
}

static void option_box_init_func( void )
{
	if( node ) {
		delete node;
	}
	node = NULL;

	AlPickList::firstPickItem();
	AlObject * obj = AlPickList::getObject();
	if( obj ) {
		node = obj->asDagNodePtr();
	}
	if( !node ) {
		delete obj;
		return;
	}

	double x,y,z;
	node->translation( x,y,z );
	AlSetDouble( "control.pos_x", x );
	AlSetDouble( "control.pos_y", y );
	AlSetDouble( "control.pos_z", z );
	option_box_open = 1;
	AlMessage::sendMessage( my_message_type, NULL, AlMessage::kIdleQueue );
}

static void option_box_action_func( void )
{
	if( node ) {
		delete node;
	}
	node = NULL;
	option_box_open = 0;
	on_idle_event( NULL);
}

static AlMessageTypeHandle t;
static AlMomentaryFunction m;
static AlMomentaryFunction inf;
static AlFunctionHandle mHandle;
static AlFunctionHandle iHandle;

extern "C"
PLUGINAPI_DECL int plugin_init( const char* dirName )
{
	char *dirScm;

	AlUniverse::initialize();

	inf.create( "control.init_func", option_box_init_func );

	dirScm = makeAltPath(dirName,"scm");

	//	control.action_func matches the icon control.action_func.(S/M)
	//
	m.create( "control.action_func", option_box_action_func );
	AlInvokeSchemeFile( "controlExample_init.scm" , dirScm );

	// 'Control' is the name that appears on the menu
	//
	mHandle.create( "Control", &m );
	mHandle.setAttributeString( "control" );

	iHandle.create( "iControl", &inf );
	iHandle.setAttributeString( "icontrol" );

	if ( sSuccess != mHandle.setOptionBox( "controlExample.scm", "control.options", 
		dirScm ) ) {
		AlPrintf( 
			kPrompt, 
			"Control plug-in unable to find .scm file for option box"
		);
		return (1);
	}

	mHandle.setIconPath( makeAltPath( dirName, NULL ) );
	mHandle.appendToMenu( "mp_objtools" );

	iHandle.setIconPath( makeAltPath( dirName, NULL ) );
	iHandle.appendToMenu( "mp_objtools" );

	my_message_type = AlMessage::getMessageType( "control_message" );
	if ( my_message_type == -1 )
	{
		t = AlMessage::addMessageType( "control_message" );
		if( !t.isValid() ) 
			return -1;
		my_message_type = t.type();
	}
	
	AlMessage::addMessageHandler( my_message_type, (void*)on_idle_event );

	AlPrintf( kPrompt, "Control plugin installed on Palette 'Object Edit'");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
//
//	Remove ourselves and cleanup
//
{
	AlMessage::removeMessageHandler( my_message_type, (void*)on_idle_event );

	// remove the menu options
	//
    mHandle.deleteObject();
    iHandle.deleteObject();

	// remove the functions
    inf.deleteObject();
    m.deleteObject();
	return 0;
}

