/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

// cont.plugin
//
// This file implements a simple continuous function plugin, a plugin
// that gets events from Alias and can interact with the user.  It can 
// be used as a skeleton from which to construct other, more complicated
// continuous functions if desired.
//

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlUniverse.h>
#include <AlDagNode.h>
#include <AlPickList.h>
#include <AlPickable.h>
#include <AlWindow.h>
#include <AlTM.h>

char inbuf[ 256 ];
char text_buf[ 100 ];
const char * my_prompt = "Enter some text: %s";

#ifdef _WIN32
AlOutputType outputTyp = kPrompt;
#else
AlOutputType outputTyp = outputTyp;
#endif

void go_button_pressed( void )
{
	AlPrintf( outputTyp, "Go button pressed!\n" );
	AlContinuousFunction::createGoButton( go_button_pressed );
}

static void init_func( void ) {
	AlPrintf( outputTyp, "Now entering continuous plugin.\n" );
	AlContinuousFunction::createGoButton( go_button_pressed );
}

static void down_func( int input, Screencoord x, Screencoord y )
{
	int button;
	double dx=-1, dy=-1, dz=-1, ex=-1, ey=-1, ez=-1;
	AlWindow * w;

	switch( AlContinuousFunction::translateInput( input, button ) ) {
	case kInputButton:
		w = AlUniverse::currentWindow();
		w->mapToWorldSpace( x, y, dx, dy, dz, ex, ey, ez );
		AlPrintf( outputTyp, "Mousedown at %d,%d [%lf,%lf,%lf %lf,%lf,%lf].  Button %d.\n",
			x,y, dx,dy,dz, ex,ey,ez, button );
		delete w;
		break;
	}
}
	
static void move_func( int input, Screencoord x, Screencoord y )
{
	int button;
	double dx=-1, dy=-1, dz=-1, ex=-1, ey=-1, ez=-1;
	AlWindow * w;

	switch( AlContinuousFunction::translateInput( input, button ) ) {
	case kInputButton:
		w = AlUniverse::currentWindow();
		w->mapToWorldSpace( x, y, dx, dy, dz, ex, ey, ez );
		AlPrintf( outputTyp, "Mousemove at %d,%d [%lf,%lf,%lf %lf,%lf,%lf].  Button %d.\n",
			x,y, dx,dy,dz, ex,ey,ez, button );

		// If a construction plane is active, then the coordinates above
		// are within the plane.  We map these coordinates to world space
		// using the following code.
		if ( AlUniverse::constructionPlaneInEffect() )
		{
			AlTM tm;
			tm.identity();
			if ( AlUniverse::constructionPlaneTransform( tm ) == sSuccess )
			{
				double v[4];
				// Find the average point
				v[0] = (ex+dx)/2 ; v[1] = (ey+dy)/2 ; v[2] = (ez+dz)/2; v[3] = 1.0;
				if ( tm.transPoint( v ) == sSuccess )
					AlPrintf( outputTyp, "\tConstruction plane location: %lf,%lf,%lf\n",v[0],v[1], v[2],v[3]);	
			}
		}
		delete w;
		break;
	case kInputKeyboard:
		sscanf( inbuf, "%s", text_buf );
		AlPrintf( outputTyp, "Text input: %s [%s]\n", text_buf, inbuf );
		break;
	}
}

static void up_func( int input, Screencoord x, Screencoord y )
{
	int button;
	double dx=-1, dy=-1, dz=-1, ex=-1, ey=-1, ez=-1;
	AlWindow * w;

	switch( AlContinuousFunction::translateInput( input, button ) ) {
	case kInputButton:
		w = AlUniverse::currentWindow();
		w->mapToWorldSpace( x, y, dx, dy, dz, ex, ey, ez );
		AlPrintf( outputTyp, "Mouseup at %d,%d [%lf,%lf,%lf %lf,%lf,%lf].  Button %d.\n",
			x,y, dx,dy,dz, ex,ey,ez, button );
		delete w;
		break;
	}
}

static void cleanup_func() {
	AlPrintf( outputTyp, "Now leaving continuous plugin.\n" );
	AlContinuousFunction::clearGoButton( TRUE );
}

static AlFunctionHandle h;
static AlContinuousFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize();

	hFunc.create( "Continuous", init_func, down_func, move_func, 
								up_func, cleanup_func, TRUE );
	hFunc.setPrompt( my_prompt, inbuf, kFilterNone );
	h.create( "Cont Test", &hFunc );
	h.setAttributeString( "cont" );
	h.setIconPath( makeAltPath( dirName, NULL ) );
	h.addToMenu( "mp_objtools" );

	AlPrintf( kPrompt, "Continuous example installed on Palette 'Object Edit.");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	h.removeFromMenu();
	h.deleteObject();
	hFunc.deleteObject();

	return 0;
}
