/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlCurve.h>
#include <AlCurveCV.h>
#include <AlCurveNode.h>

#include <AlSurface.h>
#include <AlSurfaceCV.h>
#include <AlSurfaceNode.h>

#include <AlPolyset.h>
#include <AlPolysetVertex.h>
#include <AlPolysetNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlCommand.h>
#include <AlUserCommand.h>
#include <AlNotifyDagNode.h>
#include <AlPickList.h>
#include <AlTM.h>

#include <fstream>
#include <iostream>

using namespace std;

//
//	This code provides an example of a construction history command.
//	It constrains a CV to a fixed position by 'undoing' any changes
//	that are made to the position of the CV.
//
//	It examines the new world space position of the CV, calculates teh
//	the vector (in world space) to move it back to its original position
//	then transforms this change back into local coordinates.
//
//	Note the use of the 'dagModified' function.
//	Whenever the dag node is modified, we re-invert our matrix used
//	to transform the world to local coordinates.
//	This is significantly cheaper than doing it for each CV move.
//

class Position
{
public:
	double x, y, z, w;
	
	Position& operator =( const Position& p )
	{
		x = p.x; y = p.y; z = p.z; w = p.w;
		return *this;
	}
	int operator ==( const Position& p ) const
	{
		return x == p.x && y == p.y && z == p.z && w == p.w;
	}
};

#define CVCMD_CLASS_ID	42

class cvCmdData
{
public:
	cvCmdData*			cvCmdDataPtr()	{ return this; }
	AlObject*			cv;
	AlDagNode*			dagNode;
	Position			pos;
	Position			originalPos;
	AlTM				worldToLocal;
};

class cvCmd: public AlUserCommand, private cvCmdData
{
	enum error { kOkay = 0, kInvalid = 1, kDagNodeNotHandled };

public:
	cvCmd();
	~cvCmd();

    virtual int         isValid();
    virtual int         execute();
    virtual int         declareReferences();
    virtual int         instanceDag( AlDagNode *oldDag, AlDagNode *newDag );
	virtual int			undo();

    virtual int         geometryModified( AlDagNode *dag );
    virtual int         listModifiedDagNodes( const AlNotifyDagNode *dagMod, AlObject *obj );
    virtual int         debug( const char *prefix );
    virtual void *      asDerivedPtr();

    virtual statusCode  retrieveWire( AlInput *input );
    virtual statusCode  storeWire( AlOutput *output );

	virtual int dagModified( AlDagNode *dag );

    // for your own use
    virtual int         type();

	// we didn't bother to implement these commands since they don't apply
	// to us (they would be similar to instanceDag && geometryModified)
	//
    // virtual int curveOnSurfaceModified( AlCurveOnSurface *surf );

	// The following command is not yet supported
	// virtual statusCode  storeSDL( ofstream &outputSDL );

public:
	// methods that the UI commands can use to set our fields
	statusCode			set( AlObject *newCV, AlDagNode *newDag );

	//	own utility functions
	static AlDagNode* parentDagNode( AlObject *cv );
	static statusCode unaffectedPosition( AlObject* cv, Position& pos );
	static statusCode worldPosition( AlObject* cv, Position& pos );
};

cvCmd::cvCmd()
//
//	Initialize the structure
//
{
	dagNode = NULL;
	cv = NULL;
	pos.x = 0;
	pos.y = 0;
	pos.z = 0;
	pos.w = 1;
}

cvCmd::~cvCmd()
//
// nothing to clean up
//
{
}

void *cvCmd::asDerivedPtr()
//
//	Provide safe down casting.  This isn't really necessary (a simple
//	typecast should work).
//
{
	return this;
}

int cvCmd::type()
//
//	User defined value so you can determine what your class type is.
//
{
	return CVCMD_CLASS_ID;
}

int cvCmd::isValid()
//
//	Figure out if the command is valid
//
{
	if( dagNode == NULL || cv == NULL )
		return kInvalid;

	switch( dagNode->type() )
	{
		case kCurveNodeType:
		case kSurfaceNodeType:
		case kPolysetNodeType:
			return kOkay;

		default:
			return kDagNodeNotHandled;
	}
}

statusCode cvCmd::worldPosition( AlObject* cv, Position& pos )
{
	// get the world coordinates for the CV
	switch( cv->type() )
	{
		case kCurveCVType:
		{
			cv->asCurveCVPtr()->worldPosition( pos.x, pos.y, pos.z, pos.w );
			break;
		}
		case kSurfaceCVType:
		{
			cv->asSurfaceCVPtr()->worldPosition( pos.x, pos.y, pos.z, pos.w );
			break;
		}
		case kPolysetVertexType:
		{
			cv->asPolysetVertexPtr()->worldPosition( pos.x, pos.y, pos.z );
			pos.w = 1;
			break;
		}
		default:
			return sFailure;
	}
	return sSuccess;
}

statusCode cvCmd::unaffectedPosition( AlObject* cv, Position& pos )
{
	// get the world coordinates for the CV
	switch( cv->type() )
	{
		case kCurveCVType:
		{
			cv->asCurveCVPtr()->unaffectedPosition( pos.x, pos.y, pos.z, pos.w);
			break;
		}
		case kSurfaceCVType:
		{
			cv->asSurfaceCVPtr()->unaffectedPosition( pos.x, pos.y,pos.z,pos.w);
			break;
		}
		case kPolysetVertexType:
		{
			cv->asPolysetVertexPtr()->unaffectedPosition( pos.x, pos.y, pos.z);
			pos.w = 1;
			break;
		}
		default:
			return sFailure;
	}
	return sSuccess;
}

AlDagNode *cvCmd::parentDagNode( AlObject *cv )
{
	AlDagNode *parent;
	switch( cv->type() )
	{
		case kCurveCVType:
		{
			AlCurve *curve = cv->asCurveCVPtr()->curve();
			if( curve )
			{
				parent = curve->curveNode();
				delete curve;
			}
			break;
		}

		case kSurfaceCVType:
		{
			AlSurface *surf = cv->asSurfaceCVPtr()->surface();
			if( surf )
			{
				parent = surf->surfaceNode();
				delete surf;
			}
			break;
		}

		case kPolysetVertexType:
		{
			AlPolyset *pset = cv->asPolysetVertexPtr()->polyset();
			if( pset )
			{
				parent = pset->polysetNode();
				delete pset;
			}
			break;
		}

		default:
			return (AlDagNode *)0;
	}
	return parent;
}

int cvCmd::execute()
//
//	Execute the command.  Note that the isValid() command is implicitly
//	called before this routine.
//
{
	Position world, change;
	worldPosition( cv, world );

	//	If the point didn't move, then don't do any work
	//
	if( world == pos )
		return 0;

	//
	//	Map the points (pos,world) in world coordinates back to
	//	(pos, local) in local coordinates
	//
	Position localPos = pos;
	worldToLocal.transPoint( localPos.x, localPos.y, localPos.z, localPos.w);

	Position local = world;
	worldToLocal.transPoint( local.x, local.y, local.z, local.w );

	// figure out the change in local coordinates
	//
	change.x = local.x - localPos.x;
	change.y = local.y - localPos.y;
	change.z = local.z - localPos.z;
	change.w = local.w - localPos.w;
	
	//	Now subtract off the change to put the point back to its
	//	original position
	//
	Position newPos;
	unaffectedPosition( cv, originalPos );
	newPos.x = originalPos.x - change.x;
	newPos.y = originalPos.y - change.y;
	newPos.z = originalPos.z - change.z;
	newPos.w = originalPos.w - change.w;

	//	Now move the vertex
	switch( cv->type() )
	{
		case kCurveCVType:
		{
			AlCurveCV *curveCV = cv->asCurveCVPtr();
			curveCV->doUpdates( FALSE );
			curveCV->setUnaffectedPosition(	newPos.x, newPos.y,
											newPos.z, newPos.w );
			break;
		}
		case kSurfaceCVType:
		{
			AlSurfaceCV *surfCV = cv->asSurfaceCVPtr();
			surfCV->doUpdates( FALSE );
			surfCV->setUnaffectedPosition(	newPos.x, newPos.y,
											newPos.z, newPos.w );
			break;
		}
		case kPolysetVertexType:
		{
			AlPolysetVertex *polyCV = cv->asPolysetVertexPtr();
			polyCV->doUpdates( FALSE );
			polyCV->setUnaffectedPosition(	newPos.x, newPos.y,
											newPos.z );
			break;
		}
		default:
			break;
	}
	dagNode->updateDrawInfo();
	return 0;
}

int cvCmd::instanceDag( AlDagNode *oldDag, AlDagNode *newDag )
//
//	Handle a dag node being instanced.  Go through the class and replace
//	any references to oldDag with newDag
//
{
	if( AlAreEqual( dagNode, oldDag ) )
	{
		//	Toss our old wrapper and replace it with a new one
		//	Which is a copy the new one
		delete dagNode;
		dagNode = newDag->copyWrapper()->asDagNodePtr();
	}

	return 0;
}

int cvCmd::declareReferences()
//
//	Declare any reference to constructors
//	Declare any references to dagnodes that may be destructors
//
{
	addConstructorRef( dagNode );
	return 0;
}

int cvCmd::geometryModified( AlDagNode *dag )
{
	AlDagNode *parent = parentDagNode( cv );

	if( AlAreEqual( parent, dag) )
	{
		delete parent;
		return 0;
	}

	// the cv isn't there anymore
	// Make the command invalid and modified.  The next time the history
	// list is executed, the command will be freed.
	delete cv;	cv = NULL;
	delete dagNode;	dagNode = NULL;

	// signal that the command has been modified by call
	AlCommand *cmd = command();
	cmd->modified();
	delete cmd;

	return 0;	
}

int cvCmd::dagModified( AlDagNode *dag )
//
//	Our cv's dag node might be changed (i.e. its transformation changes)
//	In this case, we need to update our worldToLocal transformation matrix
//
{
	if( AlIsValid( dag ) && AlIsValid( dagNode ) &&
		AlAreEqual( dag, dagNode ))
	{
		AlTM	dagTM;
		dagNode->globalTransformationMatrix( dagTM );
		worldToLocal = dagTM.inverse();
	} 
	return 0;
}

static void PrintPos( const char *string, Position &pos )
{
	cerr << string << "(";
	cerr << pos.x << ",";
	cerr << pos.y << ",";
	cerr << pos.z << ",";
	cerr << pos.w << ")" << endl;
}

int cvCmd::debug( const char *prefix )
{
	const char *dag_name = dagNode->name();
	const char *cv_name = cv->name();
	Position pw;

	cerr << prefix	<< "constrain-CV" << endl;
	cerr << prefix	<< "Dag-Node " << "(" << dag_name << ")" << endl;

	const char *typeName = "Unknown";
	switch( cv->type() )
	{
		case kCurveCVType:
			cv->asCurveCVPtr()->unaffectedPosition( pw.x, pw.y, pw.z, pw.w );
			typeName = "CurveCV";
			break;

		case kSurfaceCVType:
			cv->asSurfaceCVPtr()->unaffectedPosition( pw.x, pw.y, pw.z, pw.w );
			typeName = "SurfaceCV";
			break;

		case kPolysetVertexType:
			cv->asPolysetVertexPtr()->unaffectedPosition( pw.x, pw.y, pw.z );
			typeName = "PolysetCV";
			break;
	}
	cerr << prefix << prefix << typeName << "(" << cv_name << ")" << endl;

	PrintPos( "Unaffected pos ", pw );
	PrintPos( "Position ", pos );
	return 0;
}

int cvCmd::undo()
//
//	Undo everything the 'execute' did.
//
{
	switch( cv->type() )
	{
		case kCurveCVType:
			cv->asCurveCVPtr()->doUpdates( FALSE );
			cv->asCurveCVPtr()->setUnaffectedPosition(
				originalPos.x, originalPos.y, originalPos.z, originalPos.w );
			break;

		case kSurfaceCVType:
			cv->asSurfaceCVPtr()->doUpdates( FALSE );
			cv->asSurfaceCVPtr()->setUnaffectedPosition(
				originalPos.x, originalPos.y, originalPos.z, originalPos.w );
			break;

		case kPolysetVertexType:
			cv->asPolysetVertexPtr()->doUpdates( FALSE );
			cv->asPolysetVertexPtr()->setUnaffectedPosition(
				originalPos.x, originalPos.y, originalPos.z );
			break;

		default:
			break;
	}
	return 0;
}

statusCode cvCmd::set( AlObject *newCV, AlDagNode *newDag )
//
//	This method lets the UI code set one of our fields (the curve CV).
//	For our test class cvCmd, we have chosen not to share wrappers between
//	the UI and the history command, so we make a copy.
//
{
	if( cv == NULL && dagNode == NULL &&
		AlIsValid( newCV ) && AlIsValid( newDag ) &&
		(	newDag->asCurveNodePtr() ||
			newDag->asSurfaceNodePtr() ||
			newDag->asPolysetNodePtr() ) )	
	{
		AlObject *cvCopy;
		if( cvCopy = newCV->copyWrapper() )
		{
			AlObject *dagCopy;
			if( dagCopy = newDag->copyWrapper() )
			{
				dagNode = dagCopy->asDagNodePtr();
				cv = cvCopy;

				//	Get the transformation matrix
				//
				AlTM	dagTM;
				dagNode->globalTransformationMatrix( dagTM );
				worldToLocal = dagTM.inverse();

				// save the original world space location of the CV
				return worldPosition( cv, pos );
			}
			delete cvCopy;
		}
	}
	return sFailure;
}

int cvCmd::listModifiedDagNodes( const AlNotifyDagNode *dagMod, AlObject *obj )
//
//	This routine should call dagMod->notify on every dag node that might
//	be affected if obj is modified.  In our example, nothing should change,
//	but for illustrative purposes assume that modifying our CV changes the
//	dag node.
//
{
	if( AlAreEqual( obj, cv ) )
	{
		//
		// if( sSuccess == dagMod->notify( dag ))
		//		return 0;
		dagMod = dagMod;
		return 0;
	}
	return -1;
}

statusCode cvCmd::retrieveWire( AlInput *input )
//
//	This routine copies the saved data chunk from the wire file.
//	Pointers are dealt with in the resolve routine.
//
{
	// replace the old pointers with the new ones.  use
	// the same order as in storeWire().
	AlObject *objDag = input->resolveObject();
	AlObject *objCV =  input->resolveObject();

	// if a pointer was not resolved, then NULL is returned
	//
	// This can happen if the geometry has been deleted on you
	// (the geometry was modified and the plugin was not around)
	// Alternatively, we could have just returned sSuccess.
	// When our command is executed, it will be invalid and so it will be
	// deleted.

	dagNode = objDag ? objDag->asDagNodePtr() : NULL; 
	cv  = objCV;

	if( dagNode && cv )
		return sSuccess;
	else
		return sFailure;
}

statusCode cvCmd::storeWire( AlOutput *output )
//
//	This routine writes out our data to the wire file.  It also translates
//	the pointers into handles.  This routine is to be called ONCE with
//	your final data block.
//
{
	// declare all of our references to data so that we can get them back
	// on retrieval
	output->declareObject( dagNode );
	output->declareObject( cv );
	return sSuccess;
}

//
//	End command definition
//

//
//	Begin command invocation (the UI chunk of the code) 
//
void do_add_cmd( AlObject *cv )
{
	AlCommand::setDebug( TRUE );
	AlCommand cmd;

	if( sSuccess == cmd.create( "myCmd" ) )
	{
		cvCmd *cv_cmd = (cvCmd *)cmd.userCommand()->asDerivedPtr();

		AlDagNode *parent = cvCmd::parentDagNode( cv );
		if( parent )
		{
			cv_cmd->set( cv, parent );
			if( cmd.execute() == 0 )
				cmd.install();
			else
				cmd.deleteObject();
			delete parent;
		}
	}
}

void do_ConstrainPickedCV()
{
	for(	statusCode stat = AlPickList::firstPickItem();
			stat == sSuccess;
			stat = AlPickList::nextPickItem() )
	{
		AlObject *pickedItem = AlPickList::getObject();
		if( pickedItem )
		{
			if( pickedItem->asCurveCVPtr() ||
				pickedItem->asSurfaceCVPtr() ||
				pickedItem->asPolysetVertexPtr() )
			{
				do_add_cmd( pickedItem );
			}
			delete pickedItem;
		}
	}
}

AlUserCommand *allocCVCmd()
//
//	Allocate & return a new cv command.  This function will be passed
//	to the AlCommand::add routine
//
{
	return new cvCmd;
}

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
    AlUniverse::initialize( kZUp );

	AlCommand::add(  allocCVCmd, "myCmd" );

	hFunc.create( do_ConstrainPickedCV );

    h.create( "Test plugin command", &hFunc );
    h.setAttributeString( "plugin cmd" );
    h.setIconPath( makeAltPath( dirName, NULL ) );
    h.appendToMenu( "mp_objtools" );

	AlPrintf( kPrompt, "Constrain CV installed on Palette 'Object Edit/ConstrainCV'");
    return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	AlCommand::remove( "myCmd" );
    h.deleteObject();
	hFunc.deleteObject();

    // do nothing
    return 0;
}

