/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//  distance.plugin
//
//  This plugin is meant to demonstrate the AlMinDist function for determining
//  the minimum distance between any two points, curves, and/or surfaces.
//

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlPickList.h>
#include <AlCurve.h>
#include <AlMeasure.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>
#include <AlCurveNode.h>

// Functions to determine which of five valid selection lists exists (if any)

boolean two_curves_selected(AlCurveNode** crv1, AlCurveNode** crv2)
  {
	if (sSuccess!=AlPickList::firstPickItem())
	  return FALSE;
	AlObject* obj=AlPickList::getObject();
	if (!obj || obj->type()!=kCurveNodeType)
	  return FALSE;
	*crv1=obj->asCurveNodePtr();

	if (sSuccess!=AlPickList::nextPickItem())
	  return FALSE;
	obj=AlPickList::getObject();
	if (!obj || obj->type()!=kCurveNodeType)
	  return FALSE;
	*crv2=obj->asCurveNodePtr();
	return TRUE;
  }

boolean one_curve_selected(AlCurveNode** crv)
  {
	if (sSuccess!=AlPickList::firstPickItem())
	  return FALSE;
	AlObject* obj=AlPickList::getObject();
	if (!obj || obj->type()!=kCurveNodeType)
	  return FALSE;
	*crv=obj->asCurveNodePtr();
	return TRUE;
  }

boolean one_surface_selected(AlSurfaceNode** srf)
  {
	if (sSuccess!=AlPickList::firstPickItem())
	  return FALSE;
	AlObject* obj=AlPickList::getObject();
	if (!obj || obj->type()!=kSurfaceNodeType)
	  return FALSE;
	*srf=obj->asSurfaceNodePtr();
	return TRUE;
  }

boolean two_surfaces_selected(AlSurfaceNode** srf1, AlSurfaceNode** srf2)
  {
    if (sSuccess!=AlPickList::firstPickItem())
      return FALSE;
    AlObject* obj=AlPickList::getObject();
    if (!obj || obj->type()!=kSurfaceNodeType)
      return FALSE;
    *srf1=obj->asSurfaceNodePtr();

    if (sSuccess!=AlPickList::nextPickItem())
      return FALSE;
    obj=AlPickList::getObject();
    if (!obj || obj->type()!=kSurfaceNodeType)
      return FALSE;
    *srf2=obj->asSurfaceNodePtr();
    return TRUE;
  }


boolean surface_and_curve_selected(AlSurfaceNode** srf, AlCurveNode** crv)
{
	if (sSuccess!=AlPickList::firstPickItem())
	  return FALSE;
	AlObject* obj=AlPickList::getObject();
	if (!obj || obj->type()!=kSurfaceNodeType)
	  {
	    if (!obj || obj->type()!=kCurveNodeType)
		  return FALSE;
		*crv=obj->asCurveNodePtr();

		if (sSuccess!=AlPickList::nextPickItem())
	  	  return FALSE;
		obj=AlPickList::getObject();
		if (!obj || obj->type()!=kSurfaceNodeType)
	  	  return FALSE;
		*srf=obj->asSurfaceNodePtr();
	  }
	else
	  {
		*srf=obj->asSurfaceNodePtr();

		if (sSuccess!=AlPickList::nextPickItem())
	  	  return FALSE;
		obj=AlPickList::getObject();
		if (!obj || obj->type()!=kCurveNodeType)
	  	  return FALSE;
		*crv=obj->asCurveNodePtr();
	  }
	return TRUE;
}


// Examples of the five different AlMinDist() function calls


void do_minimum_dist( void )
{
	AlCurveNode *curveNode1 = NULL, *curveNode2 = NULL;
	AlSurfaceNode *surfNode1 = NULL, *surfNode2 = NULL;
	if (two_curves_selected(&curveNode1,&curveNode2))
	{
		double t1, t2;
		double P1[3], P2[3];
		AlCurve *curve1 = curveNode1->curve();
		AlCurve *curve2 = curveNode2->curve();
		double dist=AlMeasure::minDist(curve1,curve2,&t1,P1,&t2,P2);
		AlPrintf(kPrompt,	"MinDist=%2.2f t1=%2.2f t2=%2.2f "
							"P1=%2.2f,%2.2f,%2.2f P2=%2.2f,%2.2f,%2.2f",
						  dist,t1,t2,P1[0],P1[1],P1[2],P2[0],P2[1],P2[2]);
		delete curve1;
		delete curve2;
	  }
	else if (surface_and_curve_selected(&surfNode1,&curveNode1))
	  {
		double u,v,t;
		double P1[3],P2[3];
		AlCurve*	curve1 = curveNode1->curve();
		AlSurface*	surf1 = surfNode1->surface();

		double dist=AlMeasure::minDist(surf1, curve1, &u,&v,P1,&t,P2);
		AlPrintf(kPrompt, "MinDist=%2.2f u=%2.2f v=%2.2f P1=%2.2f,%2.2f,%2.2f "
						 "t=%2.2f P2=%2.2f,%2.2f,%2.2f", 
				          dist,u,v,P1[0],P1[1],P1[2],t,P2[0],P2[1],P2[2]);
		delete curve1;
		delete surf1;
	  }
	else if (two_surfaces_selected(&surfNode1,&surfNode2))
	  {
		double u1,v1,u2,v2;
		double P1[3],P2[3];
		AlSurface*	surf1 = surfNode1->surface();
		AlSurface*	surf2 = surfNode2->surface();
		double dist=AlMeasure::minDist(surf1, surf2, &u1,&v1,P1,&u2,&v2,P2);
	  AlPrintf(kPrompt,"MinDist=%2.2f u1=%2.2f v1=%2.2f P1=%2.2f,%2.2f,%2.2f "
						"u2=%2.2f v2=%2.2f P2=%2.2f,%2.2f,%2.2f", 
				        dist,u1,v1,P1[0],P1[1],P1[2],u2,v2,P2[0],P2[1],P2[2]);
		delete surf1;
		delete surf2;
	  }
	else if (one_curve_selected(&curveNode1))
	  {
		double P1[3]={0.0,0.0,0.0};
		double t;
		double P2[3];
		AlCurve *curve1 = curveNode1->curve();
		double dist=AlMeasure::minDist(curve1,P1,&t,P2);
		AlPrintf(kPrompt,"MinDist=%2.2f t=%2.2f P2=%2.2f,%2.2f,%2.2f",
				          dist,t,P2[0],P2[1],P2[2]);
		delete curve1;
	  }
	else if (one_surface_selected(&surfNode1))
	  {
		double P1[3]={0.0,0.0,0.0};
		double u,v;
		double P2[3];
		AlSurface*	surf1 = surfNode1->surface();
		double dist=AlMeasure::minDist(surf1,P1,&u,&v,P2);
		AlPrintf(kPrompt,"MinDist=%2.2f u=%2.2f v=%2.2f P2=%2.2f,%2.2f,%2.2f",
				          dist, u, v, P2[0], P2[1], P2[2]);
		delete surf1;
	  }
	else
	  {
		AlPrintf(kPrompt,"Please select one or two curves or surfaces.");
	  }
		
	delete curveNode1;
	delete curveNode2;
	delete surfNode1;
	delete surfNode2;
}

static AlMomentaryFunction hFunc;
static AlFunctionHandle handle;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize( kZUp );

	hFunc.create( "minimum_dist", do_minimum_dist );

	handle.create( "Minimum_dist", &hFunc );
    handle.setAttributeString( "Closest Distance" );
    handle.appendToMenu( "al_goto" );
    handle.setIconPath( makeAltPath( dirName, NULL ) );
	AlPrintf( kPrompt, "Closest Distance on Menu 'Utilities'");

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	handle.deleteObject();
    hFunc.deleteObject();
	return 0;
}
