/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/


#include <string.h>
#include <AlUniverse.h>
#include <AlDagNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlIterator.h>
#include <AlBlindData.h>

#define MYTAG	42

extern char * makeAltPath(const char *dirName, const char *suffix);

// Here we derive our own blind data class from AlBlindData. This is
// necessary for each type of blind data packet. The purpose of this
// derived class is to give form to the blind data.
class myBlindData : private AlBlindData
{
		// We define a class which describes the blind data. Pointers are
		// not permitted, but arrays are. Pointers are not permitted as
		// they cannot be recreated between sessions. If you need to
		// reference another object you will need another mechanism to
		// reference them (such as names, some other unique identifier).
		//
		// The names of the data members are not important, and are
		// named as such below just for clarity.
		class blindData
		{
			public:
				int intMember;
				float floatMember;
				short shortMember;
				double doubleMember;
				char characterMember[20];
		};

	public:
		// These next three methods override the base class methods.
		myBlindData(AlObject *obj, int userID);
		myBlindData(int userID);
		statusCode create();

		// A convenience method for checking private data.
		bool hasData() { return (bd != 0); }

		// Convenience methods for accessing the blind data. It is preferrable
		// to use access methods to reference the data as it can be deleted
		// or otherwise made unavailable without your application being
		// informed. The convenience methods check the validity of the
		// data before operating on it.
		statusCode getIntMember(int&) const;
		statusCode getShortMember(short&) const;
		statusCode getFloatMember(float&) const;
		statusCode getDoubleMember(double&) const;
		statusCode getCharacterMember(char *&) const;
		statusCode setIntMember(const int);
		statusCode setShortMember(const short);
		statusCode setFloatMember(const float);
		statusCode setDoubleMember(const double);
		statusCode setCharacterMember(const char*);

	private:
		// A private reference to the blind data.
		blindData *bd;
};

// A constructor for associating blind data with an AlUniverse. The userID
// must be unique and is generally assigned to you by Alias Systems
myBlindData::myBlindData(int userID)
	:AlBlindData(userID, sizeof(blindData)), bd(0)
{
	// Check whether blind data with this userID already exists on the
	// AlUniverse.
	bd = (blindData*)getData();

	if (bd)
	{
		// If it does, make sure that the data members are converted to
		// the endianness of this platform.
		convert(bd->intMember);
		convert(bd->shortMember);
		convert(bd->doubleMember);
		convert(bd->floatMember);

		// Once the conversions are complete tell the blind data to
		// remember the endianness of the current platform. Failure to
		// do this last (or at all) will result in your data being
		// incorrect when moved to another platform.
		setEndian();    // Must always be done, and done last.
	}
}

// A constructor for associating blind data with an AlObject. The userID
// must be unique and is generally assigned to you by Alias Systems.
// Not all AlObjects support blind data. Those which currently do are:
// AlDagNode, AlCurveCV, AlSurfaceCV, AlCurveOnSurface, AlPolygon,
// AlPolyset, AlPolysetVertex, AlTrimRegion, and AlShader.
myBlindData::myBlindData(AlObject *obj, int userID)
	:AlBlindData(obj, userID, sizeof(blindData)), bd(0)
{
	// Check whether blind data with this userID already exists on the
	// AlObject.
	bd = (blindData*)getData();

	if (bd)
	{
		// If it does, make sure that the data members are converted to
		// the endianness of this platform.
		convert(bd->intMember);
		convert(bd->shortMember);
		convert(bd->doubleMember);
		convert(bd->floatMember);

		// Once the conversions are complete tell the blind data to
		// remember the endianness of the current platform. Failure to
		// do this last (or at all) will result in your data being
		// incorrect when moved to another platform.
		setEndian();    // Must always be done, and done last.
	}
}

// The create method will create new instances of the blind data on
// the AlObject or AlUniverse referenced by this object if the blind
// data doesn't already exist. If it does, this method fails.
statusCode myBlindData::create()
{
	if (!bd)
	{
		// Note that this test isn't really necessary as AlBlindData::create
		// does much the same thing.
		statusCode results = AlBlindData::create();
		bd = (blindData*)getData();
		return results;
	}
	else
		return sAlreadyCreated;
}

// The following methods access the various elements of the blind data
// and return their values. It is safer to use access methods than to access
// the data directly as the data could be deleted (when the parent object
// is deleted) or be unavailable (the universe in which it exists is not
// current).
statusCode myBlindData::getIntMember(int &val) const
{
	if (!isValid() || !bd)
		return sInvalidObject;

	val = bd->intMember;
	return sSuccess;
}

statusCode myBlindData::getShortMember(short &val) const
{
	if (!isValid() || !bd)
		return sInvalidObject;

	val = bd->shortMember;
	return sSuccess;
}

statusCode myBlindData::getFloatMember(float &val) const
{
	if (!isValid() || !bd)
		return sInvalidObject;

	val = bd->floatMember;
	return sSuccess;
}

statusCode myBlindData::getDoubleMember(double &val) const
{
	if (!isValid() || !bd)
		return sInvalidObject;

	val = bd->doubleMember;
	return sSuccess;
}

statusCode myBlindData::getCharacterMember(char *&val) const
{
	if (!isValid() || !bd)
		return sInvalidObject;

	// Must ensure that val points to enough memory to hold the
	// copy of characterMember.
	val = strdup(bd->characterMember);
	return sSuccess;
}

// As with the get methods above it is safer to use set methods to set the
// values of the blind data elements.
statusCode myBlindData::setIntMember(const int val)
{
	if (!isValid() || !bd)
		return sInvalidObject;

	bd->intMember = val;
	return sSuccess;
}

statusCode myBlindData::setShortMember(const short val)
{
	if (!isValid() || !bd)
		return sInvalidObject;

	bd->shortMember = val;
	return sSuccess;
}

statusCode myBlindData::setFloatMember(const float val)
{
	if (!isValid() || !bd)
		return sInvalidObject;

	bd->floatMember = val;
	return sSuccess;
}

statusCode myBlindData::setDoubleMember(const double val)
{
	if (!isValid() || !bd)
		return sInvalidObject;

	bd->doubleMember = val;
	return sSuccess;
}

statusCode myBlindData::setCharacterMember(const char *val)
{
	if (!isValid() || !bd)
		return sInvalidObject;

	// Should additionally do a test to ensure that the string
	// pointed to by val will fit in the statically allocated
	// character array.
	if (val)
		strcpy(bd->characterMember, val)
;
	return sSuccess;
}

// This is a simple function which associates blind data with the
// current AlUniverse. If the blind data already exists its contents
// are printed, otherwise a new packet is added.
void attachToUniverse( void )
{
	myBlindData *data = new myBlindData(MYTAG);
	if (data->hasData())
	{
		AlPrintf( kPrompt, "The universe already has blind data (%d).", MYTAG);
		int i;
		short s;
		float f;
		double d;
		char *c;
		data->getIntMember(i);
		data->getShortMember(s);
		data->getFloatMember(f);
		data->getDoubleMember(d);
		data->getCharacterMember(c);

		printf("Int member has value %d\n", i);
		printf("Short member has value %d\n", (int)s);
		printf("Float member has value %f\n", (double)f);
		printf("Double member has value %f\n", d);
		printf("Character member has value %s\n", c);

		free(c);

		return;
	}

	AlPrintf( kPrompt, "Adding blind data to universe." );

	// Create the blind data on the object.
	data->create();

	data->setIntMember(42);
	data->setShortMember(-42);
	data->setFloatMember(42.0);
	data->setDoubleMember(-42.42);
	data->setCharacterMember("Forty two.");
}

// This is a simple function which associates blind data with the
// first AlDagNode in the current universe. If the blind data already
// exists its contents are printed, otherwise a new packet is added.
void attachToDagNode( void )
{
	AlDagNode   *dag = AlUniverse::firstDagNode();

	if ( AlIsValid( dag ) )
	{
		myBlindData *data = new myBlindData(dag, MYTAG);
		if (data->hasData())
		{
			AlPrintf( kPrompt, "%s already has blind data (%d).",
				dag->name(), MYTAG);
			int i;
			short s;
			float f;
			double d;
			char *c;
			data->getIntMember(i);
			data->getShortMember(s);
			data->getFloatMember(f);
			data->getDoubleMember(d);
			data->getCharacterMember(c);

			printf("Int member has value %d\n", i);
			printf("Short member has value %d\n", (int)s);
			printf("Float member has value %f\n", (double)f);
			printf("Double member has value %f\n", d);
			printf("Character member has value %s\n", c);

			free(c);

			return;
		}

		AlPrintf( kPrompt, "Adding blind data to %s.", dag->name() );

		// Create the blind data on the object.
		data->create();

		data->setIntMember(42);
		data->setShortMember(-42);
		data->setFloatMember(42.0);
		data->setDoubleMember(-42.42);
		data->setCharacterMember("Forty two.");
	}
}

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;
static AlFunctionHandle h2;
static AlMomentaryFunction h2Func;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	hFunc.create( "attach_blinddata", attachToDagNode );
	h2Func.create( "attach_universeblinddata", attachToUniverse );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	h.create( "Attach Blinddata", &hFunc ); 
	h2.create( "Attach Universe Blinddata", &h2Func ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "add blinddata" );
	h2.setAttributeString( "add universe blinddata" );

	h2.setIconPath( makeAltPath( dirName, NULL ) );

	// Indicate which menu to add the plugin to. 
	h.installOnMenu( "mp_objtools", FALSE /* bottom */ );
	h2.installOnMenu( "mp_objtools", FALSE /* bottom */ );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.

    AlPrintf( kPrompt, "Blinddata plug-in installed on Palette 'Object Edit'." );

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h2.deleteObject();
	h2Func.deleteObject();
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}

