/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//
//
//	This is a sample OpenAlias plugin which will essentially perform a
//	'Use results' for expressions, that is it will convert expressions
//	into keyframed action curves.  In general, it is difficult to determine
//	how to properly evaluate an expression and convert it into an action
//	curve.  Simple expressions based upon mathematical functions may be
//	converted by evaluating the function over time (if the function is
//	dependent upon time).  However, within Alias it is possible to create
//	expressions which are dependent upon the position of other objects.
//
//	This plugin converts expressions into action curves by performing a
//	'view frame' operation, and then sampling the current value of each
//	expression, which is then turned into a keyframe.  The user can set
//	both the frame range to evaluate the expressions, as well as the sampling
//	frequency, via an option box.
//
//	The plugin operates in three seperate stages: collect a list of all
//	expressions; perform a 'view frame' and sample the current value;
//	and finally create keyframed actions for the expression channels.
//
//	Collecting a list of expressions is done by walking through either the
//	list of all channels and looking for expression channels (if the user
//	wants ALL objects to participate), or by walking through each picked
//	object and finding any expression channels attached to it.  At the
//	same time, a new AlParamAction is created for each expression, which
//	will be used to accumulate the sampled keyframes.
//
//	Each frame is viewed using the AlPlayFrame class, which allows for the
//	quickest form of view frame (as long as we obey the rules for using the
//	AlPlayFrame class).  Since we do not know which objects may affect an
//	expression, we will just perform a view frame for the whole universe.
//	Once the view frame has been done (expressions are automatically updated
//	during the view frame) we just need to sample the current value for
//	each expression channel, and save the samples as keyframes in the
//	AlParamActions that we created.
//
//	Finally, once the requested frame range has been evaluated, each of the
//	expression channels is then replaced by an animation channel, which
//	has the keyframes we sampled as its AlParamAction curve.  (Note that
//	the expression DOES get replaced by the action curve, since Alias does
//	not have a simple means for allowing expressions and keyframed actions
//	to exist on the same channel)
//

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlChannel.h>
#include <AlParamAction.h>
#include <AlLinkItem.h>
#include <AlList.h>
#include <AlIterator.h>
#include <AlAnimatable.h>
#include <AlPlayFrame.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlConstraint.h>
#include <AlIKHandle.h>
#include <AlIKHandleNode.h>

#include <AlPerformance.h>
#include <AlPlayBack.h>
#include <AlJoint.h>


#include <WalkTree.h>
#include <string.h>
#include <awMath.h>
#include <AlRender.h>		// for getting blur shutter angle

#define PLUGIN_NAME "bakeAnimation"
#define PLUGIN_LABEL "Bake Animation"

// Our instance of the function handle
//
static AlFunctionHandle useExprHandle;
static AlMomentaryFunction	useExprFunc;

//==============================================================================
//
// STRUCTURE DECLARATIONS
//
//==============================================================================

// A structure to hold all the information in the option dialog
//
struct TexprOptionInfo
{

	boolean bExpressions;		// Should we process expressions?
	boolean bConstraints;		// Should we process constraints?

	int		iObjects;			// Which objects to operate upon
	int		iParams;			// Which param controls to use
	int		iHierarchy;			// Hierachy settings
	int		iRange;				// Frame range source
	double	dStart;				// Start frame
	double	dEnd;				// End frame
	double	dBy;				// Sample

	// options for compensation
	//
	boolean	bCompensation;		// Is the constraint compensation wanted?
	boolean	bMotionblur;		// Should we do compensation for motion blue?
	boolean bFields_rnd;		// Should we do compensation for fields render?
	boolean	bDetect_only;		// Is the compensation for detection only?

	double	dTranTol;			// Tolerance for translation.
	double	dRotTol;			// Tolerance for rotation.

	boolean	bDelete_constr;		// Delete constraints after bake?
};

class TuseResults;

// A static entry to hold the current hierarchy setting
//
static enumHier hierSetting;

//==============================================================================
//
// CLASS DECLARATIONS
//
//==============================================================================

// A linked list class to hold information about newly created actions
//
class TactionList : public AlLinkItem 
{
public:
	TactionList( void );
	virtual ~TactionList ();
	TactionList *next (void) {return ((TactionList *)AlLinkItem::next ());}

	void setChannel( AlObject * obj, int param );

	void destroyChannel( void );

	void cacheCompensationValue( int index );

	boolean	keyCompensationFrames( double, double, const double[5],
									const boolean[5], boolean );

	// data about the action properties
	//
	AlParamAction	*keyframeAction;		// The sampled action curve
	AlObject		*animatedItem;			// object
	int				parameter;				// object type

	// temp data used for compensation
	//
	boolean			compensate;				// This action needs compensation?
	double			com_tol;				// Tolerance for compensation

	double			p_value;				// Previous key value
	double			c_value;				// Current key value
	double			s_value[5];				// CompenSating values for shutter
											// close/open and fields
};

// A derived class to walk through the picked items
//
class TexprPickIterator : public AlIterator
{
public:
	TexprPickIterator( TuseResults* r, boolean c, boolean e ) 
		: results( r )
		, constraints( c )
		, expressions( e ) {}
	virtual ~TexprPickIterator () {}
	virtual int func (AlObject *pTheObject);

	TuseResults * results;
	boolean constraints;
	boolean expressions;
};

// A derived class to walk through an object's hierarchy
//
class TexprHierIterator : public AlIterator
{
public:
	TexprHierIterator( TuseResults * r, boolean c, boolean e )
		: results( r )
		, constraints( c )
		, expressions( e ) {}

	virtual ~TexprHierIterator () {}
	virtual int func (AlObject *pTheObject);

	TuseResults * results;
	boolean constraints;
	boolean expressions;
};

// Our actual useResults control class
//
class TuseResults
{
public:
	TuseResults ();
	~TuseResults ();

	statusCode eval( void );

	void collectActions( void );

	void findAllExpressions( void );
	void findAllConstraints( void );

	void findConstraintsAt( AlDagNode* );
	void findExpressionsAt( AlObject* );

	void createChannels (void);
	void viewCompensationTime( int index );

	void appendAction( AlChannel * chan, boolean bMarkCompensate );
	void appendAction( AlObject * obj, int param, boolean bMarkCompensate );

	boolean	sampleChannels( boolean bLastFrame, boolean bFirstFrame );
	boolean	detectOnly() const {return fExprOptionInfo.bDetect_only; };

private:
	TexprOptionInfo	fExprOptionInfo;
	AlList			*fBaseList;
	AlPlayFrame		*fPlayFrame;
	boolean			fNeedCompensation;

	// data used for eval()
	//
	boolean			fCompensate;	// do we need any compensation?

	double			fLastFrame;		// previous frame time
	double			fCurrFrame;		// current frame time
	double			fCompTime[5];	// time points for compensation
	boolean			fTimeValid[5];	// fCompTime[i] is valid or not
};

//==============================================================================
//
// TactionList METHODS
//
//==============================================================================

TactionList::TactionList( void )
		: keyframeAction( NULL )
		, animatedItem( NULL )
		, parameter( 0 ) 
		, compensate( FALSE )
		, com_tol( 0.0 )
		, p_value( 0.0 )
		, c_value( 0.0 )
{
	for (int i=0; i<5; i++)
	{
		s_value[i] = 0.0;
	}
}

TactionList::~TactionList()
{
	if( keyframeAction ) {
		delete keyframeAction;
	}

	if( animatedItem ) {
		delete animatedItem;
	}
}

void TactionList::setChannel( AlObject* obj, int param )
{
	if( AlIsValid( obj ) ) {
		animatedItem = obj;
		parameter = param;
	}
}

void TactionList::destroyChannel( void )
{
	if( !AlIsValid( animatedItem ) ) {
		return;
	}

	AlAnimatable* anim = animatedItem->asAnimatablePtr();
	AlChannel * cur = anim->findChannel( parameter );

	if( cur ) {
		cur->deleteObject();
		delete cur;
	}
}

void TactionList::cacheCompensationValue( int index )
{
	double	value;

	// get current value and cache it into s_value
	//
	if (AlIsValid( animatedItem ) && compensate )
	{
		AlAnimatable * anim = animatedItem->asAnimatablePtr();
		if ( anim )
		{
			value = anim->sampleChannel( parameter );
			s_value[index] = value;
		}
	}
}

boolean TactionList::keyCompensationFrames(
	double p_time,			// last frame
	double c_time,			// current frame
const double s_time[5],		// time points for compensation
const boolean s_valid[5],	// is s_time[i] valid or not
	boolean	detect_only)	// is "detect only"
{
	if ( !AlIsValid( animatedItem ) ||
		 !compensate				||
		 (fabs(p_time - c_time) <= FLT_MIN)	||
		 ( c_time < p_time ) )
	{
		return FALSE;
	}

    double      linearV, dT, dV;
    boolean		compensated;

    compensated = FALSE;

    /*
     * Insert a keyframe at s_time[i]:
     * If <s_time[i], s_value[i]> is not collinear with
     *    <s_time[i-1], s_value[i-1]> and <c_time, c_value> within the tolerance     * insert a keyframe with value of s_value[i].
     */
    dT = c_time - p_time;
    dV = c_value - p_value;

    for (int i=0; i<5; i++)
    {
        if (!s_valid[i])
            continue;

        linearV = c_value - ((dV * (c_time - s_time[i])) / dT);

        if (fabs(linearV - s_value[i]) > com_tol)
        {
            if (!detect_only)
            {
				keyframeAction->addKeyframe( s_time[i], s_value[i], FALSE,
											kTangentLinear, kTangentLinear);
                compensated = TRUE;
            }
            else
            {
                compensated = TRUE;
                break;
            }

            /* prepare for next loop */
            dT = c_time - s_time[i];
            dV = c_value - s_value[i];
        }
    }

	return compensated;
}


//==============================================================================
//
// TexprPickIterator METHODS
//
//==============================================================================

/*
======================================================================
TexprPickIterator::func() - PickList iterator function to extract expression
                           channels from each picked object
----------------------------------------------------------------------
Parameters:
    pTheObject      AlObject *          Current object being walked
Returns:
    Nothing
======================================================================
*/
int
TexprPickIterator::func (AlObject *pTheObject)
{
	TexprHierIterator *hierWalk = new TexprHierIterator( 
		results, constraints, expressions );

	// Walk down the hierarchy for this object
	//
	WalkTree::walkHierarchy (hierSetting, pTheObject, hierWalk);
	return (0);
}

//==============================================================================
//
// TexprHierIterator METHODS
//
//==============================================================================

/*
======================================================================
TexprHierIterator::func() - PickList iterator function to extract expression
                           channels from each picked object
----------------------------------------------------------------------
Parameters:
    pTheObject      AlObject *          Current object being walked
Returns:
    Nothing
======================================================================
*/
int
TexprHierIterator::func (AlObject *pTheObject)
{
	if (pTheObject == NULL) {
		return (0);
	}

	if( constraints ) {
		AlDagNode * dn = pTheObject->asDagNodePtr();
		if( dn ) {
			// If this DagNode is constrained, add it.
			results->findConstraintsAt( dn );
		}
	}

	if( expressions ) {
		results->findExpressionsAt( pTheObject );
	}

	return (0);
}

//==============================================================================
//
// TuseResults METHODS
//
//==============================================================================

/*
======================================================================
TuseResults::TuseResults() - Constructor for our
                       use expression results control object
----------------------------------------------------------------------
Parameters:
    None
Notes:
    The constructor will retrieve all the settings in the option dialog
Returns:
    Nothing
======================================================================
*/
TuseResults::TuseResults ()
:	fBaseList (NULL)
,	fPlayFrame(NULL)
,	fNeedCompensation( FALSE )
,	fCompensate( FALSE )
,	fLastFrame( 0.0 )
,	fCurrFrame( 0.0 )
{
	int res;

	// Read our option dialog settings
	//
	AlGetInteger( "ao_expr_results_expressions", res );
	fExprOptionInfo.bExpressions = res ? TRUE : FALSE;
	AlGetInteger( "ao_expr_results_constraints", res );
	fExprOptionInfo.bConstraints = res ? TRUE : FALSE;

	AlGetInteger ("ao_expr_results_objects", fExprOptionInfo.iObjects);
	AlGetInteger ("ao_expr_results_params", fExprOptionInfo.iParams);
	AlGetInteger ("ao_expr_results_hierarchy", fExprOptionInfo.iHierarchy);
	hierSetting = (enumHier)fExprOptionInfo.iHierarchy;
	AlGetInteger ("ao_expr_results_rangefrom", fExprOptionInfo.iRange);
	if (fExprOptionInfo.iRange == 1) {	// Frame range from the option box
		AlGetDouble ("ao_expr_results_startframe", fExprOptionInfo.dStart);
		AlGetDouble ("ao_expr_results_endframe", fExprOptionInfo.dEnd);
		AlGetDouble ("ao_expr_results_byframe", fExprOptionInfo.dBy);
	}
	else {								// Frame range from current time slider
		fExprOptionInfo.dStart = 0.0;
		fExprOptionInfo.dEnd = 30.0;
		fExprOptionInfo.dBy = 1.0;

		AlPlayBack::getStartEndBy(fExprOptionInfo.dStart,
								  fExprOptionInfo.dEnd,
								  fExprOptionInfo.dBy);
	}

	// Initialize options for compensation
	//
	fExprOptionInfo.bCompensation = FALSE;
	fExprOptionInfo.bMotionblur = FALSE;
	fExprOptionInfo.bFields_rnd	= FALSE;
	fExprOptionInfo.bDetect_only= FALSE;

	fExprOptionInfo.dTranTol = 0.0;
	fExprOptionInfo.dRotTol = 0.0;

	fExprOptionInfo.bDelete_constr = FALSE;

	// Read options for compensation
	//
	if (fExprOptionInfo.bConstraints)
	{
		AlGetInteger( "ao_bake_compensation", res );
		switch(res)
		{
			case	0:		// none	(as default)
				break;
			case	1:		// motion blur only
				fExprOptionInfo.bCompensation = TRUE;
				fExprOptionInfo.bMotionblur = TRUE;
				break;
			case	2:		// fields_render only
				fExprOptionInfo.bCompensation = TRUE;
				fExprOptionInfo.bFields_rnd = TRUE;
				break;
			case	3:		// both
				fExprOptionInfo.bCompensation = TRUE;
				fExprOptionInfo.bMotionblur = TRUE;
				fExprOptionInfo.bFields_rnd = TRUE;
				break;
		}

		if (fExprOptionInfo.bCompensation)
		{
			AlGetInteger( "ao_bake_cmp_detect", res );
			if (res)
				fExprOptionInfo.bDetect_only = TRUE;

			AlGetDouble( "ao_bake_cmp_transtol", fExprOptionInfo.dTranTol );
			AlGetDouble( "ao_bake_cmp_rottol", fExprOptionInfo.dRotTol);
		}

		AlGetInteger( "ao_bake_delete_constr", res );
		fExprOptionInfo.bDelete_constr = (res)? TRUE : FALSE;
	}
}

/*
======================================================================
TuseResults::TuseResults() - Destructor for our
                       use expression results control object
----------------------------------------------------------------------
Parameters:
    None
Notes:
    The destructor is responsible for freeing up the list of actions
    that we created a long the way, as well as delete'ing all the OpenAlias
    channel wrappers contained in the list
Returns:
    Nothing
======================================================================
*/
TuseResults::~TuseResults ()
{
	if( fBaseList ) {
		delete fBaseList;
	}
}

void TuseResults::appendAction( AlChannel *chan, boolean bMarkCompensate)
{
	TactionList * tal;

	if( !AlIsValid( chan ) ) {
		return;
	}

	int param = chan->parameter();
	AlObject * obj = chan->animatedItem();

	if( !fBaseList ) {
		fBaseList = new AlList;
	}

	for( tal = (TactionList*)(fBaseList->first());
			tal; tal = tal->next() ) {
		if( (tal->parameter == param) && AlAreEqual(obj, tal->animatedItem) ) {
			delete obj;
			return;
		}
	}
 
	tal = new TactionList;
	tal->keyframeAction = new AlParamAction;
	tal->setChannel( obj, param );

	if (bMarkCompensate)
	{
		switch (param)
		{
			case kFLD_DAGNODE_XTRANSLATE:
			case kFLD_DAGNODE_YTRANSLATE:
			case kFLD_DAGNODE_ZTRANSLATE:
				tal->com_tol = fExprOptionInfo.dTranTol;
				tal->compensate	= TRUE;
				break;
			case kFLD_DAGNODE_XROTATE:
			case kFLD_DAGNODE_YROTATE:
			case kFLD_DAGNODE_ZROTATE:
				tal->com_tol = fExprOptionInfo.dRotTol;
				tal->compensate	= TRUE;
				break;
		}

		fNeedCompensation = TRUE;
	}

	fBaseList->append( tal );
}

void TuseResults::appendAction(
	AlObject	*obj,
	int			param,
	boolean		bMarkCompensate )
{
	TactionList * tal;

	if( !AlIsValid( obj ) ) {
		return;
	}

	if( !fBaseList ) {
		fBaseList = new AlList;
	}
		
	for( tal = (TactionList*)(fBaseList->first());
			tal; tal = tal->next() ) {
		if( (tal->parameter == param) && AlAreEqual(obj, tal->animatedItem) ) {
			return;
		}
	}

	tal = new TactionList;
	tal->keyframeAction = new AlParamAction;
	tal->setChannel( obj->copyWrapper(), param );

	if (bMarkCompensate)
	{
		switch (param)
		{
			case kFLD_DAGNODE_XTRANSLATE:
			case kFLD_DAGNODE_YTRANSLATE:
			case kFLD_DAGNODE_ZTRANSLATE:
				tal->com_tol = fExprOptionInfo.dTranTol;
				tal->compensate	= TRUE;
				break;
			case kFLD_DAGNODE_XROTATE:
			case kFLD_DAGNODE_YROTATE:
			case kFLD_DAGNODE_ZROTATE:
				tal->com_tol = fExprOptionInfo.dRotTol;
				tal->compensate	= TRUE;
				break;
		}
		fNeedCompensation = TRUE;
	}

	fBaseList->append( tal );
}

/*
======================================================================
TuseResults::eval() - Control module to evaluate
                       all expressions in the known universe, and convert
                       them to keyframe curves
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
statusCode
TuseResults::eval()
{
	double	dBy = fExprOptionInfo.dBy;
	double	dEnd = fExprOptionInfo.dEnd;
	fCurrFrame = fExprOptionInfo.dStart;
	fLastFrame = fCurrFrame;

	boolean	bMB;			// to compensate MotionBlur
	boolean bHalfMB;		// the MB compensation is a half compensation
	boolean bFR;			// to compensate FieldsRendering
	boolean bDT;			// to detect if compensation needed
	double	mb_delta_time;	// delta time for MB compensation
	double	fr_delta_time;	// delta time for FR compensation

	int		iCompFrames, i;


	//.............................................................
	// Initialize for compensation
	//
	bMB	= fExprOptionInfo.bMotionblur;
	bFR = fExprOptionInfo.bFields_rnd;
	bDT	= fExprOptionInfo.bDetect_only;
	if (bDT && !(bMB || bFR))
	{
		AlPrintf (kPrompt, "There is no need for compensation.\n");
		return (sFailure);
	}

	//.............................................................
	// Create a list of the expressions and constraints that we are 
	// interested in
	//
	collectActions();
	if (fBaseList == NULL)
	{
		AlPrintf (kPrompt, "No expressions or constraints evaluated.\n");
		return (sFailure);
	}

	//.............................................................
	// Prepare for compensation
	//
	bMB	= bMB && fNeedCompensation;
	bFR = bFR && fNeedCompensation;
	fCompensate = bMB || bFR;

	bHalfMB = FALSE;
	mb_delta_time = 0.0;
	fr_delta_time = 0.0;
	iCompFrames = 0;

	if (bMB)
	{
		double	shutterAngle = AlRender::Blur::shutterAngle();
		if (shutterAngle > 0.0 && shutterAngle <= 360)
		{
			mb_delta_time = (dBy * shutterAngle)/720.0;
			if (fabs(shutterAngle - 360.0) <= fExprOptionInfo.dRotTol)
				bHalfMB = TRUE;

			if (bFR)
				mb_delta_time *= 0.5;
		}
		else
			bMB = FALSE;
	}

	if (bFR)
		fr_delta_time = dBy * 0.5;


	if (bDT && !fCompensate)
	{
		AlPrintf (kPrompt, "There is no need for compensation.\n");
		return (sFailure);
	}

	// setup valid time points
	//
	for (i=0; i<5; i++)
		fTimeValid[i] = FALSE;

	if (fCompensate)
	{
		if (bMB)
			fTimeValid[0] = TRUE;

		if (bMB && bFR && !bHalfMB)
			fTimeValid[1] = TRUE;

		if (bFR)
			fTimeValid[2] = TRUE;

		if (bMB && bFR && !bHalfMB)
			fTimeValid[3] = TRUE;

		if (bMB && (!bHalfMB || bFR))
			fTimeValid[4] = TRUE;

		// initialize fCompTime
		//
		// Setup compensating view time points for motion blur
		// and fields render compensation before next key frame.
		//
		// The compensating points are taken as followed rules:
		//
		//                        (fields point)
		//                (shutterOpen)  |  (shutterClose)
		//   (shutterClose)      |       |       |      (shutterOpen)
		//          V            V       V       V           V
		//  O-------x------------x-------X-------x-----------x-------O
		// last   comp0        comp1   comp2   comp3       comp4  (last+by)
		//


   		// fCompTime[2] is fields point
   		fCompTime[2] = fLastFrame + fr_delta_time;

   		// fCompTime[0] is shutter close for fLastFrame
   		fCompTime[0] = fLastFrame + mb_delta_time;

   		// comp_tiem[1] is shutter open for fields point
   		fCompTime[1] = fCompTime[2] - mb_delta_time;

   		// fCompTime[3] is shutter close for fields point
   		fCompTime[3] = fCompTime[2] + mb_delta_time;

   		// fCompTime[4] is shutter open for current key frame
   		fCompTime[4] = fLastFrame + dBy - mb_delta_time;
	}


	//.............................................................
	// Use the AlPlayFrame class to view each frame
	//
	boolean firstFrame = TRUE;
	fPlayFrame = new AlPlayFrame;
	while (fCurrFrame <= dEnd)
	{
		AlPrintf (kPromptNoHistory, "Evaluating frame %.3f\n", fCurrFrame);

		if (!firstFrame && fCompensate)
		{
			// Do extra views on the fCompTime[] points. And take snap
			// shot of compensational values into each action item;
			//
			for (i=0; i<5; i++)
			{
				if (fTimeValid[i])
					viewCompensationTime(i);
			}
		}

		// view current frame
		//
		fPlayFrame->viewFrame (fCurrFrame, TRUE);

	  	// Sample the current value of each expression
	  	//
		boolean bCompensated;
	  	bCompensated = sampleChannels ( (boolean)((fCurrFrame + dBy) > dEnd),
						 				firstFrame );


		if (bCompensated && bDT)
		{
			iCompFrames++;
			AlPrintf (kPrompt,
				"Need compensation between frames [%.2f, %.2f].\n",
				fLastFrame, fCurrFrame);
		}

		// prepare for next frame
		//
		fLastFrame = fCurrFrame;
		fCurrFrame += dBy;

		if (fCompensate && !firstFrame)
		{
			for (i=0; i<5; i++)
				fCompTime[i] += dBy;
		}

		firstFrame = FALSE;
	}

	// Delete the AlPlayFrame object, since we are about the delete and
	// create actions and channels
	//
	delete fPlayFrame;
	fPlayFrame = NULL;

	// Create the keyframed channels for each of the expressions
	//
	if ( bDT )
	{
		if ( iCompFrames > 0 )
			AlPrintf (kPrompt, "There are %d frames needing compensation.\n",
						iCompFrames);
		else
			AlPrintf (kPrompt, "There is no need for compensation.\n");
	}
	else
		createChannels ();

	return (sSuccess);
}

void TuseResults::findExpressionsAt( AlObject * obj )
{
	AlAnimatable *pAnimatedObject = obj->asAnimatablePtr ();
	if( pAnimatedObject ) {

		// Look at all the channels and add them if necessary
		AlChannel *pChannel = pAnimatedObject->firstChannel ();
		if( pChannel ) {
			do {
				if (pChannel->channelType () == kExprChannel) {
					appendAction( pChannel, FALSE );
				}
			} while( sSuccess == pAnimatedObject->nextChannelD (pChannel)) ;

			delete pChannel;
		}
	}
}

void TuseResults::findConstraintsAt( AlDagNode * dn )
{
	boolean point = FALSE;
	boolean orient = FALSE;
	boolean aim = FALSE;
	AlObjectType tp;

	if( !AlIsValid( dn ) ) {
		return;
	}

	AlConstraint * con = dn->firstConstraint();
	AlConstraint * tmp;

	// What kind of constraints are on this node?
	while( con ) {
		if( con->on() ) {
			tp = con->type();
			switch( tp ) {
			case kPointConstraintType:
				point = TRUE;
				break;
			case kOrientationConstraintType:
				orient = TRUE;
				break;
			case kAimConstraintType:
				aim = TRUE;
				break;
			}
		}
		
		tmp = con;
		con = con->next();
		delete tmp;
	}

	// Now, add the approriate list items.

	if( point ) {
		// Add the X,Y and Z translations of this DagNode as items in the list.
		appendAction( dn, kFLD_DAGNODE_XTRANSLATE, TRUE );
		appendAction( dn, kFLD_DAGNODE_YTRANSLATE, TRUE );
		appendAction( dn, kFLD_DAGNODE_ZTRANSLATE, TRUE );
	}

	if( orient || aim ) {
		// Add the X,Y and Z rotations of this DagNode as items in the list.
		appendAction( dn, kFLD_DAGNODE_XROTATE, TRUE );
		appendAction( dn, kFLD_DAGNODE_YROTATE, TRUE );
		appendAction( dn, kFLD_DAGNODE_ZROTATE, TRUE );
	}

	// If this node is a joint and has a handle on it, do the handle.
	AlJoint * joint = dn->joint();
	if( joint ) {
		AlIKHandle * handle = joint->endEffectorHandle();
		if( handle ) {
			AlIKHandleNode * hn = handle->handleNode();
			if( hn ) {
				findConstraintsAt( hn );
				delete hn;
			}
			delete handle;
		}
		delete joint;
	}
}

/*
======================================================================
TuseResults::collectActions() - find all the objects that
	we'll need to look at after each viewFrame operation.
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void TuseResults::collectActions( void )
{
	switch( fExprOptionInfo.iObjects ) {
	case 0: 	// Objects::All
		if( fExprOptionInfo.bExpressions ) {
			findAllExpressions();
		}
		if( fExprOptionInfo.bConstraints ) {
			findAllConstraints();
		}
		break;
	case 1: 	// Objects::Active
		{
			TexprPickIterator	*exprWalk;
			int					iResult;

			// Use an iterator to walk through the pick list
			//
			exprWalk = new TexprPickIterator( 
				this, 
				fExprOptionInfo.bConstraints, 
				fExprOptionInfo.bExpressions );
			AlPickList::applyIteratorToItems( exprWalk, iResult );
			delete exprWalk;
		}
		break;
	}
}

/*
======================================================================
TuseResults::findAllConstraints() - Find all constrained objects
						for which we wish to create animation.
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void TuseResults::findAllConstraints( void )
{
	TexprHierIterator	*hierWalk;
	AlDagNode 			*node;
	AlDagNode 			*tmp;

	node = AlUniverse::firstDagNode();
	hierWalk = new TexprHierIterator( this, TRUE, FALSE );

	while( node ) {
		WalkTree::walkHierarchy( kHierBelow, node, hierWalk ); 

		tmp = node;
		node = node->nextNode();
		delete tmp;
	}
}

/*
======================================================================
TuseResults::findAllExpressions() - Find all expression channels
                       in the universe, and create actions for the current
                       value of each one
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void
TuseResults::findAllExpressions( void )
{
	AlChannel	*pChannel = AlUniverse::firstChannel ();

	if ( pChannel ) {
		// Walk through each channel in the universe
		//
		do {
			if (pChannel->channelType () == kExprChannel) {
				// Add this expression to our list of expressions
				//
				appendAction( pChannel, FALSE );
			}
		} while( sSuccess == AlUniverse::nextChannelD (pChannel) );
	}
}

/*
*/
void
TuseResults::viewCompensationTime(int index)
{
	TactionList	*theEntry;

	// view the given time
	//
	fPlayFrame->viewFrame(fCompTime[index], FALSE);

	// Walk through the list of action list to cache the values
	//
	theEntry = (TactionList *)(fBaseList->first ());
	while (theEntry)
	{
		theEntry->cacheCompensationValue(index);
		theEntry = theEntry->next ();
	}
}


/*
======================================================================
TuseResults::sampleChannels() - Sample all expression channels
                       in the universe, and create actions for the current 
					   value of each one.
----------------------------------------------------------------------
Parameters:
    bLastFrame      boolean             TRUE if this is the last evaluation
                                        (so that tangents may be recomputed)
	bCache			boolean				TRUE if need to cache the current value
Returns:
    Nothing
======================================================================
*/
boolean
TuseResults::sampleChannels(
	boolean	bLastFrame,
	boolean bFristFrame)
{
	TactionList	*theEntry;
	double		dValue;
	boolean		bCompensated = FALSE;

	// Walk through the list of expressions the we have
	//
	theEntry = (TactionList *)( fBaseList->first () );
	for ( ; theEntry; theEntry = theEntry->next() )
	{
		if( !AlIsValid( theEntry->animatedItem ) )
			continue;

		AlAnimatable * anim = theEntry->animatedItem->asAnimatablePtr();
		if( !anim )
			continue;

		// get the value at current frame
		//
		dValue = anim->sampleChannel( theEntry->parameter );

		// insert compensation frames if it needs and prepare for
		// next frame evaluation
		//
		if ( fCompensate && theEntry->compensate )
		{
			if ( bFristFrame )
				theEntry->p_value = dValue;
			else
			{
				theEntry->c_value = dValue;

				if ( theEntry->keyCompensationFrames(fLastFrame, fCurrFrame,
						fCompTime, fTimeValid, fExprOptionInfo.bDetect_only) )
				{
					bCompensated = TRUE;
				}
				theEntry->p_value = theEntry->c_value;
			}

		}

		// Create a keyframe to hold the current sample
		//
		if ( !fExprOptionInfo.bDetect_only )
		{
			theEntry->keyframeAction->addKeyframe (
						fCurrFrame, dValue, bLastFrame,
						kTangentLinear, kTangentLinear);
		}
	}

	return bCompensated;
}

/*
======================================================================
TuseResults::createChannels() - Create channels for all the
                       actions we have created
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void
TuseResults::createChannels (void)
{
	TactionList	*theEntry;
	AlChannel	*theChannel;

	AlPrintf (kPromptNoHistory, "Creating channels...\n");

	// create a wrapper for creating real new channels
	//
	theChannel = new AlChannel;

	// Walk through the list of new actions to create new channels
	//
	theEntry = (TactionList *)(fBaseList->first ());
	while( theEntry )
	{
		// destroy the old channel if it exists.
		//
		theEntry->destroyChannel();

		// create a new channel for this object and this parameter
		// with the new action.
		//
		theChannel->create (
			theEntry->animatedItem->asAnimatablePtr(), 
			theEntry->parameter, 
			theEntry->keyframeAction );

		// If we're burning constraints, we have to disable all
		// constraints from this object.
		//
		if( fExprOptionInfo.bConstraints )
		{
			AlDagNode * dn = theEntry->animatedItem->asDagNodePtr();
			if( dn )
			{
				AlConstraint *tmp = NULL;
				AlConstraint *con = dn->firstConstraint();
				while (con)
				{
					tmp = con->next();

					if (fExprOptionInfo.bDelete_constr)
						con->deleteObject();
					else
						con->setOn( FALSE );

					delete con;
					con = tmp;
				}
			}
		}

		theEntry = theEntry->next ();
	}

	delete theChannel;
}

//==============================================================================
//
// LOCAL FUNCTIONS AND PLUGIN CODE
//
//==============================================================================

// Local functions
//
void useExprResults (void);

/*
======================================================================
plugin_init() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    (int) 0 upon successful completion
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_init (const char *dirName)
{
	char *dirScm;

	// Initialize OpenAlias
	//
	AlUniverse::initialize (kYUp);

	dirScm = makeAltPath(dirName, "scm");

	// Invoke our initialisation scheme module
	//
	AlInvokeSchemeFile( PLUGIN_NAME ".i.scm", dirScm );

	// Establish ourselves as an OpenAlias plug-in
	//
	useExprFunc.create( "pl_" PLUGIN_NAME, useExprResults );
	useExprHandle.create( PLUGIN_LABEL, &useExprFunc );
	useExprHandle.setAttributeString( PLUGIN_NAME );

	// Attach our option box
	//
	if ( sSuccess != useExprHandle.setOptionBox( 
			PLUGIN_NAME ".o.scm", PLUGIN_NAME ".options",dirScm ) ) {
		AlPrintf( kPrompt, PLUGIN_LABEL " plug-in unable to find .scm file for option box" );
		return 1;
	}
		
	useExprHandle.appendToMenu ("ap_animwinds");

	AlPrintf( kPrompt, PLUGIN_LABEL " installed on Menu 'Animation'");
	return (0);
}

/*
======================================================================
plugin_exit() - OpenAlias cleanup function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_exit (void)
{
	useExprHandle.deleteObject ();
	useExprFunc.deleteObject ();
	return (0);
}


/*
======================================================================
useExprResults() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void
useExprResults (void)
{
	boolean ik_dp;
	boolean con_dp;
	boolean exp_dp;

	TuseResults	useResults;

	ik_dp = AlPerformance::ikDuringPlayback();
	con_dp = AlPerformance::constraintsDuringPlayback();
	exp_dp = AlPerformance::expressionsDuringPlayback();

	AlPerformance::setIkDuringPlayback( TRUE );
	AlPerformance::setConstraintsDuringPlayback( TRUE );
	AlPerformance::setExpressionsDuringPlayback( TRUE );

	if (useResults.eval() == sSuccess) {
		AlPrintf (kPrompt, PLUGIN_LABEL " complete.\n");
	}

	AlPerformance::setIkDuringPlayback( ik_dp );
	AlPerformance::setConstraintsDuringPlayback( con_dp );
	AlPerformance::setExpressionsDuringPlayback( exp_dp );
}
