/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	This is a sample OpenAlias plugin which shows how to simulate defining
//	your own functions for use with expressions.  This is done by creating
//	an action curve which is a keyframe sampled representation of the
//	desired function.  The "function" is then "evaluated" by using the
//	animate function within an expression, and passing the function "parameter"
//	as the evaluation time.
//
//	For example, if you wanted to add a sine function to the expression
//	parser (lets pretend that sin is not available).  This could be done
//	be creating an action curve of the sine function (which is illustrated
//	below), and then "evaluating" the function using:
//
//		animate (Sine, angle)
//
//	which would return the value of sin (angle).
//
//	This example plugin creates either a sine function or a noise function,
//	depending upon what is chosen in the option box.
//

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlChannel.h>
#include <AlParamAction.h>
#include <AlViewFrame.h>
#include <AlLinkItem.h>
#include <AlList.h>
#include <AlIterator.h>
#include <AlAnimatable.h>
#include <AlGroupNode.h>
#include <AlDagNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <math.h>
#include <string.h>

// Our instance of the function handle
//
static AlFunctionHandle applyFunctionHandle;
static AlMomentaryFunction	applyFunctionFunc;

// Local helper classes
//

// A structure to hold all the information in the option dialog
//
struct TfunctionOptionInfo {
	int		iObject;			// Which object should have the function applied
	int		iChannel;			// Which channel the function should be applied to
	int		iFunction;			// Which function to apply
	double	dStart;				// Start frame
	double	dEnd;				// End frame
	double	dBy;				// Sample
};

const double kDegRad = 0.0174532925199432958;	// Conversion from degrees to radians

// Our actual applyFunction control class
//
class TapplyFunction {
public:
	TapplyFunction ();
	~TapplyFunction ();

	statusCode createFunction ();

private:
	double evaluateFunction (double dFrame);

	TfunctionOptionInfo	fFunctionOptionInfo;
};

// Local functions
//
void createFunction (void);
double makeNoise (double dValue);

/*
======================================================================
plugin_init() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    (int) 0 upon successful completion
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_init (const char *dirName)
{
	char *dirScm;

	// Initialize OpenAlias
	//
	AlUniverse::initialize (kYUp);

	dirScm = makeAltPath(dirName,"scm");

	// Invoke our initialisation scheme module
	//
	AlInvokeSchemeFile ("applyFunctionExample.i.scm", dirScm);

	// Establish ourselves as a OpenAlias plug-in
	//
	applyFunctionFunc.create (createFunction);
	applyFunctionHandle.create ("Apply Function", &applyFunctionFunc);
	applyFunctionHandle.setAttributeString ("function");

	// Attach our option box
	//
	if ( sSuccess != applyFunctionHandle.setOptionBox ( "applyFunctionExample.o.scm",
		"function.options",dirScm) ) {
		AlPrintf( 
			kPrompt, 
			"applyFunction plug-in unable to find .scm file for option box"
		);
		return (1);
	}

	applyFunctionHandle.setIconPath( makeAltPath( dirName, NULL ) );
	applyFunctionHandle.appendToMenu ("ap_animwinds");

	AlPrintf(kPrompt, "Apply function installed on menu 'Animation/Apply Function'");
	return (0);
}

/*
======================================================================
plugin_exit() - OpenAlias cleanup function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_exit (void)
{
	applyFunctionHandle.deleteObject ();
	applyFunctionFunc.deleteObject ();
	return (0);
}


/*
======================================================================
createFunction() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void
createFunction (void)
{
	TapplyFunction	applyFunction;

	if (applyFunction.createFunction () == sSuccess) {
		AlPrintf (kPrompt, "Apply Function complete");
	}
}

/*
======================================================================
TapplyFunction::TapplyFunction() - Constructor for our
                                   apply function control object
----------------------------------------------------------------------
Parameters:
    None
Notes:
    The constructor will retrieve all the settings in the option dialog
Returns:
    Nothing
======================================================================
*/
TapplyFunction::TapplyFunction ()
{
	// Read our option dialog settings
	//
	AlGetInteger ("ao_function_object", fFunctionOptionInfo.iObject);
	AlGetInteger ("ao_function_channel", fFunctionOptionInfo.iChannel);
	AlGetInteger ("ao_function_function", fFunctionOptionInfo.iFunction);
	AlGetDouble ("ao_function_startframe", fFunctionOptionInfo.dStart);
	AlGetDouble ("ao_function_endframe", fFunctionOptionInfo.dEnd);
	AlGetDouble ("ao_function_byframe", fFunctionOptionInfo.dBy);
}

/*
======================================================================
TapplyFunction::TapplyFunction() - Destructor for our
                                   apply function control object
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
TapplyFunction::~TapplyFunction ()
{
}

/*
======================================================================
TapplyFunction::createFunction() - Control module to create the requested
                                   function and apply it to the specified object
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
statusCode
TapplyFunction::createFunction ()
{
	// A table to translate between option settings a parameter fields
	//
	static int iTransTable[] = {
		kFLD_DAGNODE_XTRANSLATE,
		kFLD_DAGNODE_YTRANSLATE,
		kFLD_DAGNODE_ZTRANSLATE,
		kFLD_DAGNODE_XROTATE,
		kFLD_DAGNODE_YROTATE,
		kFLD_DAGNODE_ZROTATE,
		kFLD_DAGNODE_XSCALE,
		kFLD_DAGNODE_YSCALE,
		kFLD_DAGNODE_ZSCALE,
		-1
	};

	// Make sure that we need to do anything at all
	//
	if ((fFunctionOptionInfo.dStart >= fFunctionOptionInfo.dEnd) ||
		(fFunctionOptionInfo.dBy <= 0.0)) {
		return (sFailure);
	}

	// Find the object that we want to apply the function to
	//
	AlObject	*theObject = NULL;
	switch (fFunctionOptionInfo.iObject) {
	case 0:		// Active object
		AlPickList::firstPickItem ();
		theObject = AlPickList::getObject ();
		break;
	case 1:		// New object
		theObject = new AlGroupNode;
		theObject->asGroupNodePtr()->create();
		theObject->setName ("functionObject");
		break;
	}
	if (!AlIsValid (theObject)) {
		return (sFailure);
	}
	// Make sure it is a dag node object
	//
	AlDagNode	*theDagNode = theObject->asDagNodePtr ();
	if (!AlIsValid (theDagNode)) {
		delete theObject;
		return (sFailure);
	}

	// Create a new action, which will represent our function
	// and then attach keyframes to represent the function
	//
	AlParamAction	*functionAction = new AlParamAction;
	// And then evaluate the function at each frame
	//
	for (double dFrame = fFunctionOptionInfo.dStart;
		dFrame <= fFunctionOptionInfo.dEnd;
		dFrame += fFunctionOptionInfo.dBy) {
		functionAction->addKeyframe (
			dFrame,
			evaluateFunction (dFrame),
			TRUE,
			kTangentSmooth,
			kTangentSmooth
		);
	}
	// Give out action a nice name
	// Note: this can only be done AFTER atleast one keyframe has been added
	//
	switch (fFunctionOptionInfo.iFunction) {
	case 0:		// Sine
		functionAction->setName ("Sine");
		// Set the extrapolation types so that the sin curve will
		// repeat itself
		//
		functionAction->setExtrapTypePRE (kEXTRAP_CYCLE);
		functionAction->setExtrapTypePOST (kEXTRAP_CYCLE);
		break;
	case 1:		// Noise
		functionAction->setName ("Noise");
		break;
	}

	// Delete any existing channels
	//
	AlAnimatable	*theAnimatedObject = theDagNode->asAnimatablePtr ();
	AlChannel	*theChannel = theAnimatedObject->firstChannel ();
	if (AlIsValid (theChannel)) {
		do {
			if (theChannel->parameter () ==
				iTransTable[fFunctionOptionInfo.iChannel]) {
				theChannel->deleteObject ();
				delete theChannel;
				theChannel = theAnimatedObject->firstChannel ();
			}
		} while( sSuccess == theAnimatedObject->nextChannelD (theChannel));
		delete theChannel;
	}
	// Now add the function action onto the requested channel
	//
	theChannel = new AlChannel;
	theChannel->create (
		theAnimatedObject,
		iTransTable[fFunctionOptionInfo.iChannel],
		functionAction
	);
	delete theChannel;

	delete functionAction;
	delete theObject;
	return (sSuccess);
}

/*
======================================================================
TapplyFunction::evaluateFunction() - Helper function to evaluate our
                                     function at a given frame
----------------------------------------------------------------------
Parameters:
    dFrame          double              Frame being evaluated
Returns:
    Nothing
======================================================================
*/
double
TapplyFunction::evaluateFunction (double dFrame)
{
	switch (fFunctionOptionInfo.iFunction) {
	case 0:		// Sine
		return (sin (dFrame * kDegRad));
	case 1:		// Noise
		return (makeNoise ((dFrame - fFunctionOptionInfo.dStart) / (fFunctionOptionInfo.dEnd - fFunctionOptionInfo.dStart)));
	}
	return (0.0);
}

/*
======================================================================
makeNoise() - This is a noise function based upon Ken Perlin's noise
              function as described in "Texturing and Modeling: A
              Procedural Approach"
----------------------------------------------------------------------
Parameters:
    dValue          double              Point to evaluate noise
                                        Should be normalised between 0.0 and 1.0
Returns:
    Nothing
======================================================================
*/

const int B = 0x100;
const int BM = 0xff;
const int N = 0x1000;

static int p[B + B + 2];
static double g1[B + B + 2];

double
makeNoise (double dValue)
{
	int		bx0, bx1;
	double	rx0, rx1, sx, t, u, v;
	boolean	bInit = FALSE;

	if (!bInit) {
		int	i, j, k;

		for (i = 0; i < B; i++) {
			p[i] = i;
			g1[i] = (double)((random () % (B + B)) - B) / B;
		}

		while (--i) {
			k = p[i];
			j = (int)(random () % B);
			p[i] = p[j];
			p[j] = k;
		}

		for (i = 0; i < B + 2; i++) {
			p[B + i] = p[i];
			g1[B + i] = g1[i];
		}

		bInit = TRUE;
	}

	t = dValue + N;
	bx0 = ((int)t) & BM;
	bx1 = (bx0 + 1) & BM;
	rx0 = t - (int)t;
	rx1 = rx0 - 1.0;

	sx = rx0 * rx0 * (3.0 - 2.0 * rx0);

	u = rx0 * g1[p[bx0]];
	v = rx1 * g1[p[bx1]];
	return (u + sx * (v - u));
}
