/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/


#include <AlUniverse.h>
#include <AlDagNode.h>
#include <AlGroupNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <WalkTree.h>

/*
======================================================================
WalkTree::walkHierarchy() - Helper function to walk through a Dag hierarchy
                            and call a function for each object
----------------------------------------------------------------------
Parameters:
    walkType        enumHier            Type of walk to perform
    theObject       AlObject *          The object to start walking from
    theFunction     AlIterator *        The function to call for each
                                        object in the hierarchy
Returns:
    Nothing
======================================================================
*/
void
WalkTree::walkHierarchy (enumHier walkType, AlObject *theObject, AlIterator *theFunction)
{
	if (!theFunction) {
		return;
	}

	// If the starting object is NULL then the entire world is walked
	//
	if (!theObject) {
		AlDagNode *pDagNode = AlUniverse::firstDagNode();
		while (pDagNode) {
			AlDagNode *nextDagNode = pDagNode->nextNode();

			walkHierarchy (kHierBelow, pDagNode, theFunction);
			delete pDagNode;

			pDagNode = nextDagNode;
		}
		return;
	}

	// If hierarchy is none, then call the function on this object,
	// and then we are done
	//
	if (walkType == kHierNone) {
		(theFunction->func) (theObject);
		return;
	}

	// If hierarchy is above or both, then walk up
	//
	if ((walkType == kHierAbove) || (walkType == kHierBoth)) {
		AlDagNode *dagNode = theObject->asDagNodePtr();
		if( dagNode )
		{
			AlGroupNode	*pParent = dagNode->parentNode ();
			if (pParent) {
				walkHierarchy (kHierAbove, pParent, theFunction);
				while (pParent->nextInstanceD() == sSuccess) {
					walkHierarchy (kHierNone, pParent, theFunction); 
				}
				delete pParent;
			}
		}
	}

	// Now call the function on this object
	//
	(theFunction->func) (theObject);

	// Then walk down, if necessary
	//
	if ((walkType == kHierBelow) || (walkType == kHierBoth)) {
		AlGroupNode* groupNode = theObject->asGroupNodePtr();
		if( groupNode && !groupNode->prevInstance() )
		{
			AlDagNode	*pChild = groupNode->childNode ();
			while (pChild) {
				AlDagNode* nextChild = pChild->nextNode ();
	
				walkHierarchy (kHierBelow, pChild, theFunction);
				delete pChild;
	
				pChild = nextChild;
			}
		}
	}
}

