/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <math.h>
#include <Tsun.h>
#include <statusCodes.h>
#include <AlTM.h>
#include <AlStyle.h>

int Tsun::fDayMonths[] = {0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};

Tsun::Tsun (int iMonth, int iDay, int iHour, int iMinute, double dLatitude)
:	fMonth (iMonth),
	fDay (iDay),
	fHour (iHour),
	fMinute (iMinute),
	fLatitude (dLatitude)
{
	// Convert from degrees.minutes to degrees.decimal
	//
	if (fLatitude >= 0.0) {
		fLatitude = floor (fLatitude) + ((fLatitude - floor (fLatitude)) / 0.6);
	}
	else {
		fLatitude = ceil (fLatitude) + ((fLatitude - ceil (fLatitude)) / -0.6);
	}
	calcAltitudeAzimuth ();
}

Tsun::Tsun (double dAltitude, double dAzimuth)
:	fAltitude (dAltitude),
	fAzimuth (dAzimuth)
{
}

Tsun::~Tsun ()
{
}

void
Tsun::calcAltitudeAzimuth (void)
{
	boolean	bSouth, bAfternoon;
	double	dLatitude = fLatitude * kDegRad;
	double	dHour = (double)fHour + ((double)fMinute / 60.0);

	// Check for the southern hemisphere
	//
  	if (dLatitude < 0.0) {
		bSouth = TRUE;
		dLatitude = -dLatitude;
  	}
	else {
		bSouth = FALSE;
	}

	if ((fMonth < 1) || (fMonth > 12)) {
		fMonth = 1;
		fDay = 1;
	}

	int iDay = fDay + fDayMonths[fMonth - 1];
	// Adjust the day for the southern hemisphere
	//
  	if (bSouth) {
		if (iDay > 183) {
		  	iDay -= 183;
		}
		else {
		  	iDay += 182;
		}
  	}
	// Check for afternoon
	//
  	if (dHour > 12.0) {
		bAfternoon = TRUE;
		dHour = 24.0 - dHour;
  	}
	else {
		bAfternoon = FALSE;
	}

	double dHourAngle = (180.0 - dHour * 15.0) * kDegRad;
	double dDeclination = 23.45 * sin (2.0 * kPI * ((284.0 + (double)iDay) / 365.0)) * kDegRad;

	double cosDecl = cos (dDeclination);
	if (cosDecl == 0.0) cosDecl = 0.00001;
	
	double cosLat = cos (dLatitude);
	if (cosLat == 0.0) cosLat = 0.00001;

	double sinLat = sin (dLatitude);
	if (sinLat == 0.0) sinLat = 0.00001;

	fAltitude = asin (cosLat * cosDecl * cos (dHourAngle) + sinLat * sin (dDeclination));

	double cosAltitude = cos (fAltitude);
	if (cosAltitude == 0.0) cosAltitude = 0.00001;

	if (cos (dHourAngle) > sin (dDeclination) / cosDecl / (sinLat / cosLat)) {
		fAzimuth = asin (cosDecl * sin (dHourAngle) / cosAltitude);
	}
	else {
		fAzimuth = 180.0 * kDegRad - asin (cosDecl * sin (dHourAngle) / cosAltitude);
	}

	if (bAfternoon) {
		fAzimuth = -fAzimuth;
	}

  	fAltitude /= kDegRad;
  	fAzimuth /= kDegRad;
  	fAzimuth = 180.0 - fAzimuth;

	// Adjust for the southern hemisphere
	//
  	if (bSouth) {
		if (fAzimuth < 180.0) {
			fAzimuth = 180.0 - fAzimuth;
		}
		else {
			fAzimuth = 540.0 - fAzimuth;
		}
	}
}
