/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlVector.h>
#include <math.h>

AlVector::AlVector()
//
//	Description:
//		Do nothing constructor
//
{
}

AlVector::AlVector( const AlVector &v )
{
	*this = v;
}

AlVector::AlVector( double vx, double vy, double vz )
{
	x = vx;
	y = vy;
	z = vz;
}

AlVector::AlVector( const double array[3] )
{
	x = array[0];
	y = array[1];
	z = array[2];
}

void AlVector::put( double array[3] ) const
{
	array[0] = x;
	array[1] = y;
	array[2] = z;
}

AlVector::~AlVector()
//
//	Description:
//		Do nothing destructor.
//
{
}

AlVector& AlVector::operator =( const AlVector& v )
{
	x = v.x;
	y = v.y;
	z = v.z;

	return *this;
}

int AlVector::operator ==( const AlVector& v ) const
{
	return x == v.x && y == v.y && z == v.z;
}

int AlVector::operator !=( const AlVector& v ) const
{
	return !operator==(v);
}

AlVector AlVector::operator +( const AlVector& v ) const
{
	AlVector result( *this );
	result += v;
	return result;
}

AlVector AlVector::operator -( const AlVector& v ) const
{
	AlVector result( *this );
	result -= v;
	return result;
}

AlVector AlVector::operator ^( const AlVector& v ) const
{
	AlVector result( *this );
	result ^= v;
	return result;
}

AlVector AlVector::operator *( const double d ) const
{
	AlVector result( *this );
	result *= d;
	return result;
}

double AlVector::operator *( const AlVector& v ) const
//
//	Vector dot product:	c = a.b
//
{
	return x*v.x + y*v.y + z*v.z;
}

AlVector operator *( const double d, const AlVector& v )
{
	return v * d;
}

AlVector AlVector::operator /( const double d ) const
{
	AlVector result( *this );
	result /= d;
	return result;
}

AlVector AlVector::operator -() const
{
	return AlVector( -x, -y, -z );
}

AlVector& AlVector::operator +=( const AlVector& v )
{
	x += v.x;
	y += v.y;
	z += v.z;
	return *this;
}

AlVector& AlVector::operator -=( const AlVector& v )
{
	x -= v.x;
	y -= v.y;
	z -= v.z;
	return *this;
}

// 
// vector (cross) product b = b^c
//
AlVector& AlVector::operator ^=( const AlVector& c )
{
	AlVector b( *this );

	x = b.y * c.z - b.z * c.y;
	y = b.z * c.x - b.x * c.z;
	z = b.x * c.y - b.y * c.x;
	return *this;
}

//
// scale
//
AlVector& AlVector::operator *=( const double d )
{
	x *= d;
	y *= d;
	z *= d;
	return *this;
}

AlVector& AlVector::operator /=( const double d )
{
	x /= d;
	y /= d;
	z /= d;
	return *this;
}

double AlVector::size() const 
{
	return sqrt( x*x + y*y + z*z );
}

AlVector AlVector::norm() const
{
	double length;

	length = size();
	if( length )
		return (*this)/length;
	else
		return *this; 
}
