/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//	.NAME AlUserCommand - Encapsulates the user defined interface to command history
//
//	.SECTION Description
//		This file contains the definitions required to define an user
//		command history command.
//

#ifndef _AlUserCommand
#define _AlUserCommand

#include <AlStyle.h>

class AlObject;
class AlNotifyDagNode;
class AlCommand;
class AlDagNode;
class AlCurveOnSurface;
class AlContact;
class AlCommandRef;

class ofstream;

class STUDIOAPI_DECL AlData
{
	public:
		enum { kDataUnset, kDataChar, kDataInt, kDataDouble };
		
		AlData( void );
		~AlData( void );
		
		bool isValid( void );
		
		statusCode create( int typeId, char *ptr, int count );
		statusCode create( int typeId, int *ptr, int count );
		statusCode create( int typeId, double *ptr, int count );
		
		int type(void);
		int id(void);
		int count();
		
		const char *asCharPtr( void );
		const int *asIntPtr( void );
		const double *asDoublePtr( void );
		
	private:
		void *_ptr;
		int _count;
		int _type;
		int _id;
};

class STUDIOAPI_DECL AlOutput
{
	public:
		AlOutput()	{};
		virtual ~AlOutput()	{};
		
		virtual statusCode declareObject( AlObject *obj );
		virtual statusCode declareData( AlData *obj );
};

class STUDIOAPI_DECL AlInput
{
	public:
		AlInput()	{};
		virtual ~AlInput()	{};

		virtual AlObject* resolveObject();
		virtual AlData* resolveData( int type, int typeId );
		virtual AlData*	resolveData( int index );
};

class AlUserCommandPrivate;

class STUDIOAPI_DECL AlUserCommand {
	friend class AlCommandFriend;

public:
	// overloaded user functions
	// do not call these directly or the command history may not be correctly
	// maintained
	//
	virtual				~AlUserCommand();
	virtual int			isValid();
	virtual int			execute();
	virtual int			declareReferences();
	virtual int			instanceDag( AlDagNode *oldDag, AlDagNode *newDag );
	virtual int			undo();
	virtual int			dagModified( AlDagNode *dag );
	virtual int			geometryModified( AlDagNode *dag );
	virtual int			curveOnSurfaceModified( AlCurveOnSurface *surf );
	virtual int  		listModifiedDagNodes( const AlNotifyDagNode *dagMod, AlObject *obj );
	virtual int			debug( const char *prefix );
	virtual void *		asDerivedPtr();

	virtual statusCode	storeWire( AlOutput *output );
	virtual statusCode	storeSDL( ofstream &outputSDL );
	virtual statusCode 	retrieveWire( AlInput *input );

	// for your own use
	virtual int			type();

public:
// general utility functions
	const char *		name() const;
	AlCommand *			command() const;

// declare utility functions
	AlCommandRef*		firstConstructorRef();
	statusCode			addConstructorRef( AlDagNode * );
	statusCode			addConstructorRef( AlCurveOnSurface *);
	statusCode			addConstructorContact( AlDagNode *, AlContact * );

	AlCommandRef*		firstTargetRef();
	statusCode			addTargetRef( AlDagNode * );
	statusCode			addTargetRef( AlCurveOnSurface * );
	statusCode			addTargetContact( AlDagNode *, AlContact * );
	statusCode			deleteAllReferences();

	API_BOOLEAN				isDagNodeAConstructor( AlDagNode * ) const;
	API_BOOLEAN				isDagNodeATarget( AlDagNode *, API_BOOLEAN includeCOS = FALSE ) const;
	API_BOOLEAN				isCurveOnSurfaceAConstructor( AlCurveOnSurface *) const;
	API_BOOLEAN				isCurveOnSurfaceATarget( AlCurveOnSurface *) const;

protected:
	// This constructor is not used in the base class (since it is pure).
	// Defined your 'userCmdConstructor' to call the constructor for your
	// derived class.
	AlUserCommand();

private:
	// don't touch this
	AlUserCommandPrivate *privateData;

private:
	// Disallow the copy constructor and copy assignment
	AlUserCommand (const AlUserCommand &);
	AlUserCommand &operator= (const AlUserCommand &);
};

#endif	// _AlUserCommand
