/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
//
//	.NAME AlUniverse - Encapsulates the retrieval, data access and storage of Alias Wire Files.
//
//	.SECTION Description
//		This is a static class in that all of its member functions are
//		static.  It provides the foundation on which to interface with
//		Alias Wire Files (OpenModel) or with Alias directly (via LiveData).
//
//		In Alias, there is the concept of a 'stage', an independant entity
//		which encapsulates all the top-level information about a modeling
//		environment.  The stage effectively represents the running version
//		of an individual Wire file.  In the interactive package, stages can
//		be controlled through the Stage Editor.  The identical control also
//		exists in the AlUniverse class, via the Stage functions.  These let
//		you switch between stages, create and destroy stages, merge stages,
//		and alter their features.
//
//		LiveData and OpenModel use object data which defines, among other
//		things, transformations, curve and surface geometry, cameras and lights.
//		Transformations are kept in objects called "dag nodes".
//		Dag nodes can be connected to other dag nodes to form a list of
//		dag nodes.  There are particular types of dag nodes which refer
//		specifically to geometry, cameras, lights or refer to another
//		list of dag nodes to form a hierarchical structure.  Together,
//		these different types of dag nodes can be used to build a list
//		of hierarchical structures which is collectively called a
//		directed acyclic graph or "DAG".
//
//		The AlUniverse class gives access to the first dag node in the DAG,
//		and from that dag node, the list of all dag nodes in the universe
//		can be traversed.
//
//		The object data may also contain animation information on various
//		animatable items, such as a dag node or a piece of geometry.
//		The animation information can be accessed through the item that
//		is animated.  The AlUniverse class also gives access to animation
//		data through global lists.
//
//		Geometry will also contain shading information.  Each surface
//		or face (a particular kind of curve) will contain a list of
//		shaders.  The AlUniverse class also gives access to a global list of
//		all shaders in the universe, as well as the single environment class.
//
//		The AlUniverse class also gives access to the list of sets in the
//		universe via the firstSet() method.  Similarly, the firstCluster()
//		method will give access to all the clusters in the universe.
//		However, clusters are also accessible through methods in the
//		classes which can belong to clusters (i.e. AlDagNode, AlCurveCV and
//		AlSurfaceCV).  A cluster is an object in the DAG which has a collection
//		of dag nodes and/or CVs.  The transformations in the dag nodes
//		above the cluster are applied to the CVs in the cluster after the
//		latter's own transformations have been applied.
//
//		AlUniverse provides base access to the ImagePlane list and
//		the Window list through firstImagePlane, firstWindow and related
//		functions.
//
//		Whenever a new Alias Object of any kind is created, it is automatically
//		added to the appropriate place in the universe.  For example, if
//		a dag node is created, it will be traversable through the
//		AlUniverse::firstDagNode() method.  If a new animation action is
//		created, it will be accessable by walking the list of actions via
//		AlUniverse::firstAction()/nextAction().
//
//		However, there are a few exceptions.
//		The AlCurve, AlFace, AlSurface and AlKeyframe classes can all be
//		created and will not be part of the universe initially.  They will
//		become part of the universe when they are added to other objects.
//		For example, an AlCurve will become part of the universe when its
//		AlCurveNode is created.  An AlKeyframe will become part of the
//		universe when its AlParamAction is created.  If a class is not
//		part of the universe, this means that if AlUniverse::store() is
//		called, that class will not be stored.  If a LiveData application
//		exits without making an object part of the universe, then the data
//		will be lost and wasted memory will result.
//
//		The initialize method must be called before any other AlUniverse
//		method is called.  An Alias AlObject cannot be created until the
//		initialize method is called.
//
*/

#ifndef _AlUniverse
#define _AlUniverse

/*
 * window refresh flags for LiveData
 */

#define kRedrawInactive 001
#define kRedrawActive   002
#define kRedrawTemplate 004
#define kRedrawWindows	010
#define kRedrawListers	020
#define kRedrawNone     0
#define kRedrawAll  (kRedrawInactive | kRedrawActive | kRedrawTemplate | kRedrawWindows | kRedrawListers )

typedef enum {
	kFromPreviewWindow,
	kMinMax,
	kGlobal
} AlFrameRangeType;

#include <AlCoordinateSystem.h>
#include <AlMessage.h>

class AlList;
class AlDagNode;
class AlSet;
class AlChannel;
class AlAction;
class AlShader;
class AlSwitchShader;
class AlLayeredShader;
class AlCluster;
class AlEnvironment;
class AlSurface;
class AlObject;
class AlWindow;
class AlImagePlane;
class AlPerspectiveCamera;
class AlPtrTable;
class AlIterator;
class AlUpdate;
class AlLocator;
class AlConstructionEntity;
class AlLayer;
class AlCloud;
class AlBlendCurve;
class AlCategory;
class AlReferenceFile;
class AlReferenceFileSet;

struct Aa_Channel;
struct Aa_Action_s;
struct AlRetrieveOptions;
struct AlStoreOptions;

class STUDIOAPI_DECL AlUniverse : public AlMessage {
	friend					class AlFriend;
	friend					class AlUpdate;

public:
	enum AlFileType {
		kUnknown,   // File type is unknown and will be ignored by ::retrieve()

		kWire,       // Alias Wire File
		kIges,       // Initial Graphics Exchange Standard
		kVdais,      // Verband Der Automobilindustrie IGES Subset
		kC4x,        // EDS/GM format.
		kJamais,     // Japan Automobile Manufacturers Association IGES Subset.
		kVdafs,	     // Verband Der Automobilindustrie Freeform Surface Interface.
		kDxf,        // Drawing Exchange Format
		kDwg,        // Autodesk Drawing Format

                             // The file types below are available only in OpenAlias.

		kDes,        // EDS/GM Data Exchange Standard
		kTri,        // Alias Object Separated Triangle Format
		kQuad,       // Alias Object Separated Quad Format
		kProRender,  // Pro/Engineer Render File Format.
		kInventor,   // OpenInventor Metafile Subset V1.0
		kStl,        // 3D Systems SLA Format
		kObj,        // Alias TAV format.
		kUnused,     // For future use. 
		kEpsf,       // Adobe Encapsulated Postscript v2.x
		kIllustrator,// Illustrator Format.
		kSlc,        // 3D Systems Slice Format.
        kEdf,        // ICEM EDF format.
        kFbx,        // FBX format.
        kAcis,       // ACIS format.
        kCatiaV5,    // CATIAV5 format.
        kJT,         // JT format.
        kNx,         // Unigraphics (Nx) format.
	};

	// Project subdirectory types.
	// These indicate the subdirectories where different types of files are
	// located.

	enum AlDirectoryType {
		kAnimData,			// Animation data (Save Anim / Retrieve Anim)
		kCanvasData,		// Paint canvas files
		kDepthData,			// Depth files
		kEnvironmentData,	// Environment files (saved/retrieved via
							//					  multi-lister)
		kLightData,			// Light files (saved/retrieved via multi-lister)
		kMaskData,			// Mask files
		kMiscData,			// Miscellaneous data files: color, window etc.
		kOptionData,		// UI option files
		kPixData,			// Image files for file textures, image planes etc.
		kPlotData,			// Plot files
		kRibData,			// Pixar RIB files
		kSlaData,			// Stereo Lithography files (.stl, .slc, .tri)
		kSdlData,			// Scene Description Files
		kShaderData,		// Shader files (saved/retrieved via multi-lister)
		kStageData,			// Wire files representing stages.
		kTextureData,		// Texture files (saved/retrieved via multi-lister)
		kWireData			// Wire files an CAD files (e.g. IGES, DXF, etc.)
	};

	enum AlApplicationMode
	{
		kOpenModel,
		kOpenAlias
	};	
	
	enum AlLayerSortMode{
		kLayerNumber,
		kLayerName		
	};
public:
	static statusCode		initialize( AlCoordinateSystem = kZUp,
										API_BOOLEAN initProjectEnv = FALSE  );
	static API_BOOLEAN			isInitialized();

	static AlCoordinateSystem	coordinateSystem ();

	static statusCode		expandFileName( char[], const char *,
											AlDirectoryType directoryType );
	static API_BOOLEAN			isWireFile( const char *fileName, char fileVersion[8] );

	static statusCode		wireFileWindowSize( int &sizeX, int &sizeY );
	static statusCode		setWireFileWindowSize( int sizeX, int sizeY );

	static AlFileType		fileType( const char * );
    static AlFileType		fileType( const wchar_t * );
	static statusCode       retrieveOptions( AlRetrieveOptions& );
	static statusCode		setRetrieveOptions( const AlRetrieveOptions& );
	static statusCode		retrieve( const char * );
	static const char* 		lastWireFileRetrieved();
    static const char* 		lastFileRetrieved();
	static const char*		lastFileSaved();

    static statusCode       storeOptions( AlStoreOptions& );
	static statusCode		setStoreOptions( const AlStoreOptions& );
    static statusCode		store( const char *, AlDagNode* = 0 );
	static statusCode		store( const char *, AlDagNode*, 
                               bool embedImageReferences, 
                               bool includeInstalledImages,
                               AlFileType filetype = kWire );
	static statusCode		storeActive( const char *,
                                         AlFileType filetype = kWire );

	static AlDagNode*		firstDagNode();
	static AlSet*			firstSet();
	static AlCluster*		firstCluster();

	static const char*		currentStage( void );
	static statusCode		setCurrentStage( const char* );

	static statusCode		mergeStage( const char* );
	static statusCode		mergeAllStages( void );

	static statusCode		deleteStage( const char* );
	static statusCode		deleteAllStages( void );

	static statusCode		stageVisibility( const char *, API_BOOLEAN& );
	static statusCode		setStageVisibility( const char *, API_BOOLEAN );

	static const char*		stageWindowSource( void );
	static statusCode		setStageWindowSource( const char * );

	static const char*		stageBackgroundSource( void );
	static statusCode		setStageBackgroundSource( const char * );

	static statusCode		renameStage( const char *, const char * );
	static AlList*			stageNames( void );

	static statusCode		createNewStage( const char *, char*& );

	static statusCode		retrieveStageSet( const char *,API_BOOLEAN );
	static statusCode		saveStageSet( const char *);
	static statusCode		setStageWireFileName( const char *, const char * );
	static const char*		stageWireFileName( const char * );

	static AlWindow*		firstWindow();
	static AlWindow*		currentWindow();
	static AlWindow*		sbdWindow();

	static AlImagePlane*	firstImagePlane();

	static statusCode		blindData( int, long&, const char*& );
	static statusCode		blindData( int, int, long&, const char*& );
	static statusCode		setBlindData( int, long, const char* );
	static statusCode		removeBlindData( int );
	static statusCode		removeBlindData( int, int );

	static AlChannel*		firstChannel();
	static AlChannel*		nextChannel(AlChannel *);
	static statusCode		nextChannelD( AlChannel *);

	static AlAction*		firstAction();
	static AlAction*		nextAction(AlAction *);

	static AlShader*		firstShader();
	static AlShader*		nextShader( AlShader* );
	static statusCode		nextShaderD( AlShader* );
	static AlSwitchShader*	firstSwitchShader();
	static AlSwitchShader*	nextSwitchShader(AlSwitchShader*);
    static AlLayeredShader* firstLayeredShader();
    static AlLayeredShader*	nextLayeredShader(AlLayeredShader*);
	static AlEnvironment*	firstEnvironment();

	static AlLayer*			firstLayer();
	static AlLayer*			nextLayer( AlLayer* );
	static statusCode		nextLayerD( AlLayer* );
	static AlLayer*			layerByNumber(int);
    static AlLayer*			layerByName(char*);

	static AlLayer*			creationLayer();
	static statusCode		setCreationLayer(AlLayer *);

	static API_BOOLEAN		layersEnabled();
	static statusCode		setLayersEnabled( API_BOOLEAN );

	static AlPerspectiveCamera*		firstCamera();
	static AlPerspectiveCamera*		nextCamera( AlPerspectiveCamera* );
	static statusCode				nextCameraD( AlPerspectiveCamera* );

	static AlLocator*       firstLocator();
	static AlLocator*       nextLocator(AlLocator *);
	static statusCode       deleteAllLocators();

	static AlConstructionEntity*		firstConstructionEntity();
	static AlConstructionEntity*		nextConstructionEntity(AlConstructionEntity *);
	static statusCode       			deleteAllConstructionEntities();

	static statusCode		importCloudFile( const char *, int );
	static AlCloud 			*firstCloud();
	static int				numberOfClouds();

	static statusCode		importSubdiv( const char* filename );

	static AlBlendCurve*	firstBlendCurve();
	static int				numberOfBlendCurves();

    static AlCategory       *firstCategory();
    static AlCategory       *nextCategory(AlCategory *);

    static AlReferenceFile*         importReferenceFile(const char* filename);
    static statusCode               removeReferenceFile(AlReferenceFile*);
    static AlReferenceFileSet*      createReferenceAlternative(const char* name);
    static AlReferenceFileSet*		firstReferenceFileSet();
	static AlReferenceFileSet*		nextReferenceFileSet( AlReferenceFileSet* );
	static statusCode		        nextReferenceFileSetD( AlReferenceFileSet* );
    static statusCode       packReference( const char* packDir );

	static statusCode		redrawScreen( unsigned flags = kRedrawInactive | kRedrawActive );

	static statusCode		applyIteratorToImagePlanes( AlIterator*, int& );
	static statusCode		applyIteratorToWindows( AlIterator *, int& );
	static statusCode		applyIteratorToDagNodes( AlIterator *, int& );
	static statusCode		applyIteratorToActions( AlIterator *, int& );
	static statusCode		applyIteratorToChannels( AlIterator *, int& );
	static statusCode		applyIteratorToSets( AlIterator *, int& );
	static statusCode		applyIteratorToClusters( AlIterator *, int& );
	static statusCode		applyIteratorToShaders( AlIterator *, int& );
	static statusCode       applyIteratorToLocators(AlIterator *, int &);
	static statusCode		applyIteratorToLayers(AlIterator *, int &);
	static statusCode		applyIteratorToClouds(AlIterator *, int &);
	static statusCode		applyIteratorToBlendCurves(AlIterator *, int &);

	static statusCode		writeSDLNoAnimation( const char *, API_BOOLEAN );
	static statusCode		writeSDL( const char *, API_BOOLEAN, double, double, double );

	static statusCode		frameRange(AlFrameRangeType, double&, double&, double&);
	static double			currentTime();

	static statusCode		doUpdates( API_BOOLEAN = TRUE );
	static API_BOOLEAN			updatesOn()	{ return updateOn; }
	static AlApplicationMode	applicationMode();

	static statusCode		userPrefColor( int, int&, int&, int& );
	static statusCode		setUserPrefColor( int, int, int, int );

	static API_BOOLEAN		constructionPlaneInEffect();
	static statusCode		constructionPlaneTransform( AlTM& );

	static statusCode		deleteAll();

	static statusCode		retrieve( const char *, void(*)(int) );

	static void             reverseFlippedObjects();

    static API_BOOLEAN		isOldWireFile( const char * );
    static API_BOOLEAN		isOldWireFile( const wchar_t * );

    // Store TIF image, only available in OpenAlias:
    static statusCode       storeCurrentWindow( const char* filename, int width, 
                                                int height, bool wantAntiAlias );
#ifndef WIRE_LIBRARY
	static char*			sessionName();
	static void				setSessionName( const char * );
#endif

protected:
	static void				setCurrentChannelToNull();
	static void				setCurrentActionToNull();

private:
	static void				initMessages();
	static void				finiMessages();

	static statusCode		retrieveMain( const char * );
	static int				alDeleteObject( AlObject* );

	// Animation private data
	static void				*lastChannel;	// list element for last Aa_Channel
	static void				*lastAction;	// list element for last Aa_Action
	static void				*lastExprChan;	// list element for last expression Aa_Channel
	static void				*lastCamera;	// list element for last AN_Item
	static char				*sessionNameStr;

#ifdef _WIN32
public:
#endif
	static API_BOOLEAN			updateOn;
	static API_BOOLEAN			updateNeeded;

	// curve private data
	static AlPtrTable		trimCurveTable;
};

typedef AlUniverse AlStage;

#endif /* _AlUniverse */
