/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlTrimRegion - A list of trim boundaries defining a trim region.
//
//	.SECTION Description
//		This class defines a region of a surface's parametric space which
//		identifies an area of interest on a surface.  It holds at least one
//		AlTrimBoundary object. The first boundary on the list represents
//		the outermost boundary of a trim region and the rest represent
//		the inner boundaries (holes) if there are any.
//
//		For each trim boundary, the surface is to the left of the boundary
//		when viewed from above (hence a hole is to the right of the boundary).
//
//		For example, in this diagram we have a surface represented by A's.
//		Each additional letter is an island trimmed out of A and each would
//		be a separate AlTrimRegion.  The AlTrimRegion for A would have two
//		AlTrimBoundaries, B would have three, and C, D, and E would have
//		one each.
//	.nf
// .ft C
//		AAAAAAAAAAAAAAAAAAAAAAAAAAAA
//		AAAAAAAAAAAAAAAAAAAAAAAAAAAA
//		AA                        AA
//		AA BBBBBBBBBBB CCCCCCCCCC AA
//		AA BBBBBBBBBBB CCCCCCCCCC AA
//		AA BB     BBBB            AA
//		AA BB EEE BBBBBBBBBBBBBBB AA
//		AA BB EEE BBBB         BB AA
//		AA BB     BBBB DDDDDDD BB AA
//		AA BBBBBBBBBBB DDDDDDD BB AA
//		AA BBBBBBBBBBB         BB AA
//		AA BBBBBBBBBBBBBBBBBBBBBB AA
//		AA                        AA
//		AAAAAAAAAAAAAAAAAAAAAAAAAAAA
//		AAAAAAAAAAAAAAAAAAAAAAAAAAAA
// .ft P
//	.fi
//
//

#ifndef _AlTrimRegion
#define _AlTrimRegion

#include <AlModel.h>
#include <AlObject.h>

class AlTM;
class AlTrimBoundary;
class AlIterator;
class AlDagNode;
class AlPersistentID;

class STUDIOAPI_DECL AlTrimRegion : public AlObject {
	friend class		AlFriend;

public:
	virtual					~AlTrimRegion();
	virtual AlObject*		copyWrapper() const;

	virtual AlObjectType	type() const;

	AlDagNode*				parentDagNode() const;

	AlTrimBoundary*			firstBoundary() const;
	statusCode				applyIteratorToBoundaries( AlIterator*, int& ) const;

	AlTrimRegion*			nextRegion() const;
	statusCode				nextRegionD();

	AlTrimRegion*			prevRegion() const;
	statusCode				prevRegionD();

// Stuff to get surface information

	curveFormType           uForm() const;
	curveFormType           vForm() const;
	int                     uDegree() const;
	int                     vDegree() const;
	int                     uNumberOfSpans() const;
	int                     vNumberOfSpans() const;
	int                     uNumberOfCVs() const;
	int                     vNumberOfCVs() const;

	statusCode              periodicToNonPeriodic( int, int );
	statusCode              periodicToNonPeriodic() const;

	statusCode              CVsWorldPosition( double[], int[], int[])const;
	statusCode              CVsAffectedPosition( const AlTM&, double[], int[], int[]) const;
	statusCode              CVsUnaffectedPosition( double[], int[], int[])const;
	int                     uNumberOfKnots() const;
	int                     vNumberOfKnots() const;
	statusCode              uKnotVector( double[] ) const;
	statusCode              vKnotVector( double[] ) const;
	int                     uNumberOfCVsInclMultiples() const;
	int                     vNumberOfCVsInclMultiples() const;
	statusCode              CVsWorldPositionInclMultiples( double[] ) const;
	statusCode              CVsAffectedPositionInclMultiples( const AlTM&, double[] ) const;
	statusCode              CVsUnaffectedPositionInclMultiples( double[] )const;
	
	statusCode				persistentID( AlPersistentID *&id, int ut = 1 );
	statusCode				hasPersistentID( int ut = 1 );
	statusCode				setPersistentID( AlPersistentID &id, int ut );

	int                     realuNumberOfKnots() const;
	int                     realvNumberOfKnots() const;
	statusCode              realuKnotVector( double[] ) const;
	statusCode              realvKnotVector( double[] ) const;

	statusCode				eval(double,double,API_BOOLEAN,double P[3]=0, double Pu[3]=0, double Pv[3]=0, double n[3]=0,API_BOOLEAN=FALSE,API_BOOLEAN=FALSE ) const;

protected:
							AlTrimRegion( void* );
private:
	void*					fParent;
};

#endif
