/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlTexture - Base object for representing texture data
//
//	.SECTION Description
//		This class encapsulates the basic functionality for checking and
//		setting the name of a texture as well as accessing the textures that
//		this texture refers to, and the animation on this texture.
//		These texture objects can be obtained from the AlShader class and the
//		AlEnvironment class.
//
//		A texture object itself may also reference other textures.  For
//		this reason the firstTexture() and nextTexture() methods are used.
//
//		firstTexture() returns the first texture that the texture object
//		references.  nextTexture() moves from a given referenced texture
//		to the next texture in order, as related to the texture object.
//		(See the similar methods for the AlShader/AlEnvironment classes.)
//
//		The animation on a texture can be accessed through the
//		firstChannel() and nextChannel() methods.  All the channels on the
//		texture can be deleted by calling deleteAnimation().
//
//		The texture parameters can be accessed through the parameter() and
//		setParameter() methods.  Each texture has a specific set of parameters
//		that are valid for it that depend on its type.  The full list of
//		texture parameters can be seen in the file AlShadingFields.h.  For example,
//		all parameters specific to the water texture have names of the form
//		kFLD_SHADING_WATER_*.  Parameters common to all textures have the form
//		kFLD_SHADING_COMMON_TEXTURE_*.  All parameters are treated as doubles
//		even though this may not necessarily be what they are.  This was done
//		to make the interface as simple and consistent as possible.
//
//		New AlTextures cannot be created by allocating a new AlTexture object
//		because an AlTexture must be associated with a shader.  Use the
//		AlShader::addTexture() method to create the new AlTexture.
//

#ifndef _AlTexture
#define _AlTexture

#include <AlObject.h>
#include <AlAnimatable.h>
#include <AlShadingFields.h>

struct UI_Widget;
struct IR_ShaderEntry;
struct AlPixel;

class AlIterator;
class AlPolyset;

class STUDIOAPI_DECL AlTexture : public AlObject,
				public AlAnimatable {
	friend class			AlFriend;

public:

	virtual					~AlTexture();
	virtual statusCode		deleteObject();
	virtual AlObject		*copyWrapper() const;


	virtual AlObjectType	type() const;
	virtual const char*		name() const;
	virtual statusCode		setName(const char *);

	AlTexture*				firstTexture() const;
	AlTexture*				nextTexture( const AlTexture* ) const;
	statusCode				nextTextureD( AlTexture* );

	const char*				textureType() const;
	const char*				fieldType() const;
	statusCode				parameter( const AlShadingFields, double& ) const;

	statusCode				setParameter( const AlShadingFields, const double );

	/* this section is only for file textures */
	const char*				filename() const;
	statusCode				setFilename( const char* );
	const char*				firstPerObjectPixEntry();
	const char*				nextPerObjectPixEntry( const char *);
	const char*				getPerObjectPixFilename( const char*) const;
	statusCode				setPerObjectPixFilename( const char*, const char*);
	statusCode				addPerObjectPixEntry( const char*, const char*);
	statusCode				removePerObjectPixEntry( const char* );

	API_BOOLEAN					isParametric( void ) const;
	statusCode				eval( double, double, double, double, double, double, double, double, double, double, double*, double*, double*, double*, API_BOOLEAN = TRUE, double = 1.0, double = 0.0, char *objectname = NULL ) const;

	statusCode				parameterize( AlPolyset *, const int,
										  const API_BOOLEAN,const API_BOOLEAN ) const;

	AlList*					fields() const;
	AlList*					mappedFields() const;

	statusCode				addTexture( const char*, const char*, AlTexture** returnedTexture = NULL );
	statusCode				removeTexture( const char* );

	AlTextureNode*			textureNode() const;

	statusCode				applyIteratorToTextures( AlIterator*, int& );
	
	/* for environment textures only */
	statusCode				environmentPixMap( AlPixel *, int, API_BOOLEAN );

protected:
							AlTexture();
	static statusCode		deleteObject(IR_ShaderEntry*);

private:
	API_BOOLEAN					extractType( int&, void*&, void *& ) const;
};

#endif // _AlTexture
