/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlSurfaceCV - Encapsulates methods common to surface CVs.
//
//	.SECTION Description
//
//		AlSurfaceCV is the class used to access and manipulate
//		surface CV's (also referred to as Control Points).
//		There is one AlSurfaceCV object for each CV of an AlSurface as you see
//		it in the interactive Alias package.  You cannot create or delete
//		an AlSurfaceCV.  AlSurfaceCV's are only created through AlSurface's.
//
//		There are methods to query and set the multiplicity of a CV,
//		and method to query the world position and the "unaffected"
//		position of a CV.  The "unaffected" position of a CV is its
//		position BEFORE any transformations (from its parent dag node
//		or from affecting clusters) have been applied to it.
//		There is also a method to access the clusters that affect a CV.
//		Methods setWorldPosition are used for setting either the
//		unaffected or affected(clusters) position of a CV.
//
//		For more information on CVs please see AlCurveCV.
//

#ifndef _AlSurfaceCV
#define _AlSurfaceCV

#include <AlObject.h>
#include <AlClusterable.h>
#include <AlAnimatable.h>
#include <AlSettable.h>
#include <AlPickable.h>

class AlTM;
struct ag_snode;
typedef ag_snode AG_SNODE;
struct Spline_surface;

class STUDIOAPI_DECL AlSurfaceCV	: public AlObject , public AlClusterable , public AlAnimatable , public AlSettable , public AlPickable {
	friend class		AlFriend;
	friend class		AlSurface;
public:

	virtual				~AlSurfaceCV();
	virtual AlObject*	copyWrapper() const;


	AlObjectType		type() const;

	int					uIndex() const;
	int					vIndex() const;

	AlSurfaceCV*		nextInU() const;
	AlSurfaceCV*		nextInV() const;
	AlSurfaceCV*		prevInU() const;
	AlSurfaceCV*		prevInV() const;

	statusCode			nextInUD();
	statusCode			nextInVD();
	statusCode			prevInUD();
	statusCode			prevInVD();

	int					multiplicityInU() const;
	int					multiplicityInV() const;
	statusCode			worldPosition( double&, double&, double&, double& ) const;
	statusCode			affectedPosition( const AlTM&, double&, double&, double&, double& ) const;
	statusCode			unaffectedPosition( double&, double&, double&, double& ) const;

	statusCode			setMultiplicity( int, int );
	statusCode			setWorldPosition( double, double, double, double, API_BOOLEAN );
	statusCode			setWorldPosition( double, double, double, double, AlTM );
	statusCode			setWorldPosition( double, double, double, API_BOOLEAN );
	statusCode			setWorldPosition( double, double, double, AlTM );
	statusCode			setUnaffectedPosition( double, double, double, double );

	AlSurface*			surface() const;

	statusCode			blindData( int, long&, const char *& );
	statusCode 			setBlindData( int, long, const char * );
	statusCode          removeBlindData( int );

	statusCode			applyIteratorToCVsInU( AlIterator* iter, int &rc );
	statusCode			applyIteratorToCVsInV( AlIterator* iter, int &rc );

	statusCode			doUpdates( API_BOOLEAN newState = TRUE );

protected:
						AlSurfaceCV( Spline_surface* );
private:
	API_BOOLEAN				updateOn;
	API_BOOLEAN				updateNeeded;
	void				updatePerform();

	virtual API_BOOLEAN		extractType( int&, void*&, void*& ) const;

	static void         initMessages();
	static void         finiMessages();

	Spline_surface*		fParent;
};
#endif
