/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
 
//
//	.NAME AlShader - Base object for representing shader data
//
//	.SECTION Description
//		This class encapsulates the basic functionality for checking and
//		setting the name of a shader as well as accessing the textures that
//		a particular shader refers to, and the animation on the shader.
//		Shader objects are accessible through both the AlUniverse class and
//		the objects that reference them (AlSurface and AlFace classes).
//
//		A shader object may reference textures.  The firstTexture() and
//		nextTexture() methods are used to access these textures.
//
//		firstTexture() returns the first texture that the shader object
//		references.  nextTexture() moves from a given referenced texture
//		to the next texture in order, as related to the shader object.
//		(See the similar methods for the AlTexture/AlEnvironment classes.)
//
//		The animation on a shader can be accessed through the
//		firstChannel() and nextChannel() methods.  All the channels on the
//		shader can be deleted by calling deleteAnimation().
//
//		The shader parameters can be accessed through the parameter() and
//		setParameter() methods.  Each shader has a specific set of parameters
//		that are valid for it that depend on its type.  The full list of
//		shader parameters can be seen in the file AlShadingFields.h.  For example,
//		all parameters specific to the Blinn shader have names of the form
//		kFLD_SHADING_BLINN_*.  Parameters common to all shaders have the form
//		kFLD_SHADING_COMMON_*.  All parameters are treated as doubles even
//		though this may not necessarily be what they are.  This is done to
//		make the interface as simple and consistent as possible.
//

#ifndef _AlShader
#define _AlShader

#include <AlObject.h>
#include <AlAnimatable.h>
#include <AlAnim.h>
#include <AlPixFile.h>

class AlList;
class AlIterator;

class STUDIOAPI_DECL AlShader : public AlObject , public AlAnimatable {
	friend class			AlFriend;
public:
							AlShader();
	virtual					~AlShader();
	virtual statusCode		deleteObject();
	virtual AlObject		*copyWrapper() const;

	statusCode				create();


	virtual AlObjectType	type() const;

	virtual const char*		name() const;
	virtual statusCode		setName( const char * );

	statusCode				parameter( const AlShadingFields, double& ) const;
	statusCode				setParameter( const AlShadingFields, const double );

	statusCode				blindData( int, long& , const char*& );
	statusCode				setBlindData( int, long, const char* );
	statusCode				removeBlindData( int user_type );

	const char*				shadingModel() const;
	statusCode				setShadingModel( const char* );

	AlTexture*				firstTexture() const;

	AlTexture*				nextTexture( const AlTexture* ) const;
	statusCode				nextTextureD( AlTexture* ) const;

	AlList*					fields() const;
	AlList*					mappedFields() const;

	statusCode				addTexture( const char*, const char*, AlTexture** returnedTexture = NULL );
	statusCode				removeTexture( const char* );

	statusCode				applyIteratorToTextures( AlIterator*, int& );

	AlShader*				copyObject();
	API_BOOLEAN					isUsed();
	
	AlShader*				convertSolidToFileTexture( AlDagNode *, int ,API_BOOLEAN , const char *  );
	AlShader*				convertEnvironmentToFileTexture( int , const char *, AlPixFile::Format, API_BOOLEAN  );

private:
	static void				initMessages();
	static void				finiMessages();

	API_BOOLEAN					extractType( int&, void *&, void *& ) const;
};

#endif // _AlShader
