/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
//
//	.NAME AlRetrieveOptions - A structure used to transfer options that control
//		  the AlUniverse::retrieve() method.
//
//	.SECTION Description
//		An AlRetrieveOptions structure is used set or determine the options
//		that the AlUniverse::retrieve() method uses to tailor the import
//		of files.
//		To avoid setting all of the fields of AlRetrieveOptions using
//		AlUniverse::setRetrieveOptions(), it is recommended that the
//		current values be acquired first using AlUniverse::retrieveOptions().
//
// .br
//		For example:
//
//	.nf
//		// No error checking is done here for brevity in the example.
//		AlRetrieveOptions	options;
//		AlUniverse::retrieveOptions( options );
//		options.autocad.units = k_millimeters;
//		options.autocad.in_create_shaders = True;
//		AlUniverse::setRetrieveOptions( options );
//
//	.fi
//		See File->Retrieve in the Alias manuals for a more detailed
//		description of these options.
//
//	General Options:
//
//	%cnew_stage
//		Determines if all retrieved geometry is put in the current stage,
//		or is put in a new stage named after the filename. If set to TRUE,
//		then the retrieved geometry will be put in a new stage called
//		<filename><ext> (e.g. model.iges -> modeliges) and this stage will
//		be made current.
//
//	Wire File Options:
//
//		All wire file options default to TRUE.
//
//	%ckeep_windows
//		If set to TRUE, the modelling window layout contained in wire files
//		will be retrieved, and if set to FALSE, the window layout will not be
//		retrieved.
//
//	%ckeep_cameras
//		If set to TRUE, the cameras contained in wire files will be retrieved,
//		and if set to FALSE, the cameras will not be retrieved.
//
//	%ckeep_animation
//		If set to TRUE, both the animation and the model contained in wire
//		files will be retrieved, and if set to FALSE, only the model will be
//		retrieved.
//
//	Note: If a window contains an animated camera and keep_animation is TRUE
//		  then this window will be retrieved also, even if keep_windows is
//		  FALSE.
//
//	%ckeep_backgrounds
//		If set to TRUE, the background contained in wire files will be
//		retrieved, and if set to FALSE, the background will not be retrieved.
//
//	%ckeep_unitsTolerances
//		If set to TRUE, linear units, angular units, and construction
//		toleranced contained in the wire will be retrieved and set.
//		If set to FALSE, the units and tolerances in the wire file are
//		ignored. Only relevent for OpenAlias plugins.
//
//	%ckeep_renderGlobals
//		If set to TRUE, the render global information stored in a wire file will
//		be retrieved and set. If set to FALSE, this information will be
//		ignored. Only relevent for OpenAlias plugins.
//
//	IGES/VDAIS/C4X/JAMA-IS and VDAFS File Options:
//
//	%cgroup
//		If set to TRUE, all geometry retrieved from the file will be
//		grouped under a node named after the translator. For example, IGES
//		geometry will be grouped under a node called "IGES_FILE". and STEP
//		geometry will be grouped under a node called "STEP_FILE".
//		If set to FALSE, this group node node will not be created.
//		The default is FALSE.
//
//	%ccoalesce
//		If set to TRUE, multiple knots will be removed based on continutity
//		in IGES Parametric Curve and Surface geometry, and all VDAFS curve and
//		surface geometry. The default is FALSE.
//
//	%cannotation
//		If set to TRUE, supported geometric entities that have been flagged
//		for use as annotation will be retrieved. If set to FALSE, these entities
//		will not be retrieved. This option does not apply to VDAFS.
//
//	%ctrim_to_surf
//		If set to TRUE, retrieved Trimmed or Bounded surfaces whose boundaries
//		are the same as, or iso-parametric to, the natural boundaries of the
//		untrimmed surface, will be converted to untrimmed surfaces by shrinking
//		the	surface to the trim boundaries. If set to FALSE, trimmed surfaces
//		will be represented as such even if the trim boundary information is
//		unnecessary. The default is TRUE.  This option does not apply to VDAFS.
//
//	%cscale
//		All retrieved geometry will be scaled by this factor.
//		The default is 1.0.
//
//  %clayer_or_set
//      This controls the mapping of the IGES/VDAIS/JAMIAS Level. The options are
//      a) kLayer, maps level to layer, this is the default
//      b) kSet, maps level to set
//      c) kLayerAndSet, maps level to layer and set
//      d) kNoLayerOrSet, there is no mapping level is ignored.
//      This option does not apply to VDAFS or C4
//
//  %cdefault_trim
//      This controls the type of curves that will be used in trimming a surface.
//      The options are 
//      a) kParameter, use the parameter (2D) space curves, this is 
//         the default. 
//      b) kWorld, use the world (3D) space curves
//      c) kUseFlag, use the preference in the IGES/VDAIS/JAMAIS/C4 file
//      NOTE kUseFlag is invalid for VDAFS.
//
//  %ccreate_scan_data
//      This control the conversion of "Scan Data" 106 Forms 1-3, 11-13 and 
//      63. If on this entity is treated as Scan Data on IGES import.
//      During VDAFS import, PSET and MDI entities are treated as Scan Data.
//
//	DWG/DXF File Options.
//
//	%cgroup
//		See description above.
//
//	%cstitch
//		Determines whether to stitch the surfaces of a solid into a shell (TRUE)
//		or to group them instead (FALSE).
//
//	%canonymous_blocks
//		Determines whether "anonymous" DWG/DXF BLOCK entities are processed (TRUE)
//		or ignored (FALSE). The default is FALSE, since mostly, these BLOCK
//		entities contain pattern hatching lines.
//
//	%cunits
//		If the units of the DWG/DXF coordinate data is known, it can be set using
//		this option so that the data is properly scaled. This option is
//		necessary since the units of the DWG/DXF coordinate data is not stored
//		in the file. The default is inches (kInches), but it can be set to
//		any of the following values: kMiles, kFeet, kInches, kMils,
//		kMicroInches, kKilometers, kMeters, kCentimeters, kMillimeters, or
//		kMicrons.
//
//	%cscale
//		 See above for description of this field.
//
*/

#ifndef _AlRetrieveOptions
#define _AlRetrieveOptions

	enum AlUnitsType {
		kUnknown,  // Unknown unit (should not be used).
		kMicrons,
		kMillimeters,
		kCentimeters,
		kMeters,
		kKilometers,
		kMicroInches,
		kMils,
		kInches,
		kFeet,
		kMiles
	};

    enum AlLayerOrSetType {
        kLayer,
        kSet,
        kLayerAndSet,
        kNoLayerOrSet
    };

    enum AlTrimType {
        kParameterTrim,
        kWorldTrim,
        kUseFlagTrim
    };
     

typedef struct AlRetrieveOptions {

/* Options that apply to all file types */

	API_BOOLEAN	new_stage;					/* == TRUE all retrieved geometry will*/
										/*    placed in a new stage whose name*/
										/*    is based on the filename        */
										/*    (default is FALSE)              */

/* Wire File options */

	struct _wire {
		API_BOOLEAN	keep_windows;  			/* == TRUE windows will be retrieved  */
										/*    (default is TRUE) */

		API_BOOLEAN	keep_cameras; 			/* == TRUE cameras will be retrieved  */
										/*    (default is TRUE) */

		API_BOOLEAN	keep_lights; 			/* == TRUE lights will be retrieved  */
										/*    (default is TRUE) */

		API_BOOLEAN	keep_shaders; 			/* == TRUE shaders will be retrieved  */
										/*    (default is TRUE) */

		API_BOOLEAN	keep_animation;			/* == TRUE animation will be retrieved*/
										/*     (default is TRUE)              */

		API_BOOLEAN	keep_backgrounds;		/* == TRUE background will be         */
										/*    retrieved. (default is TRUE)    */

		API_BOOLEAN	keep_unitsTolerances;	/* == TRUE linear/angular units and   */
										/*    construction tolerances will    */
										/*    be retrieved and set.           */
										/*    (default is TRUE)               */

		API_BOOLEAN	keep_renderGlobals;     /* == TRUE render global settings     */
										/*    will be retrieved and set       */
										/*    (default is TRUE)               */

		API_BOOLEAN keep_layers;			/* == TRUE */

		API_BOOLEAN keep_bookmarks;			/* == TRUE */
	} wire;

/* IGES File options */

	struct _iges {
		API_BOOLEAN	group;			    	/* == TRUE retrieved geometry will be */
										/*    grouped under a node            */
										/*    (default is FALSE)			  */

		API_BOOLEAN	coalesce;				/* == TRUE multiple knot removal base */
										/*    continutity will occur for      */
										/*    retrieved parametric spline     */
										/*    geometry( Entities 112 and 114) */
										/*    (default is FALSE)			  */

		API_BOOLEAN	annotation;  			/* == TRUE IGES geometry entities     */
										/*    flagged as annotation will be   */
										/*    retrieved.                      */
										/*    (default is FALSE)              */

		API_BOOLEAN	trim_to_surf;      	    /* == TRUE IGES Trimmed and Bounded   */
										/*    surfaces (entities 144 and 143) */
										/*    will be represented as untrimmed*/
										/*    surfaces if possible            */
										/*    (default is TRUE)               */

		float	scale;				    /*    All retrieved geometry will be  */
										/*    scaled by this factor.          */
										/*    (default is 1.0)                */

        API_BOOLEAN create_scan_data;       /* == TRUE Scan Data will be created  */
		                                /*    from 106 Form 1-3, 11-13 and    */
		                                /*    63                              */
		                                /*    (default is FALSE)              */
	} iges;

/* VDA-IS File options  - see IGES options for field descriptions */
	struct _vdais {
		API_BOOLEAN	group;
		API_BOOLEAN	coalesce;
		API_BOOLEAN	annotation;
		API_BOOLEAN	trim_to_surf;
		float	scale;
	} vdais;

/* C4X File options  - see IGES options for field descriptions */
	struct _c4 {
		API_BOOLEAN	group;
		API_BOOLEAN	coalesce;
		API_BOOLEAN	annotation;
		API_BOOLEAN	trim_to_surf;
		float	scale;
	} c4;

/* JAMA-IS File options  - see IGES options for field descriptions */
	struct _jamais {
		API_BOOLEAN	group;
		API_BOOLEAN	coalesce;
		API_BOOLEAN	annotation;
		API_BOOLEAN	trim_to_surf;
		float	scale;
	} jamais;

/* VDAFS File options  - see IGES options for field descriptions */

	struct _vdafs {
		API_BOOLEAN	group;
		API_BOOLEAN	coalesce;
		float	scale;
        API_BOOLEAN create_scan_data;       /* == TRUE Scan Data will be created  */
		                                /*    from VDAFS PSET and MDI entities*/
		                                /*    (default is FALSE)              */
	} vdafs;

/* DWG/DXF File options */
	struct _autocad {
	
		/*
		 * dwg and dxf export use the same translator but they each owns
		 * different option values because they have separate UIs.
		 */
		API_BOOLEAN		wantCurves_dwg;
		API_BOOLEAN		splitG1Breaks_dwg;	

		API_BOOLEAN		wantCurves_dxf;
		API_BOOLEAN		splitG1Breaks_dxf;

		int				file_type;	/* 0-dwg, 1-dxf	*/
	} autocad;
   
    struct _level_map {
        int iges_layer_or_set;          /* == kLayer create layer from level  */
                                        /* == kSet create set from level      */
                                        /* == kLayerAndSet create layer and   */
		                                /*    set from level                  */
                                        /* == kNoLayerOrSet no mapping        */
        int vdais_layer_or_set;         /* See IGES description of this field */
        int jamais_layer_or_set;        /* See IGES description of this field */
    } level_map;

    struct _trimming_map {
        int iges_default_trim;               /* == kParameter trim with parameter  */
		                                     /*    space curves                    */  
                                             /* == kWorld trim with world space    */
		                                     /*    curves                          */
		                                     /* == kUseFlag use preference flag in */
		                                     /*    file                            */
        int vdais_default_trim;              /* See IGES description of this field */
        int c4_default_trim;                 /* See IGES description of this field */
        int jamais_default_trim;             /* See IGES description of this field */
        int vdafs_default_trim;              /* See IGES description of this field */
		                                     /* kUseFlag is not valid for vdafs    */
    } trimming_map;

    struct _edf {
		   API_BOOLEAN retrieve_want_groups;
           API_BOOLEAN retrieve_want_log;
	} edf;

    struct _fbx {
           API_BOOLEAN retrieve_want_log;
           API_BOOLEAN retrieve_want_optimization;
           API_BOOLEAN retrieve_want_curves;
           API_BOOLEAN retrieve_want_bookmarks;
           API_BOOLEAN retrieve_want_lights;
           API_BOOLEAN retrieve_want_cameras;
           API_BOOLEAN retrieve_want_shaders;
           API_BOOLEAN retrieve_want_divide_periodic;
           API_BOOLEAN retrieve_want_z_axis;
           API_BOOLEAN retrieve_want_subdivmesh;
	} fbx;

    struct _sat {
           API_BOOLEAN retrieve_want_log;
           API_BOOLEAN retrieve_want_optimization;
    } sat;

    struct _iges_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_want_annotation;
	    API_BOOLEAN retrieve_want_parametric;
	    API_BOOLEAN retrieve_want_construction;
	    API_BOOLEAN retrieve_want_nested_group;
	    API_BOOLEAN retrieve_want_curve_as_wire;
	    API_BOOLEAN retrieve_want_unbounded_plane_as_wp;
	    API_BOOLEAN retrieve_want_transformation_as_wp;
	    API_BOOLEAN retrieve_want_shrink_surface;
	    API_BOOLEAN retrieve_want_scan_data;
	} iges_atf;

    struct _nx_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_associative_meshes;
	    API_BOOLEAN retrieve_facets;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
	    API_BOOLEAN retrieve_simplify_primitives;
	    API_BOOLEAN retrieve_enable_simplification;
	    API_BOOLEAN retrieve_enable_healing;
	    API_BOOLEAN retrieve_enable_edge_split_and_merge;
	    API_BOOLEAN retrieve_enable_face_split_and_merge;
	    API_BOOLEAN retrieve_preserve_surface_parameterization;
	    API_BOOLEAN retrieve_merge_faces;
	    API_BOOLEAN retrieve_empty_layers;
	    API_BOOLEAN retrieve_categories;
	    API_BOOLEAN retrieve_empty_categories;
	    API_BOOLEAN retrieve_map_category_name_to_layer;
	    API_BOOLEAN retrieve_want_category_all_files;
	    API_BOOLEAN retrieve_want_attributes_in_sub_assembly;
	    //API_BOOLEAN retrieve_want_search_files_in_folders;
	    //int        retrieve_number_search_folders;
	    float        retrieve_tolerance;
	    API_BOOLEAN retrieve_dimensions;
	    API_BOOLEAN retrieve_dimAsLabel;
	    API_BOOLEAN retrieve_layerSelect;
	    char        *retrieve_selectedLayers;
	} nx_atf;

    struct _inventor_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    int         retrieve_bodies;
	    API_BOOLEAN retrieve_meshes;
	    API_BOOLEAN retrieve_face_meshes;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
	} inventor_atf;

    struct _step_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_meshes;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
        API_BOOLEAN retrieve_constructive_geometry;
	} step_atf;

    struct _catiav5_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_meshes;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_infinites;
	    API_BOOLEAN retrieve_cloud_data;
	    API_BOOLEAN retrieve_weld_data;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
	    API_BOOLEAN retrieve_color_meshes;
	    API_BOOLEAN retrieve_merge_vertices;
	    float       retrieve_merge_vertices_tol;
        API_BOOLEAN retrieve_geometric_sets;
        API_BOOLEAN retrieve_want_annotation;
        API_BOOLEAN retrieve_want_tighter_tolerance;
        API_BOOLEAN retrieve_empty_geometric_sets;
        API_BOOLEAN retrieve_want_merge_layers;
	} catiav5_atf;

    struct _catiav4_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_meshes;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_infinites;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
	} catiav4_atf;

    struct _stl_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_colors;
	    int         retrieve_color_schema;
	    int         retrieve_units;
	    API_BOOLEAN retrieve_merge_vertices;
	    float       retrieve_merge_vertices_tol;
	} stl_atf;

    struct _jt_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_meshes;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
	    int         retrieve_lod;
	    API_BOOLEAN retrieve_merge_vertices;
	    float       retrieve_merge_vertices_tol;
	} jt_atf;

    struct _granite_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_quilts;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
        API_BOOLEAN retrieve_construction_planes;
	} granite_atf;

    struct _solidworks_atf {
	    API_BOOLEAN retrieve_want_log;
	    API_BOOLEAN retrieve_want_optimization;
	    API_BOOLEAN retrieve_nurbs;
	    API_BOOLEAN retrieve_curves;		
	    API_BOOLEAN retrieve_points;
	    API_BOOLEAN retrieve_invisibles;
	    API_BOOLEAN retrieve_keep_topology;
	    API_BOOLEAN retrieve_shrink_surfaces;
	    API_BOOLEAN retrieve_split_periodic;
	} solidworks_atf;

    // Only available in OpenAlias:
    struct _ai {
	   double      retrieve_width;
       double      retrieve_height;
       double      retrieve_tolerance;
    } ai;

} AlRetrieveOptions;

#endif
