/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//	.NAME AlRender - Interface to Alias render options
//
//	.SECTION Description
//		This class encapsulates all access to the Alias render
//		options.  Every value in the window can be retrieved and set.
//		Please see the alias manual for a description of each of the
//		parameters and their functions.
//

#ifndef _AlRender_h
#define _AlRender_h

#include <AlStyle.h>
#include <AlLinkItem.h>
#ifdef _WIN32
#undef min
#undef max
#endif

class STUDIOAPI_DECL AlRender {
public:
	//	general settings

	enum Source			{ kAll, kActive };
	enum AnimationRange	{ kGlobalRange, kMinMax, kTimeSlider };
	enum Scope			{ kPerObject, kGlobal };
	enum Quality		{ kLow, kMedium, kHigh };

	static Source				render();
	static statusCode			setRender( Source );

	static API_BOOLEAN				animation();
	static statusCode			setAnimation( API_BOOLEAN );

	static AnimationRange		animationRange();
	static statusCode			setAnimationRange( AnimationRange );

	static statusCode			animationRange(double&start, double&end, double&by );
	static statusCode			setAnimationRange(double start, double end, double by);

	static Scope				hiddenLineParms();
	static statusCode			setHiddenLineParams( Scope );

	static Scope				qualityScope( Quality );
	static statusCode			setQualityScope( Quality, Scope );

	static Quality				qualityType();
	static statusCode			setQualityType( Quality );

	static API_BOOLEAN				modifyExtensions();
	static statusCode			setModifyExtensions( API_BOOLEAN );

	static int					extensionPadding();
	static statusCode           setExtensionPadding( int );	

	static int					startExtension();
	static statusCode			setStartExtension( int );

	static int					byExtension();
	static statusCode			setByExtension( int );

	//	Global Quality Parameters: Subdivision & AntiAlias for low,med,high
	class STUDIOAPI_DECL Subdivision
	{
	public:
		enum Type	{ kAdaptive, kUniform };

		static Type				type( Quality );
		static statusCode		setType( Quality, Type );

		static int				uniformU( Quality );
		static statusCode		setUniformU( Quality, int );					 
		static int				uniformV( Quality );
		static statusCode		setUniformV( Quality, int );					 

		static int				adaptiveMin( Quality );
		static statusCode		setAdaptiveMin( Quality, int );				 
		static int				adaptiveMax( Quality );
		static statusCode		setAdaptiveMax( Quality, int );				 
		static double			adaptiveThreshold( Quality );
		static statusCode		setAdaptiveThreshold( Quality, double );		 
	};

	class STUDIOAPI_DECL AntiAlias
	{
	public:
		static int				min( Quality );
		static statusCode		setMin( Quality, int );						 
		static int				max( Quality );
		static statusCode		setMax( Quality, int );						 

		static double			threshold( Quality );
		static statusCode		setThreshold( Quality, double );				 

		static API_BOOLEAN			jitter( Quality );
		static statusCode		setJitter( Quality, API_BOOLEAN );
	};

	class STUDIOAPI_DECL Limits
	{
	public:
		static API_BOOLEAN			createPreview( Quality );
		static statusCode		setCreatePreview( Quality, API_BOOLEAN );

		static int				maxReflections( Quality );
		static statusCode		setMaxReflections(  Quality,int );		 

		static int				maxRefractions( Quality );
		static statusCode		setMaxRefractions(  Quality,int );		 

		static int				maxShadowLevels( Quality );
		static statusCode		setMaxShadowLevels(  Quality,int );		 

		static int				subdivideRecurse( void );
		static statusCode		setSubdivideRecurse( int );	 
	};

	class STUDIOAPI_DECL Composite
	{
	public:
		static API_BOOLEAN			enabled();
		static statusCode		setEnabled( API_BOOLEAN );

		static double			coverageThreshold();
		static statusCode		setCoverageThreshold( double );	 
	};

	class STUDIOAPI_DECL Blur
	{
	public:
		static API_BOOLEAN			postFilter();
		static statusCode		setPostFilter( API_BOOLEAN );

		static int				postCenter();
		static statusCode		setPostCenter( int );				 

		static int				postAdjacent();
		static statusCode		setPostAdjacent( int );				 

		static int				postDiagonal();
		static statusCode		setPostDiagonal( int );				 

		static API_BOOLEAN			motionBlur();
		static statusCode		setMotionBlur( API_BOOLEAN );

		static double			shutterAngle();
		static statusCode		setShutterAngle( double );			 
	};

	class STUDIOAPI_DECL Misc
	{
	public:
		enum GeometrySource		{ kModeler, kSDLFile };

		static GeometrySource	geometrySource();
		static statusCode		setGeometrySource( GeometrySource );

		static API_BOOLEAN			textures();
		static statusCode		setTextures( API_BOOLEAN );

		static API_BOOLEAN			skeletons();
		static statusCode		setSkeletons( API_BOOLEAN );

		static API_BOOLEAN			noTransparencyInDepth();
		static statusCode		setNoTransparencyInDepth( API_BOOLEAN );

		static double			transparency();
		static statusCode		setTransparency( double );			 

		static API_BOOLEAN			keepNonglowedImage();
		static statusCode		setKeepNonglowedImage( API_BOOLEAN );

		static API_BOOLEAN			spotLightDepthMaps();
		static statusCode		setSpotLightDepthMaps( API_BOOLEAN );

		static API_BOOLEAN			depthMapsInMemory();
		static statusCode		setDepthMapsInMemory( API_BOOLEAN );

		static API_BOOLEAN			attenuateTransparency();
		static statusCode		setAttenuateTransparency( API_BOOLEAN );
	};

	class STUDIOAPI_DECL Image
	{
	public:
		enum Format		{ kAlias, kSGI, kTIFF, kTIFF16, kRLA, kFIDO, kHARRY};
		enum DepthFormat { kDepthAlias, kDepthComposer };
		enum Fields		{ kOff, kEven, kOdd, kBoth };

		static Format			format();
		static statusCode		setFormat( Format );

		static DepthFormat		depthFormat();
		static statusCode		setDepthFormat( DepthFormat );

		static Fields			fields();
		static statusCode		setFields( Fields );

		static API_BOOLEAN			oddFirst();
		static statusCode		setOddFirst( API_BOOLEAN );

		static API_BOOLEAN			ignoreFilmGate();
		static statusCode		setIgnoreFilmGate( API_BOOLEAN );

		static double			gammaCorrection();
		static statusCode		setGammaCorrection( double );		 

		static statusCode		source( const char *, API_BOOLEAN &, API_BOOLEAN &, API_BOOLEAN & );
		static statusCode		setSource( const char *, API_BOOLEAN, API_BOOLEAN, API_BOOLEAN );

		static API_BOOLEAN			XYRatioLock();
		static statusCode		setXYRatioLock( API_BOOLEAN );

		static int				resolutionX();
		static statusCode		setResolutionX( int );				 

		static int				resolutionY();
		static statusCode		setResolutionY( int );				 

		static double			aspectRatio();
		static statusCode		setAspectRatio( double );			 

		static statusCode		readResolutions( AlList* &list );
	};

	class STUDIOAPI_DECL HiddenLine
	{
	public:
		static API_BOOLEAN			useFillColor();
		static statusCode		setUseFillColor( API_BOOLEAN );

		static statusCode		fillColor( double &, double &, double &);
		static statusCode		setFillColor( double, double, double );	 

		static statusCode		lineColor( double &, double &, double &);
		static statusCode		setLineColor( double, double, double );	 

		static int				patchLinesU();
		static statusCode		setPatchLinesU( int );				 

		static int				patchLinesV();
		static statusCode		setPatchLinesV( int );				 
	};


	class STUDIOAPI_DECL Particle
	{
	public:
		static API_BOOLEAN			show();
		static statusCode		setShow( API_BOOLEAN );

		static int				simulationSubsteps();
		static statusCode		setSimulationSubsteps( int );		 

		static int				framesPerSecond();
		static statusCode		setFramesPerSecond( int );			 
	};
};

class STUDIOAPI_DECL AlResolutionItem : public AlLinkItem {
	friend class AlRender;
	friend class AlRender::Image;

public:
	AlResolutionItem*	nextItem() { return (AlResolutionItem *) next();}
	AlResolutionItem*	prevItem() { return (AlResolutionItem *) prev();}

	char*	name;
	int		x, y;
	double	aspect;

protected:
						AlResolutionItem( const char *, int, int, double);
	virtual				~AlResolutionItem();
};
#endif
