/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//+
//
//	.NAME AlPerspectiveCamera - Encapsulates creation, deletion and manipulation of perspective cameras
//
//	.SECTION Description
//
//		This class encapsulates all the functionality for creating,
//		deleting and manipulating a perspective camera.  The user
//		cannot create, delete or manipulate orthographic cameras.
//
//		A camera is made of 4 parts - an AlPerspectiveCamera object and 3 attached
//		AlCameraNodes which represent eye, view and up position of
//		the camera.  These camera nodes are members of the universe's
//		dag structure.
//
//		The view direction vector is the vector between the eye
//		position and the view position.  The up direction vector is
//		the vector between the eye position and the up position.
//		There are methods to get these positions and to access the
//		attached camera nodes.
//
//		To create a perspective camera, the user must instantiate and
//		call the create method on an AlPerspectiveCamera object.
//		This creates the necessary eye, view and up AlCameraNodes,
//		groups them under an AlGroupNode and inserts the group into
//		the universe's dag structure.  The user cannot instantiate
//		an AlCameraNode directly.
//
//		When a camera is created, if the coordinate system is specified
//		as kZUp (when initializing the universe) the camera's default eye,
//		view and up positions are respectively (0.0, -12.0, 0.0),
//		(0.0, 0.0, 0.0), (0.0, -12.0, 1.0).  If the coordinate system is
//		specified as kYUp, then eye, view and up positions are
//		(0.0, 0.0, 12.0), (0.0, 0.0, 0.0), (0.0, 1.0, 12.0).
//
//		There are two ways to delete a camera object.  When the
//		deleteObejct() method of an AlPerspectiveCamera object is called, its
//		three camera nodes are
//		deleted.  Alternatively, when a camera node is deleted, its
//		associated camera (and other camera nodes) are deleted.
//		The group node that originally grouped the eye, view and
//		up nodes is not deleted.
//

#ifndef _AlPerspectiveCamera
#define _AlPerspectiveCamera

#include <AlCamera.h>
#include <AlAnimatable.h>
#include <AlSettable.h>
#include <AlPickable.h>

class AlCameraNode;

typedef void *	Camera_ptr;
struct Dag_node;
class AlDagNode;

class STUDIOAPI_DECL AlPerspectiveCamera	: public AlCamera , public AlSettable , public AlPickable , public AlAnimatable
{

	friend class			AlFriend;
public:
	enum AlCameraWindowFitType {
		kFillFit,
		kHorizontalFit,
		kVerticalFit
	};

public:
							AlPerspectiveCamera();
	virtual					~AlPerspectiveCamera();
	virtual statusCode		deleteObject();
	virtual AlObject*  		copyWrapper() const;
	statusCode				create();


	virtual AlObjectType	type() const;

	AlCameraNode*			eye() const;
	AlCameraNode*			view() const;
	AlCameraNode*			up() const;

	statusCode				worldEye( double&, double&, double& ) const;
	statusCode				worldView( double&, double&, double& ) const;
	statusCode				worldUp( double&, double&, double& ) const;
	statusCode				worldEyeViewUp( double&, double&, double&, double&, double&, double&, double&, double&, double& ) const;

	statusCode				setWorldEye( double, double, double);
	statusCode				setWorldView( double, double, double );
	statusCode				setWorldUp( double, double, double );
	statusCode				setWorldEyeViewUp( double, double, double, double, double, double, double, double, double );

	double					twistAngle() const;

	statusCode				setTwistAngle( double );
	statusCode				changeTwistAngleBy( double );

	statusCode				filmBack(double&, double&) const;
	statusCode				filmOffset(double&, double&) const;
	statusCode				setFilmBack(double, double);
	statusCode				setFilmOffset(double, double);

	double					focalLength() const;
	statusCode				setFocalLength(double);

	double					angleOfView() const;
	statusCode				setAngleOfView( double );

	statusCode				depthOfField(API_BOOLEAN&, double&, double&) const;
	statusCode				setDepthOfField(API_BOOLEAN, double, double);

	int						placementFitCode() const;
	double					placementShift() const;
	statusCode				setPlacementFitCode( AlCameraWindowFitType );
	statusCode				setPlacementShift( double );

	statusCode				autoFocus( API_BOOLEAN &enabled ) const;
	statusCode				setAutoFocus( API_BOOLEAN enabled );

protected:
	Dag_node				*eyeDagNode() const;
	Dag_node				*viewDagNode() const;
	Dag_node				*upDagNode() const;

private:
	// used by AlAnimatable .. parent is not needed
	virtual API_BOOLEAN extractType( int&, void*&, void*& ) const;

	static void				initMessages();
	static void				finiMessages();
};

#endif
