/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlParamAction - Basic interface to derived class of actions for parameter curve actions.
//
//	.SECTION Description
//		An AlParamAction is derived from an AlAction.  This particular
//	action has a list of AlKeyframes which are CVs on a Poon-Ross spline.
//	This spline is basically a Hermite-linear spline.  That is, it is
//	a Hermite in y (the vertical axis) and a linear in x (the horizontal
//	axis).
//
//		In order to create an AlParamAction, you must have at least one
//	valid, created AlKeyframe which will be the first keyframe of the
//	action.  After the AlParamAction is created, you can add other keyframes
//	to the AlParamAction.  Note that two AlKeyframes cannot have the same
//	location.  If you add a keyframe to the action that has the same location
//	as an existing keyframe of the action, the existing keyframe will be
//	deleted.  Since an AlKeyframe is created at (0, 0), you cannot create
//	a bunch of AlKeyframes, add them to the AlParamAction, and then modify
//	their locations later, since the AlParamAction will only have one
//	keyframe (the others will have been deleted as each successive keyframe
//	with the same location is added to the AlParamAction).  You must set
//	the location of the AlKeyframe before adding it to the AlParamAction.
//
//		If you copy an AlParamAction, all the keyframes (and the keyframes'
//	streams) will also be copied.  If you delete an AlParamAction, all
//	its Keyframes will be deleted.
//

#ifndef _AlParamAction
#define _AlParamAction

#include <AlAction.h>
#include <AlKeyframe.h>

class AlIterator;

class STUDIOAPI_DECL AlParamAction : public AlAction {
	friend					class AlFriend;
	friend					class AlDictionaryOperators;

public:
							AlParamAction();
	virtual					~AlParamAction();
	virtual AlObject*		copyWrapper() const;

	virtual AlObjectType	type() const;

	AlKeyframe*				firstKeyframe() const;
	AlKeyframe*				lastKeyframe() const;

	statusCode				addKeyframe( double, double, AlKeyframe *&, API_BOOLEAN = TRUE, AlTangentType = kTangentFlat, AlTangentType = kTangentFlat );
	statusCode				addKeyframe( double, double, AlKeyframe &, API_BOOLEAN = TRUE, AlTangentType = kTangentFlat, AlTangentType = kTangentFlat );
	statusCode				addKeyframe( double, double, API_BOOLEAN = TRUE, AlTangentType = kTangentFlat, AlTangentType = kTangentFlat );

	int						numberOfKeyframes() const;

	statusCode 				applyIteratorToKeyframes( AlIterator*, int& );

	statusCode				addKeyframeRange( int, double*, double*, API_BOOLEAN = TRUE, AlTangentType* = NULL, AlTangentType* = NULL );
	statusCode				deleteKeyframeRange( double, double );

protected:
	virtual AlAction*		constructAction( Aa_Action* ) const;

private:
	static void				realRecomputeTangents( Aa_Action* );
	statusCode				create();
};

#endif	// _AlParamAction
