/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlObject - Base class for all Alias Data types.
//
//	.SECTION Description
//		This is an abstract base class which holds a reference to an
//		anonymous data structure.  Derived classes will initialize the
//		reference to refer to a particular data structure.  This class
//		provides a mechanism for completely hiding the internal data
//		structures from the user.
//
//		This class gives some homogeneity between several different
//		classes by providing methods which derived classes will redefine
//		so that the name and type of the object can be obtained and so
//		that object down casting can be done safely.
//
//		There are a group of functions (the 'downcasting' methods )for casting
//		an AlObject to one of its derived classes.  They have the form
//:
//:		Al{DerivedObject} *AlObject::as{DerivedObject}Ptr()
//:
//		where {DerivedObject} is the name of the derived class.
//:
//		All these functions return NULL in the base class, and the derived
//		class of the appropriate type simply returns a pointer to itself.
//		In this way, safe typecasting can be maintained.
//

#ifndef _AlObject
#define _AlObject

#include <AlHashable.h>
#include <AlAccessTypes.h>

class AlAmbientLight;
class AlAreaLight;
class AlCamera;
class AlCameraNode;
class AlCloud;
class AlCluster;
class AlClusterMember;
class AlClusterNode;
class AlCurveCV;
class AlCurve;
class AlCurveNode;
class AlCurveOnSurface;
class AlDagNode;
class AlDirectionLight;
class AlFace;
class AlFaceNode;
class AlGroupNode;
class AlLight;
class AlLightLookAtNode;
class AlLightNode;
class AlLightUpNode;
class AlLinearLight;
class AlNonAmbientLight;
class AlObjectList;
class AlPointLight;
class AlSet;
class AlSetMember;
class AlSpotLight;
class AlVolumeLight;
class AlSphereLight;
class AlCylinderLight;
class AlBoxLight;
class AlConeLight;
class AlTorusLight;
class AlSurfaceCV;
class AlSurface;
class AlSurfaceNode;
class AlSurfaceCurve;
class AlShellNode;
class AlShell;
class AlTrimBoundary;
class AlTrimCurve;
class AlTrimRegion;
class AlShader;
class AlSwitchShader;
class AlLayeredShader;
class AlTexture;
class AlTextureNode;
class AlEnvironment;
class AlUniverse;
class AlChannel;
class AlAction;
class AlKeyframe;
class AlParamAction;
class AlMotionAction;
class AlPolysetNode;
class AlPolysetVertex;
class AlPolygon;
class AlPolyset;
class AlMeshNode;
class AlMesh;
class AlCreate;
class AlAttributes;
class AlArcAttributes;
class AlLineAttributes;
class AlCurveAttributes;
class AlPlaneAttributes;
class AlConicAttributes;
class AlRevSurfAttributes;
class AlJoint;
class AlConstraint;
class AlOrientationConstraint;
class AlPointConstraint;
class AlAimConstraint;
class AlIKHandle;
class AlIKHandleNode;

class AlClusterable;
class AlAnimatable;
class AlSettable;
class AlPickable;

class AlContact;
class AlCommandRef;
class AlCommand;
class AlLayer;

class AlOrthographicCamera;
class AlPerspectiveCamera;
class AlWindow;
class AlImagePlane;

class AlLocator;
class AlDistanceLocator;
class AlAnnotationLocator;
class AlAngleLocator;
class AlRadialLocator;
class AlDeviationLocator;
class AlMinmaxLocator;

class AlConstructionEntity;
class AlConstructionVector;
class AlConstructionPlane;
class AlPoint;
class AlCurvePoint;
class AlSpacePoint;
class AlSurfacePoint;
class AlCurveOnSurfacePoint;

class AlBlendCurve;
class AlBlendPoint;

class AlCategory;

class STUDIOAPI_DECL AlObject : public AlHashable {
	friend class					AlFriend;
	friend class					AlDebugLib;

public:
	virtual 						~AlObject();

	virtual AlObjectType			type() const = 0;

	virtual const char*				name() const;
	virtual statusCode				setName( const char* );
	virtual statusCode				deleteObject();
	virtual AlObject*				copyWrapper() const;

	// Down casting methods

	AlAnimatable*			asAnimatablePtr();
	AlClusterable*			asClusterablePtr();
	AlSettable*				asSettablePtr();
	AlPickable*				asPickablePtr();

	AlCamera*				asCameraPtr();
	AlCameraNode*			asCameraNodePtr();

	AlCluster*				asClusterPtr();
	AlClusterNode*			asClusterNodePtr();
	AlClusterMember*		asClusterMemberPtr();

	AlCurveCV*				asCurveCVPtr();
	AlCurve*				asCurvePtr();
	AlCurveNode*			asCurveNodePtr();
	AlCurveOnSurface*		asCurveOnSurfacePtr();

	AlDagNode*				asDagNodePtr();

	AlFace*					asFacePtr();
	AlFaceNode*				asFaceNodePtr();

	AlGroupNode*			asGroupNodePtr();

	AlLight*				asLightPtr();
	AlLightNode*			asLightNodePtr();
	AlAmbientLight*			asAmbientLightPtr();
	AlAreaLight*			asAreaLightPtr();
	AlDirectionLight*		asDirectionLightPtr();
	AlLinearLight*			asLinearLightPtr();
	AlNonAmbientLight*		asNonAmbientLightPtr();
	AlPointLight*			asPointLightPtr();
	AlSpotLight*			asSpotLightPtr();
	AlVolumeLight*			asVolumeLightPtr();
	AlSphereLight*			asSphereLightPtr();
	AlCylinderLight*		asCylinderLightPtr();
	AlBoxLight*				asBoxLightPtr();
	AlConeLight*			asConeLightPtr();
	AlTorusLight*			asTorusLightPtr();

	AlSurfaceCV*			asSurfaceCVPtr();
	AlSurface*				asSurfacePtr();
	AlSurfaceNode*			asSurfaceNodePtr();
   AlSurfaceCurve*		asSurfaceCurvePtr();

	AlSet*					asSetPtr();
	AlSetMember*			asSetMemberPtr();

	AlShader*				asShaderPtr();
	AlSwitchShader*			asSwitchShaderPtr();
    AlLayeredShader*        asLayeredShaderPtr();
	AlTexture*				asTexturePtr();
	AlEnvironment*			asEnvironmentPtr();

	AlKeyframe*				asKeyframePtr();
	AlChannel*				asChannelPtr();
	AlAction*				asActionPtr();
	AlParamAction*			asParamActionPtr();
	AlMotionAction*			asMotionActionPtr();

	AlPolysetVertex*		asPolysetVertexPtr();
	AlPolysetNode*			asPolysetNodePtr();
	AlPolygon*				asPolygonPtr();
	AlPolyset*				asPolysetPtr();

    AlMeshNode*             asMeshNodePtr();
    AlMesh*                 asMeshPtr();

	AlAttributes*			asAttributesPtr();
	AlArcAttributes*		asArcAttributesPtr();
	AlLineAttributes*		asLineAttributesPtr();
	AlCurveAttributes*		asCurveAttributesPtr();
	AlPlaneAttributes*		asPlaneAttributesPtr();
	AlConicAttributes*		asConicAttributesPtr();
	AlRevSurfAttributes*	asRevSurfAttributesPtr();

	AlJoint*				asJointPtr();
	AlConstraint*			asConstraintPtr();
	AlPointConstraint*		asPointConstraintPtr();
	AlOrientationConstraint*asOrientationConstraintPtr();
	AlAimConstraint*		asAimConstraintPtr();

	AlTextureNode*			asTextureNodePtr();

	AlShellNode*			asShellNodePtr();
	AlShell*				asShellPtr();

	AlTrimRegion*			asTrimRegionPtr();
	AlTrimBoundary*			asTrimBoundaryPtr();
	AlTrimCurve*			asTrimCurvePtr();

	AlContact*				asContactPtr();
	AlCommandRef*			asCommandRefPtr();
	AlCommand*				asCommandPtr();
	AlLayer*				asLayerPtr();

	AlOrthographicCamera*	asOrthographicCameraPtr();
	AlPerspectiveCamera* 	asPerspectiveCameraPtr();
	AlWindow*				asWindowPtr();
	AlImagePlane*			asImagePlanePtr();

	AlIKHandle*				asIKHandlePtr();
	AlIKHandleNode*			asIKHandleNodePtr();

	AlLocator*				asLocatorPtr();
	AlAnnotationLocator*    asAnnotationLocatorPtr();
	AlDistanceLocator*      asDistanceLocatorPtr();
	AlAngleLocator*         asAngleLocatorPtr();
	AlRadialLocator*        asRadialLocatorPtr();
	AlDeviationLocator*     asDeviationLocatorPtr();
	AlMinmaxLocator*        asMinmaxLocatorPtr();

	AlConstructionEntity*	asConstructionEntityPtr();
	AlConstructionVector*	asConstructionVectorPtr();
	AlConstructionPlane*	asConstructionPlanePtr();
	AlPoint*				asPointPtr();
	AlSpacePoint*			asSpacePointPtr();
	AlCurvePoint*			asCurvePointPtr();
	AlCurveOnSurfacePoint*	asCurveOnSurfacePointPtr();
	AlSurfacePoint*			asSurfacePointPtr();

	AlCloud*				asCloudPtr();

	AlBlendCurve*			asBlendCurvePtr();
	AlBlendPoint*			asBlendPointPtr();
	AlCategory*	  		    asCategoryPtr();

protected:
	AlObject();
	AlObject( const AlHashKey& );

private:
	// Disallow the copy constructor and copy assignment
	AlObject (const AlObject &);
	AlObject &operator= (const AlObject &);
};

extern "C" STUDIOAPI_DECL API_BOOLEAN AlIsValid( const AlObject* );
extern "C" STUDIOAPI_DECL API_BOOLEAN AlAreEqual( const AlObject*, const AlObject* );

#endif // _AlObject

