/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
/*
//
//	.NAME AlMessage - Encapsulates the handling of object messages
//
//	.SECTION Description
//		This file contains the definitions required to handle object messages
//		for Alias objects.
//		A message handler is a user supplied function that is automatically
//		called when an event occurs.
//
//		A message handler is installed using the AlMessage::addMessageHandler()
//		function.  Below is a sample message handler to handle the deleting of
//		DagNodes:
//
//	.nf
// %@ void myDagNodeDeleteHandler( int msgtype, AlDagNode *dagnode )
// %@ {
// %@%@ do_something_before_its_deleted( dagnode );
// %@ }
//	.fi
//
//		To install the handler into the universe, the following function call
//		would be made in the application's initialization code:
//
//		AlMessage::addMessageHandler( kMessageDagNodeDeleted, myDagNodeDeleteHandler );
//	.br
//		If either Alias or the application deletes a dag node by using
//		AlDagNode.deleteObject(), the message handler will be called with a
//		pointer to the dagnode before the dagnode is removed.
//
//
//		In addition to being able receive standard messages from inside
//		Alias, it is possible to define custom message types which extend
//		the default set.  Of course, Alias has no prior knowledge of these
//		new types, so hooks cannot be added into existing code to send new
//		messages, but plugins can define messages which can then be received
//		by either the same plugin, or different plugins altogether.  This
//		allows a collaborative paradigm between plugins, such as sharing of
//		code or data.
//
//		To create a custom message type, call AlMessage::addMessageType.
//		You must supply a name for this type, which is then used by others
//		to find the type.  You are returned an AlMessageTypeHandle, a receipt
//		which the creator of the message type can then use to define
//		specific properties of the message type.  In particular, the owner
//		can lock a message type against the addition of handlers, lock the
//		type against the sending of messages, and set prologue and epilogue
//		functions which get called automatically before and after the sending
//		of a message.  See the documentation for the individual member
//		functions for more details.
//
//		Note that while a method is supplied to send a message, this method
//		applies only to user-defined messages.  Predefined Alias messages
//		cannot be sent.
//
*/

#ifndef _AlMessage
#define _AlMessage

#include <AlStyle.h>

typedef enum {
	kMessageFirstMessage = 0,
	kMessageInvalid = 0,

	kMessageDagNodeDeleted,
	kMessageDagNodeInstanced,
	kMessageDeleteAll,
	kMessageDagNodeModified,
	kMessageDagNodeModifiedGeometry,
	kMessageDagNodeModifiedShaderAssignment,
	kMessageDagNodeModifiedConstraint,
	kMessageDagNodePreReplaceGeometry,
	kMessageDagNodeReplaceGeometry,
	kMessageDagNodeApplyTransformation,
	kMessageDagNodeVisible,
	kMessageDagDispModified,
	kMessageCosDeleted,
	kMessageCosModified,
	kMessageCosVisible,
	kMessageAttributesDelete,
	kMessagePreUpdate,
	kMessageUpdate,
	kMessagePostUpdate,
	kMessageAnimPlayback,
	kMessageListModifiedNodes,
	kMessagePreRefresh,
	kMessageRefresh,
	kMessageCommandInstall,
	kMessageCommandFree,
	kMessagePickListModified,
	kMessageTrimSurface,
	kMessageUntrimSurface,
	kMessagePlotRefresh,
	kMessageUniverseCreated,	/* Unsupported */
	kMessageStageCreated = kMessageUniverseCreated,
	kMessageUniverseDeleted, 	/* Unsupported */
	kMessageStageDeleted = kMessageUniverseDeleted,
	kMessageUniverseMerged, 	/* Unsupported */
	kMessageStageMerged = kMessageUniverseMerged,
	kMessageQuit,
	kMessagePreRetrieve,
	kMessagePostRetrieve,
	kMessagePreStore,
	kMessagePostStore,
	kMessageDagInserted,
	kMessageDagNameModified,
	kMessageUniverseActive,		/* Unsupported */
	kMessageStageActive = kMessageUniverseActive,
	kMessageClMemberModified,
	kMessageExprModified,
	kMessageCommandUnInstall,
	kMessageJointModified,
	kMessageHierarchyModified,
	kMessageCnetDeleted,
	kMessageCloudDeleted,
	kMessageShaderModified,
	kMessageLightModified,
	kMessageCameraModified,
	kMessageTextureModified ,
	kMessageTextureAdded ,
	kMessageTextureDeleted ,
	kMessageShaderAdded ,
	kMessageShaderDeleted ,
	kMessageReferenceFileDeleted,
	kMessageReferenceFileSetDeleted,
	kMessageLayerAssign,
	kMessageLayerSymmetryModified,
    kMessageDagNodeUndeleted,
    kMessageLayerStateModified,
    kMessageLayerVisibilityModified,
    kMessageLayersEnabled,
    kMessageLayersReordered,
    kMessageLayerAdded,
    kMessageLayerDeleted,
    kMessageLayerAttributeChanged,
    kMessageReferenceFileModified,
    kMessageReferenceFileAdded,
    kMessageReferenceFileSetModified,
    kMessageReferenceFileSetAdded,
    kMessageReferenceLayerModified,
    kMessageCosCreated,
    kMessageLocatorAdded,
    kMessageLocatorDeleted,
    kMessageLocatorModified,
    kMessageDagNodeColourModified,

	kMessageLastMessage /* this is not a valid message, just an ending marker.*/
} AlMessageType;

class AlDagNode;
class AlTM;
class AlCurveOnSurface;
class AlAttributes;
class AlSurface;
class AlFriend;
class AlNotifyDagNode;
class AlObject;
class AlCloud;
class AlShader;
class AlTexture;
class AlLight;
class AlCamera;
class AlReferenceFile;
class AlReferenceFileSet;
class AlReferenceLayer;
class AlLocator;

/*
//
//	Prototypes for the call back functions.  See AlMessage::addMessageHandler
//	for a summary of which prototypes belong to which messages.
//
*/
typedef void (AlCallbackVoid)               ( AlMessageType );
typedef void (AlCallbackInt)                ( AlMessageType, int );
typedef void (AlCallback2Ints)              ( AlMessageType, int, int );
typedef void (AlCallbackStringInt)	        ( AlMessageType, const char*, int );
typedef void (AlCallbackString)		        ( AlMessageType, const char* );
typedef void (AlCallbackCurveOnSurface)     ( AlMessageType, AlCurveOnSurface* );
typedef void (AlCallbackAttributes)         ( AlMessageType, AlAttributes* );
typedef int  (AlCallbackUpdate)             ( AlMessageType, int );
typedef void (AlCallbackSurface)            ( AlMessageType, AlSurface* );
typedef void (AlCallbackOneDagNode)         ( AlMessageType, AlDagNode* );
typedef void (AlCallbackTwoDagNodes)        ( AlMessageType, AlDagNode*, AlDagNode*);
typedef void (AlCallbackDagNodeAndMatrix)   ( AlMessageType, AlDagNode*, AlTM* );
typedef void (AlCallbackOneCloud)           ( AlMessageType, AlCloud* );
typedef void (AlCallbackOneShader)          ( AlMessageType, AlShader* );
typedef void (AlCallbackOneTexture)         ( AlMessageType, AlTexture* );
typedef void (AlCallbackOneLight)           ( AlMessageType, AlLight* );
typedef void (AlCallbackOneCamera)          ( AlMessageType, AlCamera* );
typedef void (AlCallbackOneRefFile)         ( AlMessageType, AlReferenceFile* );
typedef void (AlCallbackOneRefFileMod)      ( AlMessageType, AlReferenceFile*, int );
typedef void (AlCallbackOneRefFileSet)      ( AlMessageType, AlReferenceFileSet* );
typedef void (AlCallbackOneRefFileSetMod)   ( AlMessageType, AlReferenceFileSet*, int );
typedef void (AlCallbackOneRefLayer)        ( AlMessageType, AlReferenceLayer* );
typedef void (AlCallbackOneRefLayerMod)     ( AlMessageType, AlReferenceLayer*, int );
typedef void (AlCallbackLocator)            ( AlMessageType, AlLocator* );

typedef API_BOOLEAN (AlCallbackListModifiedNodes)( AlMessageType, const AlNotifyDagNode*, AlObject *);

class STUDIOAPI_DECL AlMessageTypeHandle
{
friend						class AlMessage;
public:
							AlMessageTypeHandle( void );
							AlMessageTypeHandle( const AlMessageTypeHandle& );
							~AlMessageTypeHandle();

	AlMessageTypeHandle&	operator =( const AlMessageTypeHandle& );

	API_BOOLEAN					isValid( void ) const;
	int						type( void ) const;

	statusCode				setPrologue( int (*)( int, void * ) );
	statusCode				setEpilogue( int (*)( int, void * ) );

	statusCode				addLock( API_BOOLEAN& );
	statusCode				setAddLock( API_BOOLEAN );

	statusCode				sendLock( API_BOOLEAN& );
	statusCode				setSendLock( API_BOOLEAN );

private:
							AlMessageTypeHandle( void* );

	void 					*opaque_pointer;
};

class STUDIOAPI_DECL AlMessage
{
	friend class AlFriend;
public:

	enum AlPriorityType {
		kImmediate,
		kQueue,
		kIdleQueue
	};

public:
    AlMessage() {}  // because of the private constructor
	static statusCode 	addMessageHandler( int, void * );
	static statusCode 	removeMessageHandler( int, void * );

	static AlMessageTypeHandle 	addMessageType( const char * );
	static int					getMessageType( const char * );
	static statusCode			sendMessage( int, void*, AlPriorityType = kImmediate );

protected:
	static void*		setPostFunction( AlMessageType, void* );
	static void			startMessageHandlers( void );

private:
	// Disallow the copy constructor and copy assignment
	AlMessage (const AlMessage &);
	AlMessage &operator= (const AlMessage &);
};

#endif	/*  _AlMessage */

