/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//    .NAME AlMesh - Interface to Alias meshes.
//
//    .SECTION Description
//
//        AlMesh is the interface to meshes and their data in Alias.
//        This class lets a developer create and manipulate mesh
//        objects. Data access methods to all properties are provided. 
//        The following properties are used throughout the mesh class. 
//
//  .nf
//      vertices -     Vertices are passed by this class using
//                     an array of floating point values. A
//                     vertex consists of three values; x,y,z
//                     respectively. Refer to the definition of
//                     triangles for a description of the
//                     relationship between vertices and triangles.
//      numVertices -  The number of vertices is equivalent
//                     to the number of x,y,z triplets stored
//                     in the vertices array. The total number
//                     of floats in the vertices array is
//                     equivalent to three times the number of
//                     vertices.
//
//      triangles -    Triangles are passed by this class using
//                     an array of integer values. A triangle
//                     consists of three integer indices. These
//                     indices reference a vertex in the vertices
//                     array such that vertices[3*index] returns
//                     the x coordinate of the vertex. Similarly,
//                     vertices[3*index+1] and vertices[3*index+2]
//                     return the y and z coordinate respectively.
//                     Indices are to be specified in counter
//                     clockwise order relative to the front of
//                     their face.
//      numTriangles - The number of triangles is equivalent to
//                     the number of integer triplets stored in the
//                     triangles array. The total number of integers
//                     in the triangles array is equivalent to three
//                     times the number of triangles.
//
//      normals -      Normals are passed by this class using
//                     an array of floating point values. They
//                     consist of three values; x,y,z respectively.
//                     Normals are per vertex. They are referenced
//                     by triangles in the same manner that vertices
//                     are referenced.
//
//      uvs -          UVs are passed by this class using an array
//                     of float values. They consist of two floating
//                     point values; u,v respectively. UVs are per
//                     vertex. They are referenced by triangles in
//                     the same manner that vertices are referenced.
//  .fi
//
#ifndef ALMESH_H
#define ALMESH_H

#include <AlObject.h>

class AlMeshNode;
struct AlRenderInfo;
class AlSwitchShader;
class AlLayeredShader;

struct Dag_node;

class STUDIOAPI_DECL AlMesh : public AlObject
{
    friend class            AlFriend;
public:
                            AlMesh();
    virtual                 ~AlMesh();
    virtual statusCode      deleteObject();
    virtual AlObject*       copyWrapper() const;

    virtual AlObjectType    type() const;
    AlMeshNode*             meshNode() const;

    statusCode              create( int numVertices, const float* vertices,
                                    int numTriangles, const int* triangles,
                                    const float* normals = 0,
                                    const float* uvs = 0);

    int                     numberOfVertices() const;
    int                     numberOfTriangles() const;

    const float*            vertices() const;
    const int*              triangles() const;
    const float*            normals() const;

    const float*            uvs() const;
    void                    uvRange( float* ) const;
    statusCode              setUVRange( float * );

    int                     drawFlag() const;
    API_BOOLEAN             isDrawFlag( int ) const;
    statusCode              setDrawFlag( int, bool );
    statusCode              setDrawFlag( int );

    API_BOOLEAN             isFlatShaded() const;
    statusCode              setFlatShaded( bool );

    float*                  affectedVertices( const AlTM& tm );
    float*                  affectedNormals( const AlTM& tm );

    statusCode              removeDuplicateTris();
    statusCode              compact();
	statusCode              mergeVertices(float tol);

    statusCode              transform( double [4][4] ) const;
    statusCode              boundingBox( double *min, double *max ) const;

    AlShader*               firstShader() const;
    AlShader*               nextShader( AlShader* ) const;
	AlSwitchShader*			getSwitchShader() const;
    AlLayeredShader*        getLayeredShader() const;
    statusCode              nextShaderD( AlShader *) const;
    statusCode              assignShader( AlShader* );
	statusCode              assignSwitchShader(AlSwitchShader*);
    statusCode              assignLayeredShader(AlLayeredShader*);
    statusCode              layerShader( AlShader* );

    statusCode              renderInfo( AlRenderInfo& ) const;
    statusCode              setRenderInfo( const AlRenderInfo& ) const;
    
protected:
                            AlMesh( Dag_node* );
                            
    Dag_node                *fParent;

private:    
    static void             initMessages();                            
    static void             finiMessages();

    virtual API_BOOLEAN     extractType( int&, void*&, void*& ) const;
};

#endif
