/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlLayer - Interface to Alias layer objects.
//
//	.SECTION Description
//		Layers provide a way of organizing the models to improve the workflow.
//		The layers have attributes such as visibility, pickability etc, which
//		apply to all the dag nodes which refer to them. The AlDagNode has 
//		methods to get AlLayer and set to a new AlLayer.
//
//		Each AlLayer has a unique identifying number and a name. By default, 
//		the layer gets a unique name based on its number, however, the user 
//		is free to change it any name. The user given layer names do not have 
//		to be unique.
//
//		Default Layer:
//			There is always a default layer in the AlUniverse with a number 0. 
//			The attributes of this default layer can never be changed. Any object 
//			of a class derived from AlDagNode class can be assigned to an AlLayer.
//		
//		Creation Layer:
//			Any new AlDagNode is automatically assigned to the layer 
//			which is made a creation layer. Any existing layer can be made a creation layer, 
//			including the default layer, to put the new nodes in that layer 
//			automatically. When a layer is made a creation layer, its attributes such 
//			as invisibility, temporarily become ineffective and it gets the 
//			same attributes as the default layer. These attributes become effective
//			again, as soon as some another layer is made a creation layer.
//
//		An AlDagNode can be transferred from one layer to another. An AlDagNode
//		can exist on only one layer at a time. When an AlDagNode is assigned to a 
//		non-default layer, all its children are automatically assigned to that layer 
//		and all the parents are automatically assigned to the default layer. In addition, 
//		all the instances of the AlDagNode are also assigned to the same layer. However, 
//		the siblings of that AlDagNode remain uneffected.
//		
//		All the attributes in an AlLayer can be obtained and set with the 
//		methods in this class. By default, all the layers have a corresponding menu 
//		in the Alias interface and are visible in the layer bar in the Alias interface. 
//		However, through visibleInLayerBar() methods, a partial list of layers with 
//		which one chooses to work can be made visible. The change in visibility in the layer 
//		bar does not change the other attributes of the Allayer.
//		
//		All the existing layers in an AlUniverse can be accessed through methods in
//		AlUniverse. The layer functionality can be temporarily disabled through the static	
//		method setLayersEnabled() in the AlUniverse class. When the layer functionality 
//		is disabled or turned off, all the layers attributes temporarily become similar 
//		to the default layer. However, the layer references of the AlDagNode are not changed.
//
//		Symmetric layers are also supported in the API.  Methods available allow the developer
//		to turn symmetric layers on, find out if a layer is symmetric, set/query the
//		origin and normal parameters of the symmetric plane and create the symmetric
//		geometry.  Symmetric layers are specified by a normal and a point(origin).
//		Note: creation of symmetric geometry is performed by the AlDagNode class.
//
//		The complete layers interface is available in Open Model as well.

#ifndef _AlLayer
#define _AlLayer

#include <AlObject.h>


class AlFriend;
class AlDagNode;

class STUDIOAPI_DECL AlLayer : public AlObject
{
	friend					class AlFriend;
public:

	enum AlLayerPickType {
		kLayerPickable,
		kLayerReference, 
		kLayerInactive
	};

	virtual					~AlLayer();	
							AlLayer();	
	
	virtual AlObject*		copyWrapper() const;
	virtual statusCode		deleteObject();
	virtual AlObjectType	type() const;

    API_BOOLEAN             isFolder() const;
    AlLayer*				childLayer() const;
  	AlLayer*	            parentLayer() const;
	AlLayer*		        nextLayer() const;
	AlLayer*		        prevLayer() const;

	statusCode				create(const char * = 0);
	statusCode				create(int&, const char *);
	statusCode				create(int);

    statusCode				createFolder(const char * = 0);
    statusCode				createFolder(int&, const char * = 0);
    
    statusCode				assignChild(int);
    statusCode				assignParent(int);
    statusCode				assignSiblingOnRight(int);
    statusCode				assignSiblingOnLeft(int);

	API_BOOLEAN			    invisible() const;
	statusCode				setInvisible( API_BOOLEAN );

	int						number() const;

	const char *			name() const;
	statusCode				setName( const char * );

	statusCode				pickability( AlLayerPickType& ) const;
	statusCode				setPickability( AlLayerPickType );

	API_BOOLEAN				playback() const;
	statusCode				setPlayback( API_BOOLEAN );
	
	API_BOOLEAN				drawInstances() const;
	statusCode				setDrawInstances( API_BOOLEAN );

	API_BOOLEAN				visibleInLayerBar() const;
	statusCode				setVisibleInLayerBar( API_BOOLEAN );

	int						color() const;		// obsolete
	statusCode				setColor( int );	// obsolete

	statusCode				setCustomColor( unsigned char,
									unsigned char, unsigned char, unsigned char );
	statusCode				customColor( unsigned char &,
									unsigned char &, unsigned char &, unsigned char & ) const;

	API_BOOLEAN				isPicked() const; 
	statusCode				pick();
	statusCode				unpick();
	
	static int				startNumber();
	static statusCode		setStartNumber( int );
	
	// Symmetric layer support
	statusCode				setSymmetric( API_BOOLEAN );
	API_BOOLEAN				isSymmetric() const;
	
	statusCode				setSymmetricOrigin( double, double, double );
	statusCode				setSymmetricNormal( double, double, double );
	statusCode				symmetricOrigin( double&, double&, double& );
	statusCode				symmetricNormal( double&, double&, double& );
    statusCode              pickNodesOnLayer();
		
protected:
	
private:
	static void				initMessages();
	static void				finiMessages();
};

#endif // _AlLayer









