/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlKeyframe - Basic interface to Alias keyframes on parameter
//						curve actions.
//
//	.SECTION Description
//		AlKeyframe represents a keyframe belonging to an AlParamAction.
//	The keyframe class does not have a create method.  New keyframes aer
//	created using the AlParamAction::addKeyframe() methods.
//	(this is faster than creating the keyframes, then adding them to the
//	paramaction).
//
//		A keyframe cannot belong to more than one action.  Thus attempting
//	to add a keyframe that has already been added to an AlParamAction
//	will fail.
//
//		If an AlKeyframe is deleted and if the AlKeyframe belongs to an
//	AlParamAction and it is the last keyframe of that parameter curve
//	action, then the AlParamAction will also be deleted.  This ensures that
//	no AlParamAction parameter curves will exist with no keyframes.
//
//		If a keyframe is added to an AlParamAction, and another keyframe
//	already exists in that AlParamAction with the same location as the
//	keyframe to be added, then the existing keyframe in the AlParamAction
//	will be deleted.
//
//		The method that applies tangent types to a keyframe (i.e.
//	setTangentTypes()) does not make much sense if the keyframe is not part
//	of an AlParamAction.  It is best to add all the desired keyframes
//	to an AlParamAction, and then walk the list of keyframes, and calling
//	the tangent methods.
//
//		You can lock a keyframe by calling the method AlKeyframe::setLock(TRUE).
//	If an AlKeyframe is locked, then none of the AlKeyframe
//	methods that change its location, value or tangents will succeed.
//	The keyframe is non-modifiable.  You can unlock the keyframe again by
//	calling AlKeyframe::setLock(FALSE).
//

#ifndef _AlKeyframe
#define _AlKeyframe

typedef enum AlTangentType {
	kTangentSmooth, kTangentLinear, kTangentFlat, kTangentStep,
	kTangentSlow, kTangentInOut, kTangentFast, kTangentFixed, kTangentUnchanged
} AlTangentType;

#include <AlObject.h>

class STUDIOAPI_DECL AlKeyframe : public AlObject 
{
	friend				class AlFriend;
public:
						AlKeyframe();
	virtual				~AlKeyframe();
	virtual statusCode	deleteObject();
	virtual AlObject*	copyWrapper() const;

	virtual AlObjectType	type() const;

	API_BOOLEAN				isLocked() const;
	statusCode			setLock(API_BOOLEAN);

	AlKeyframe*			prev() const;
	AlKeyframe*			next() const;

	statusCode			prevD();
	statusCode			nextD();

	double				value() const;
	double				location() const;
	double				inTangent() const;
	double				outTangent() const;
	statusCode			tangentTypes( AlTangentType&, AlTangentType& ) const;

	statusCode			setValue(double, API_BOOLEAN = TRUE );
	statusCode			setLocation(double, API_BOOLEAN = TRUE );
	statusCode			setInTangent(double);
	statusCode			setOutTangent(double);
	statusCode			setTangentTypes(AlTangentType, AlTangentType, API_BOOLEAN = TRUE );

private:
	static void			initMessages();
	static void 		finiMessages();
};

#endif	/* _AlKeyframe */
