/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlIterator - A base class used to derive iterators for performing tasks on elements of a list.
//
//	.SECTION Description
//		Many classes return the first element of a list, which is then
//		traversed with some operation performed on each element of the
//		list. This class encapsulates this functionality making it very
//		easy to write code which performs operations on members of a list.
//
//		To use this class the method "func" should be overloaded. On
//		success func() should return zero which will cause the iteration
//		to continue with the next element in the list. A non-zero return
//		value will cause the iteration to stop. The returned value will
//		be returned through the second reference argument in the
//		applyIterator() method. In general the applyIterator() methods
//		return sSuccess even when func() returns non-zero. A return
//		other than sSuccess indicates that the applyIterator() method
//		failed to function properly.
//
//		Two types of iterators are defined.  The first passes in an
//		AlObject* to the func().  The second iterator AlIteratorWithParent
//		passes in the AlObject* and the AlDagNode* parent of the object.
//		Note that if an AlIteratorWithParent is used and the AlObject
//		is a shader then the AlDagNode pointer will be null.
//
//		Iterators should be used to examine or set data in the visited
//		objects, but should not be used to delete the objects.
//
//		For example:
//
//	.nf
// %@ class countIterator : public AlIterator {
// %@%@ public:
// %@%@%@ countIterator() : count( 0 );
// %@%@%@ ~countIterator() {};
// %@%@%@ virtual int func( AlObject* ) { count++; };
// %@%@%@ int result() { return count; };
// %@%@ private:
// %@%@%@ int count;
// %@ };
//
// code which creates a complete polyset ...
//
// %@ countIterator* iter = new countIterator;
// %@ status = pset->applyIteratorToVertices( iter, returnCode );
// %@ numVertices = iter->result();
//	.fi
//

#ifndef _AlIterator
#define _AlIterator

#include <AlStyle.h>

class AlObject;
class AlDagNode;

class STUDIOAPI_DECL AlIterator 
{
public:
	AlIterator();
	virtual int func( AlObject* ) = 0;

private:
	// Disallow the copy constructor and copy assignment
	AlIterator (const AlIterator &);
	AlIterator &operator= (const AlIterator &);
};

class STUDIOAPI_DECL AlIteratorWithParent 
{
public:
	AlIteratorWithParent();
	virtual int func( AlObject* , AlDagNode* ) = 0;

private:
	// Disallow the copy constructor and copy assignment
	AlIteratorWithParent (const AlIteratorWithParent &);
	AlIteratorWithParent &operator= (const AlIteratorWithParent &);
};

#endif
