/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
//
//	.NAME AlImagePlane - Interface to Alias image planes.
//
//	.SECTION Description
//		Image planes are full colour images that are attached to a camera.
//		If the view from that camera is rendered, they get rendered into the
//		background of the resulting image.  This class deals with setting the
//		various parameters of image planes and their associations to cameras.
//
//		To create an image plane, you must use the addImagePlane function
//		of AlCamera.  That function will return to you a new AlImagePlane
//		object that you can use to set the image plane's parameters.  Given
//		a pointer to an AlImagePlane, you can delete it from Alias using the
//		deleteObject method.
//
//		To walk the list of all the image planes in Alias, use
//		AlUniverse::firstImagePlane.  You can move from image plane to image
//		plane in the current stage, or in a camera, by using the appropriate
//		next method.
//
//		For more information about image planes in general, see the
//		online documentation for the camera editor.
//
*/

#ifndef _AlImagePlane
#define _AlImagePlane

#include <AlObject.h>
#include <AlAnimatable.h>
#include <AlPickable.h>

class STUDIOAPI_DECL AlImagePlane 	: public AlObject , public AlAnimatable , public AlPickable
{
	friend					class AlFriend;

public:
	enum AlDisplayMode {
		kDisplayOff,
		kDisplayUnknown,
		kRGB,
		kOutline,
		kFilled
	};

	enum AlFrameType {
		kFrameOff,
		kRegular,
		kKeyframe
	};

	union AlFrameTypeParams {
		struct {
			int start;
			int end;
			int by;
			int offset;
		} regular;
		struct {
			int frame;
			double offset;
		} keyframe;
	};

	enum AlDepthType {
		kDepthOff,
		kDepthUnknown,
		kPriority
	};

	enum AlMaskType {
		kMaskOff,
		kMaskUnknown,
		kImage,
		kFile,
		kChromaKey
	};

	union AlMaskTypeParams {
		struct {
			const char 			*name;
			AlFrameType 		frameType;
			AlFrameTypeParams	*params;
		} file;
		struct {
			double 			r;
			double 			g;
			double 			b;
			double 			hue_range;
			double 			sat_range;
			double 			val_range;
			double 			threshold;
		} chroma_key;
	};

	enum AlImageSpaceType {
		kScreen,
		kFields,
		k01,
		kWorld
	};

	union AlImageSpaceTypeParams {
		struct {
			int				size;
		} fields;
		struct {
			int 			origin_x;
			int				origin_y;
			int				size_x;
			int				size_y;
		} screen;
		struct {
			double			tran_x;
			double			tran_y;
			double			size_x;
			double 			size_y;
		} zero_one;
		struct {
			double			tran_x;
			double			tran_y;
			double			size_x;
			double			size_y;
			double			pivot_x;
			double			pivot_y;
		} world;
	};

	struct AlImageSpaceParams {
		API_BOOLEAN				wrap_horiz;
		API_BOOLEAN				wrap_vert;
		int 				offset_x;
		int					offset_y;
		int					coverage_x;
		int					coverage_y;
	};

public:

							AlImagePlane();
							~AlImagePlane();

	virtual AlObject*		copyWrapper() const;
	virtual statusCode		deleteObject();

	virtual AlObjectType	type() const;

	virtual const char*		name() const;
	virtual statusCode		setName( const char * );

	AlImagePlane*			next() const;
	AlImagePlane*			nextInCamera() const;
	AlImagePlane*			prev() const;
	AlImagePlane*			prevInCamera() const;

	statusCode				nextD();
	statusCode				nextInCameraD();
	statusCode				prevD();
	statusCode				prevInCameraD();

	const char*				imageFile();
	statusCode				setImageFile( const char * );

	statusCode				displayMode( AlDisplayMode & );
	statusCode				setDisplayMode( AlDisplayMode );

	statusCode				RGBMult( double&, double&, double&, double& );
	statusCode				setRGBMult( double, double, double, double );

	statusCode				RGBOffset( double&, double&, double&, double& );
	statusCode				setRGBOffset( double, double, double, double );

	statusCode				frameType( AlFrameType&, AlFrameTypeParams* = NULL );
	statusCode				setFrameType( AlFrameType, AlFrameTypeParams* = NULL );

	statusCode				imageSize( int&, int& );
	statusCode				imageData( void*& );

	// The next four functions are not fully supported.
	statusCode				maskType( AlMaskType&, AlMaskTypeParams* = NULL );
	statusCode				setMaskType( AlMaskType, AlMaskTypeParams* = NULL );
	statusCode				maskInvert( API_BOOLEAN& );
	statusCode				setMaskInvert( API_BOOLEAN );

	statusCode				depthType( AlDepthType&, double* = NULL );
	statusCode				setDepthType( AlDepthType, double* = NULL );

	// The next two functions are not fully supported.
	statusCode				imageSpace( AlImageSpaceType, AlImageSpaceTypeParams* = NULL, AlImageSpaceParams* = NULL );
	statusCode				setImageSpace( AlImageSpaceType, AlImageSpaceTypeParams* = NULL, AlImageSpaceParams* = NULL );

	//
	//	use kScreen X/Y origin to set the position of the image in pixels
	//	Use these to set the pivot (in pixels)
	//
	statusCode				pivot( double&x, double &y);
	statusCode				setPivot( double x, double y);


private:
	virtual	API_BOOLEAN extractType( int&, void*&, void*& ) const;

	static void				initMessages( void );
	static void				finiMessages( void );
};

#endif /* _AlImagePlane */
