/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
#ifndef _AlIKHandle
#define _AlIKHandle

//
//	.NAME AlIKHandle - Interface to Inverse Kinematics Handles.
//
//	.SECTION Description
//
//		A skeleton is just a collection of joint dag nodes that have no
//		particular special behaviour until IK handles are applied to them.
//		In order to use inverse kinematics in Alias, you have to create
//		IK handles that define paths along skeletons to be constrained by IK.
//
//		An IK handle is defined by two joints. The end effector is
//		the point on the skeleton that is free to move, and the root
//		is the anchor point on the skeleton.  When the end effector is
//		moved, the rotations on all joints leading up to the root are
//		constrained to provide an appropriate IK solution.
//
//		The IK handle also specifies the solver that will be used.
//		There are two kinds of IK solvers in Alias.  The single-chain
//		solver is analytic and always produces a unique solution.  The
//		multi-chain solver is iterative, and the solution depends on
//		the starting state.
//
//		Furthermore, single-chain handles are never allowed to overlap other
//		IK handles, except for the root of one handle meeting the end
//		effector of another.  Multi-chain handles, on the other hand,
//		are allowed to interfere with one another, the result being a
//		best-possible IK solution.
//
//		Single-chain IK handles are always position handles - the rotations
//		of the joints above the end-effector, and below or at the root, are
//		transformed to meet the position of the end-effector.
//
//		Multi-solver IK handles can be position or orientation goals or both.
//		An orientation goal will try to match the orientation of the bone
//		immediately above the end-effector to the orientation of the IK handle.
//
//		Also, multi-solver IK handles have weights.  When several multi-solver
//		IK handles overlap, the weights on these handles are used to determine
//		the relative effect each solution has on the overall rotation of the
//		joints in the affected skeleton.
//

typedef enum {
	kSingleChain, kMultiSolver, kSplineSolver
} AlIKHandleSolverType;

typedef enum {
	kPositionGoal, kOrientationGoal, kBothGoal
} AlIKHandleGoalType;

typedef enum {
	kRotationOrder_Invalid,
	kRotationOrder_XYZ, kRotationOrder_YXZ, kRotationOrder_ZXY
} AlIKHandleRotationOrder;

typedef enum {
	kPositionType_Invalid,
	kParameter,	kArcLength
} AlIKHandlePositionType;

typedef enum {
	kTwistType_Invalid,
	kTwist_Linear, kTwist_FOSI, kTwist_SOFI, kTwist_EOEI
} AlIKHandleTwistType;

#include <AlObject.h>
#include <AlAnimatable.h>

struct IK_handle;
struct Dag_node;

class AlJoint;
class AlDagNode;
class AlCurveNode;
class AlIKHandleNode;

class STUDIOAPI_DECL AlIKHandle : public AlObject , public AlAnimatable
{
	friend class			AlFriend;

public:
							AlIKHandle( void );
	virtual					~AlIKHandle();
	virtual statusCode		deleteObject();
	virtual AlObject*		copyWrapper() const;

	virtual AlObjectType	type() const;

	statusCode				createSingle( AlJoint*, AlJoint* );
	statusCode				createMulti( AlJoint*, AlJoint*,
								AlIKHandleGoalType );
	statusCode				createSpline (AlJoint*, AlJoint*,
										  AlCurveNode*, AlIKHandle *&);
	statusCode				createSpline (AlJoint*, AlJoint*, AlCurveNode*);

	API_BOOLEAN					on( void ) const;

	AlIKHandleSolverType	solverType( void ) const;
	AlIKHandleGoalType		goalType( void ) const;
	double					weight( void ) const;
	AlIKHandleRotationOrder	rotationOrder( void ) const;
	API_BOOLEAN					worldOrientation( void ) const;
	statusCode				restRotation( double[3] ) const;
	AlJoint*				root( void ) const;
	AlJoint*				endEffector( void ) const;
	AlDagNode*				rootNode( void ) const;
	AlDagNode*				endEffectorNode( void ) const;
	AlIKHandlePositionType	positionType( void ) const;
	AlIKHandleTwistType		twistType( void ) const;
	AlCurveNode*			curveNode( void ) const;
	API_BOOLEAN					oneJointHandle( void ) const;
	API_BOOLEAN					touchRootHandle( void ) const;

	AlIKHandlePositionType	splineChainPosition( double* ) const;
	AlIKHandleTwistType		splineChainTwist( double* ) const;
	double					splineChainRoll( void ) const;

	statusCode				setOn( API_BOOLEAN );

	statusCode				setGoalType( AlIKHandleGoalType );
	statusCode				setWeight( double );
	statusCode				setRotationOrder( AlIKHandleRotationOrder );
	statusCode				setWorldOrientation( API_BOOLEAN );
	statusCode				setRestRotation( const double[3] );
	statusCode				assumeRestRotation( void );
	statusCode				setPositionType( AlIKHandlePositionType );
	statusCode				setTwistType ( AlIKHandleTwistType );

	AlIKHandleNode*			handleNode( void ) const;

private:
	virtual API_BOOLEAN 		extractType( int&, void*&, void*& ) const;
	static void       		initMessages();
	static void       		finiMessages();

	void					ik_handle_modified( void );
	API_BOOLEAN					createOneJointHandle(Dag_node*, Dag_node*);
};

#endif	/* _AlIKHandle */
