/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlGroupNode - A dag node which can contain a list of dag nodes.
//
//	.SECTION Description
//
//		This class encapsulates the functionality for creating,
//		manipulating and deleting a group node.  A group node is a dag
//		node which refers to a list of child dag nodes.  It is this type
//		of dag node which allows the hierarchical grouping of dag nodes.
//
//		The transformations which can be defined for a group node are
//		inherited by each child dag node.  This means that a
//		group node's transformations are combined with a child node's
//		transformations to define a global transformation for the object
//		that the child node refers to.
//
//		A group node's list of child dag nodes can be shared by more
//		than one group node.  If a group node's list of child nodes is
//		shared by another group node, both group nodes are considered
//		"instanced".  This can be achieved by using the createInstance()
//		method to create an instanced group node from another group
//		node.  The instanced group node is created as a sibling of the
//		group node.  There are methods for finding the next and previous
//		instanced group node among its siblings and for determining
//		whether a group node is an instanced node.
//
//		To create a group node, the user must call the constructor and
//		then the create method for an AlGroupNode object.  If a group
//		node is not an instanced group node, deleting it will cause the
//		deletion of all the child dag nodes and the deletion of any
//		objects the child dag nodes refer to.  Deleting an instanced
//		group node will not cause all of its child nodes to be deleted
//		since the list of child nodes is shared by another instanced
//		group node.
//
//		Note on AlGroupNode::deleteObject():
//
//		If a group node is an instanced group node, then only the group
//		node is removed from its list of siblings and is deleted.
//		The list of child dag nodes an instanced dag node refers to is
//		not deleted.  If this group node
//		is not an instanced group node (i.e. none of its siblings share
//		its list of child dag nodes), then the group node is removed
//		from list of siblings it belongs to and the group node and
//		every child node of the group node is deleted.
//

#ifndef _AlGroupNode
#define _AlGroupNode

#include <AlDagNode.h>

class AlIterator;

class STUDIOAPI_DECL AlGroupNode : public AlDagNode 
{
	friend class			AlFriend;
public:
							AlGroupNode();
	virtual					~AlGroupNode();
	virtual AlObject*		copyWrapper() const;
	statusCode				create();

	virtual AlObjectType	type() const;

	AlDagNode*				childNode() const;
	AlDagNode*				childNode(AlTM&) const;

	AlGroupNode*			nextInstance() const;
	AlGroupNode*			prevInstance() const;

	statusCode				nextInstanceD();
	statusCode				prevInstanceD();

	virtual API_BOOLEAN			isInstanceable();
	API_BOOLEAN					isInstanceNode();

	statusCode				addChildNode( AlDagNode* );

	statusCode				applyIteratorToChildren( AlIterator*, int& );

protected:
	statusCode				attachChild( AlDagNode* );
};

#endif // _AlGroupNode
