/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
//
//	.NAME AlFunction - Class for creating the OpenAlias interface to Alias
//
//	.SECTION Description
//		This class provides a means to interface the OpenAlias application
//		to the Alias user interface.
//	.br
//
//		The UI interacts with the plugin through the use of callback functions.
//		The plugin supplies pointers to functions that are called whenever
//		a menu button is pressed or the mouse moves.
//		There are two classes of UI functions (the set of callbacks associated
//		with a menu button): momentary functions and continuous functions.
//	.br
//
//		A true momentary function just executes an operation and then returns
//		to the previous function.  Momentary functions are used for 'one shot'
//		type operations.  The function does not require any user input, and
//		simply returns after it is done.  An example of a momentary functions
//		is Xform->undo.
//	.br
//
//		A continuous function is a set of 5 callback functions that correspond
//		to the stages involved in a mouse interaction.  They are the
//		'init' function, the mouse down function, the mouse move function,
//		the mouse up function and the cleanup function.
//	.br
//
//		The init function is called when you first switch to the UI function by
//		clicking on the menu item.  It does not take any arguments, and is
//		intended for setting up any data required for the operation (e.g.
//		reading option box values, making a list of picked items etc).
//
//	.br
//		The cleanup function is called when the user leaves this UI function
//		by selecting another menu item.  You will usually clean up the data
//		structures that are set up in the init function.
//	.br
//
//		The mouse down function is called when the user depresses the mouse
//		button.
//		The mouse move function is called each time the mouse is moved from
//		its current position.  Note that this function will be called
//		immediately after the init function.  A mouse 'session' will always
//		contain one call to the down function, one or more calls to the move
//		function followed by a call to the mouse up function.
//	.br
//
//		The mouse up function is called when the mouse button is released.
//		The mouse down/move/up functions are passed three arguments:  the
//		event information (a packed value containing information about the
//		mouse buttons, keyboard keys etc) and the x/y position of the mouse
//		when the event occurred (use AlContinousFunction::translateInput to
//		extract the values).
//		If a keyboard event is read, then use scanf() to process the values
//		read from the command line.  Otherwise, the mouse was clicked down
//		on the screen.  The button in question can be determined by examining
//		the Modifier masks on the value returned by translateInput
//
//	.br
//
//	.nf
//	Hybrid functions: Pseudo-momentary
//	.fi
//		By definition, a momentary just performs an action and then returns
//		to the previous function.  No user-input (mouse or keyboard) should be
//		processed in a momentary function.
//	.br
//
//		A continuous function performs the action, and that function stays
//		current until you select another function.  These types of  functions
//		 can have mouse or keyboard input.
//	.br
//
//		There are circumstances which require a momentary style function
//		but we want to also provide mouse or keyboard input.  For example
//		'set keyframe' requires keyboard input, so it should be continuous.
//		However, it can become cumbersome for the user to use.  Consider the
//		following sequence of events to set the keyframe while Xform->move is
//		selected ..
//
//	.nf
//		Xform->move is used to move a sphere
//		set keyframe is selected and becomes the 'current' function
//		a value is entered into set keyframe
//		the user must now click on Xform->move again to move the sphere.
//	.fi
//		This is the purpose behind the AlFunction::setBehaviour method.
//		It makes a continuous function behave as a  momentary function, but
//		also allows the users to provide us with keyboard or mouse input.
//		After the function has been executed, it switches back to the previous
//		function.
//	.br
//		Instead of declaring the 'set keyframe' function using ..
//	.br
// %@ AlMomentaryFunction::create( do_set_keyframe )
//	.br
//		We would use
//	.nf
// %@  AlContinuousFunction::create( NULL, // init
// %@%@ down_set_keyframe,  // reads the keyboard input
// %@%@ NULL,
// %@%@ up_set_keyframe,
// %@%@ NULL // cleanup )
// %@ AlContinuousFunction::setPrompt( "Enter frame: %f", frame_buff, kFilterNone )
//		Then call
// %@ AlContinousFunction::setBehaviour( kBehaviourMomentary );
//	.fi
//
//	.nf
//	Pre-init and post-cleanup callbacks:
//	.fi
//		There are also settable 'pre-init' and 'post-cleanup' routines.  The
//		pre-init is called before the init  but it is only called on the
//		initial selection of the function and not when you 'bounce' back to
//		the function after selecting a function (e.g. another momentary style
//		function).
//	.br
//		Note: Due to the original design of the UI, there is no way to pass a
//		user data pointer to the five functions without using globals.
//
*/

#ifndef _AlFunction
#define _AlFunction

#include <AlStyle.h>

extern "C" {

typedef void AlMouseButtonFunction( int input, Screencoord x, Screencoord y );
typedef void (*AlUndoCallbackType)(void);

#define kModifierControl    0001
#define kModifierShift      0002
#define kModifierAlt        0004
#define kModifierButton1    0010
#define kModifierButton2    0020
#define kModifierButton3    0040
#define kModifierButton4    0100
#define kModifierButton5    0200

/*
// Short cuts for the common ones
*/
#define kButton1    kModifierButton1
#define kButton2    kModifierButton2
#define kButton3    kModifierButton3

typedef enum
	{ kInputInvalid, kInputAbort, kInputKeyboard, kInputButton, kInputOther }
AlInputType;

typedef enum
	{ kCoordinateInvalid, kCoordinateAbsolute, kCoordinateRelative }
AlCoordinateType;

typedef enum
	{ kBehaviourInvalid, kBehaviourContinuous, kBehaviourMomentary }
AlBehaviourType;

typedef enum
	{ kFilterNone, kFilterLinear, kFilterAngular }
AlFilterType;

}

class AlMomentaryFunction;
class AlContinuousFunction;

class FuncLink;


class STUDIOAPI_DECL AlFunction
{
	friend class AlFunctionHandle;
	friend class AlFunctionExt;

public:
	virtual ~AlFunction();
	statusCode deleteObject();

	virtual AlMomentaryFunction*	asMomentaryFunctionPtr();
	virtual AlContinuousFunction*	asContinuousFunctionPtr();

	const char *name();

protected:
	AlFunction();
	void *my_pointer;
};

//class AlMomentaryFunction 
class STUDIOAPI_DECL AlMomentaryFunction : public AlFunction
{
public:
	virtual AlMomentaryFunction*	asMomentaryFunctionPtr();

	statusCode			create( const char *command, void (*action)( void ) );
	statusCode			create( void (*action)(void) );
    statusCode			create( const char *command, void (*action)( void ), void** ); 
};

//class AlContinuousFunction 
class STUDIOAPI_DECL AlContinuousFunction : public AlFunction
{
public:
	virtual AlContinuousFunction*	asContinuousFunctionPtr();

	statusCode				create( void (*init)( void ),
									AlMouseButtonFunction *down,
									AlMouseButtonFunction *move,
									AlMouseButtonFunction *up,
									void (*cleanup)( void ),
									API_BOOLEAN manipulatesPickList = FALSE );

	statusCode				create( const char *,
									void (*init)( void ),
									AlMouseButtonFunction *down,
									AlMouseButtonFunction *move,
									AlMouseButtonFunction *up,
									void (*cleanup)( void ),
									API_BOOLEAN manipulatesPickList = FALSE );

	statusCode				setPreInitFunction( void (*preInit)() );
	statusCode				setPostCleanupFunction( void (*postCleanup)() );

	statusCode				setPrompt( const char *staticPrompt, char *inputBuffer, AlFilterType );
	statusCode				setPrompt( const char *(*outputStringFunc)(), char *inputBuffer, AlFilterType );

	statusCode				setBehaviour( AlBehaviourType type );
	statusCode				setMouseCoordinateType( AlCoordinateType type);
	AlBehaviourType			behaviour() const;

	static statusCode		createGoButton( void (*pressed)( void ) );
	static statusCode		createGoButton( int (*pressed)( void ), API_BOOLEAN persistence, 
												const char *button1, 
												const char *button2 = NULL, 
												const char *button3 = NULL,
												const char *button4 = NULL,
												const char *button5 = NULL );
	static statusCode		clearGoButton( API_BOOLEAN do_redraw );
	static const char *		goButtonPressed( void );
	
	static statusCode				setUndoFunction( AlUndoCallbackType );
	static AlUndoCallbackType		undoFunction( void );
	
	API_BOOLEAN 					isActiveContinuousFunction();

	static void initPickBox( Screencoord, Screencoord );
	static void movePickBox( Screencoord, Screencoord );
	static void endPickBox( Screencoord, Screencoord );

public:
	static AlCoordinateType	keyboardCoordinateMode();
	static AlInputType		translateInput( int event, int &button );
	static int				inputModifierMask();	// see kModifier_xxxx

	statusCode				finished();
};

#endif	/* _AlFunction */
