/********************************************************************
 * (C) Copyright 2010 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlEvaluate - Support for Evaluation of Surfaces and Curves
//
//	.SECTION Description
//		AlEvaluate provides the programmer with the ability
//		to evaluate positional and tangent continuity of surfaces and curves.
//
//    In order to evaluate a surface/curve :
//
//      1) Declare an AlEvaluate() object.
//
//      2) Set the evaluation parameters:
//           a) By default, all geometry in the scene will be evaluated.
//              However, if you wish to only evaluate active geometry you 
//              may call wantActiveGeomOnly(true).  If you wish to
//              evaluate visible geometry only you may call 
//              wantVisibleGeomOnly(true).
//           b) By default both curves and surfaces will be evaluated but
//              you may turn either off using wantSurfaces(false) or 
//              wantCurves(false).
//           c) By detault only errors are reported.  If you wish to report
//              all the check results call wantErrorsOnly(false).
//           d) By default G0, G1 and G2 checking is off.  You must enable
//              the check(s) you want before running the evaluation.
//
//      3) Call the runEvaluation() routine to evaluate the curves/surfaces.
//         Returns :
//		         sSuccess - The evaluation succeeded.
//		         sFailure - Means that errors were encountered during the check.
//
//
//    Once the evaluation is complete, you may :
//
//      1) Call examineResults() to see if the check produced any results. It
//         will return :
//		        sInvalidObject - Means that either the check has not been run or
//                             errors were encountered during the check.
//		        sSuccess - Means the check completed with no results (no errors).
//      		sFailure - Means the check produced results, which can either mean :
//                          a) If wantErrorsOnly() was set to true, then errors 
//                             were found.
//                          b) If wantErrosOnly() was false, then either results 
//                             or errors were found
//
//      2) writeResultsToFile(fileName) - Will write the results of the check to
//                                        the specified text file.
//
//      3) displayResults() - Will open the ModelCheck Results window.  This call
//                            works in OpenAlias only (interactive).

#ifndef _AlEvaluate
#define _AlEvaluate

#include <AlObject.h>

class STUDIOAPI_DECL AlEvaluate  : public AlObject
{
    friend class			AlFriend;

	public:
        AlEvaluate();
	     virtual					~AlEvaluate();
	    
        virtual statusCode		deleteObject();
	     virtual AlObject*		copyWrapper() const;
	     virtual AlObjectType	type() const;

        // Set the evaluation parameters
        statusCode  wantVisibleGeomOnly(bool flag);
        statusCode  wantActiveGeomOnly(bool flag);
        statusCode  wantSurfaces(bool flag);
        statusCode  wantCurves(bool flag);
        // gapTolerance is specified in millimeters
        statusCode  wantG0Check(bool flag, double gapTolerance = 0.0);
        statusCode  wantG1Check(bool flag, double continuityAngle = 0.0);
        statusCode  wantG2Check(bool flag, double continuityCurvature = 0.0);
        statusCode  wantErrorsOnly(bool flag);
        statusCode  wantLocatorCleanUp(bool flag);

        // Perform the evaluation
        statusCode  runEvaluation();

        // Results
        statusCode  examineResults();
        statusCode  writeResultsToFile(char* fileName);
        void        displayResults();         // OpenAlias Only
    
    private:
        void                  resetState();

        static void				initMessages();
	    static void				finiMessages();

        bool m_wantG0;
        bool m_wantG1;
        bool m_wantG2;
        bool m_wantSurfaces;
        bool m_wantCurves;
        bool m_wantActiveOnly;
        bool m_wantVisibleOnly;
        bool m_wantErrorsOnly;
        bool m_wantLocatorCleanup;

        double m_gapTolerance;
        double m_continuityAngle;
        double m_continuityCurvature;
};

#endif //_AlEvaluate
