/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//	.NAME AlEditor - interface to Studio UI editing boxes.
//
//	.SECTION Description
//		This class allows the building of dynamic editors within
//		a plug-in.  Standard operations such as open, close and toggle
//		along with a basic set of user interface components are
//		supported.
//		For a few of the user interface components, a reference variable
//		is used as a parameter.  This reference maintains the state
//		of the component and must be permanent storage.  If you allocate
//		this variable on the stack, program errors will occur.
//		In some cases, a user interface component will contain 
//		multiple parts.  In this case, a list of ComponentDescriptors 
//		will be used to describe this object to the class.
//		Descriptors are used to describe the component to be created.
//		Currently a label is the usual input that is set in the
//		descriptor object.  The AlEditor class will dynamically allocate
//		a component id and set it in the descriptor.
//		Individual components of an editor cannot be deleted.  Instead,
//		the entire editor must be de-allocated all at once.
//
//
//

#ifndef _AlEditor
#define  _AlEditor

#include <AlStyle.h>

typedef int ComponentId;
typedef void (*IdCallback)( ComponentId itemId );
typedef void (*IdStringCallback)( ComponentId itemId, const char *string );

// Input that is passed into the AlEditor 
// class.  The AlEditor class will dynamically
// allocate an id and set it within this class.

typedef enum
{
	kGeneric
} ComponentDescriptorType;

class STUDIOAPI_DECL ComponentDescriptor
{
public:
	ComponentDescriptor( const char *label );
	~ComponentDescriptor( void );
	
	ComponentId id( void );
	const char *label( void );
	void setLabel( const char *label );
	void setId( ComponentId id );
	
	ComponentDescriptor *next( void );
	void setNext( ComponentDescriptor *descriptor );
	
	ComponentDescriptorType type( void );
	
private:
	ComponentId fComponentId;
	const char *fLabel;
	ComponentDescriptor *fNext;
	ComponentDescriptorType fType;
};

// Required data for the integer component
class STUDIOAPI_DECL IntComponentData
{
public:
	IntComponentData( int minValue, int maxValue );
	~IntComponentData( void );
	
	int minValue( void );
	int maxValue( void );
	
private:
	int fMinValue;
	int fMaxValue;
};

// Required data for the float component
class STUDIOAPI_DECL FloatComponentData
{
public:
	FloatComponentData( float minValue, float maxValue );
	~FloatComponentData( void );

	float minValue( void );
	float maxValue( void );
	
private:
	float fMinValue;
	float fMaxValue;
};

// 
// Radio group styles
//

typedef enum
{
	kDefault,
	kPushButton
} AlEditorRadioGroupStyle;

//
// Class for building dynamic editors in Studio
//
class STUDIOAPI_DECL  AlEditor
{
public:

	AlEditor( void );
	~AlEditor( void );
	
	statusCode create( const char *title );
	statusCode deleteEditor( void );
	
	statusCode open( void );
	statusCode close( void );
	statusCode update( void );
	statusCode toggle( void );
	statusCode enableItem( ComponentId itemId, API_BOOLEAN enable );
	
	API_BOOLEAN	isEnabled( ComponentId itemId );
	API_BOOLEAN isOpen( void );
	
	statusCode addSeparator( ComponentDescriptor *descriptor );
	statusCode addGrouping( ComponentDescriptor *descriptor, API_BOOLEAN opened = TRUE );

	statusCode addString( ComponentDescriptor *descriptor, const char *value, IdStringCallback callback );
	statusCode addInt( ComponentDescriptor *descriptor, int &reference, IntComponentData *data, IdCallback callback = NULL );
	statusCode addFloat( ComponentDescriptor *descriptor, float &reference, FloatComponentData *data, IdCallback callback = NULL );
	statusCode add3Floats( ComponentDescriptor *descriptor, float (&reference)[3], IdCallback callback = NULL );
	
	statusCode addCheckBox( ComponentDescriptor *descriptor, int &reference, IdCallback callback = NULL );
	statusCode addCheckBoxes( ComponentDescriptor *descriptor, int &reference, ComponentDescriptor *descriptor2, int &reference2, IdCallback callback = NULL );
	
	statusCode addButton( ComponentDescriptor *descriptor, IdCallback callback );

	statusCode addRadioGroup( ComponentDescriptor *descriptor, AlEditorRadioGroupStyle style, ComponentDescriptor *subComponentDescriptorList, IdCallback callback );
	statusCode addPulldownMenu( ComponentDescriptor *descriptor, ComponentDescriptor *subComponentDescriptorList, IdCallback callback );

   statusCode getRadioGroupValue( ComponentId itemId, int* value );
   statusCode setRadioGroupValue( ComponentId itemId, int value );

private:
	void *editor;
	int fButtonCount;	
};

#endif // _AlEditor










